/**
 * 
 */
package be.erwinsmout;

import java.io.IOException;
import java.io.InputStream;
import java.util.Properties;
import java.util.Map.Entry;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * @author Erwin
 * 
 */
public abstract class MyProperties extends Properties {

	/**
	 * @author Erwin
	 * 
	 */
	public static class SKIPAUTOLOAD {

	}

	/**
	 * 
	 */
	private static final String PROPERTIES = "properties"; //$NON-NLS-1$

	/**
	 * 
	 */
	private static final long serialVersionUID = 4230868963188380737L;

	/**
	 * 
	 */
	public static final String SUFFIX = '.' + PROPERTIES;

	/**
	 * Loads the given properties object from an InputStream for reading a resource corresponding to the given properties (where "correspondence" is established through the properties' classname). Search will be conducted consecutively in the following places (in the example, the properties' class is com.foo.BarProperties) :
	 * <OL>
	 * <LI>For a resource named com.foo.Bar.properties on the classpath (=/com/foo/Bar.properties)</LI>
	 * <LI>For a file named com.foo.Bar.properties in the current user working dir</LI>
	 * <LI>For a file named com.foo.Bar.properties on the user's home dir</LI>
	 * </OL>
	 * 
	 * @param properties
	 *            The properties object to be loaded with the contents of the properties resource corresponding to the given properties.
	 */
	private static void load (Properties properties) {
		Class<? extends Properties> propertiesClass = properties.getClass();
		String propertiesResourceName = getPropertiesBaseName(propertiesClass); // returns com.foo.Bar.properties

		try {
			InputStream propertiesStream = MyResource.getResourceInputStream(propertiesClass, propertiesResourceName);
			try {
				properties.load(propertiesStream);
			} catch (IOException e) {
				Logger.getLogger(properties.getClass().getName()).log(Level.SEVERE, MyMessageFormat.format("IO Exception loading properties for {0}", new String[] { propertiesResourceName }), e); //$NON-NLS-1$
			} finally {
				try {
					propertiesStream.close();
				} catch (IOException e) {
					Logger.getLogger(properties.getClass().getName()).log(Level.SEVERE, MyMessageFormat.format("IO Exception closing properties stream for {0}", new String[] { propertiesResourceName }), e); //$NON-NLS-1$
				}
			}
		} catch (NotFoundException e1) {
			Logger.getLogger(properties.getClass().getName()).info(MyMessageFormat.format("No resource found to load {0} from.", new String[] { properties.getClass().getName() })); //$NON-NLS-1$
		}
	}

	/**
	 * Gets the base name for the properties resource corresponding to the given class. This is as follows (first rule that applies) :
	 * <OL>
	 * <LI>full classname does not end with 'PROPERTIES' (case insensitive) ===> full classname, suffixed with '.properties' (e.g. "com.foo.Bar.properties")</LI>
	 * <LI>full classname ends with '.PROPERTIES', or is equal to 'PROPERTIES' (both case insensitive) ===> full classname (e.g. "com.foo.properties", or just "properties")</LI>
	 * <LI>full classname ends with 'PROPERTIES', not preceded by a dot (i.e. X.Y.ZPROPERTIES) ===> X.Y.Z, suffixed with '.properties' (e.g. "com.foo.Bar.properties")</LI>
	 * </OL>
	 * 
	 * @param propertiesClass
	 *            the class for which the corresponding properties resource name is to be returned
	 * @return the base name for the properties resource corresponding to the given class.
	 */
	protected static String getPropertiesBaseName (Class<? extends Properties> propertiesClass) {
		return MyResource.getResourceBaseName(propertiesClass, PROPERTIES);
	}

	/**
	 * Resolves any references to System Properties in a String value by replacing the reference with the actual System Property value. System Property references appear as a ${} sequence with the property name in between. They cannot be nested, i.e. the name of the property cannot be itself the result of resolving a nested property reference.
	 * 
	 * @param propertyValue
	 *            The String value to be resolved for System Property references
	 * @return The resolved String value
	 */
	public static String resolveSystemPropertyRefs (String propertyValue) {
		int i1;
		int i2;
		String resolvedPropertyValue = propertyValue;
		do {
			i1 = resolvedPropertyValue.indexOf("${"); //$NON-NLS-1$
			i2 = resolvedPropertyValue.indexOf('}', i1);
			if (i1 >= 0 && i2 > i1) {
				resolvedPropertyValue = resolvedPropertyValue.substring(0, i1) + System.getProperty(resolvedPropertyValue.substring(i1 + 2, i2), "") + resolvedPropertyValue.substring(i2 + 1); //$NON-NLS-1$
			}
		} while (i1 >= 0 && i2 > i1);

		return resolvedPropertyValue;
	}

	/**
	 * 
	 */
	public MyProperties ( ) {
		this(false);
	}

	/**
	 * Creates the Properties object, and tries to load it from one of the following places:
	 * <ol>
	 * <li>A resource available to this' classloader</li>
	 * <li>A file in the directory pointed to by the user.dir system property</li>
	 * <li>A file in the directory pointed to by the user.home system property</li>
	 * </ol>
	 * 
	 * @param resolveSystemPropertyReferences
	 *            flag to indicate whether System property references (e.g.${user.home}) are to be resolved when loading the properties
	 * 
	 */
	public MyProperties (boolean resolveSystemPropertyReferences) {
		load(this);
		if (resolveSystemPropertyReferences) {
			resolveSystemPropertyRefs();
		}
	}

	/**
	 * Creates the MyProperties
	 * 
	 * @param defaults
	 */
	public MyProperties (Properties defaults) {
		this(defaults, false);
	}

	/**
	 * Creates the MyProperties
	 * 
	 * @param defaults
	 * @param resolveSystemPropertyReferences
	 *            flag to indicate whether System property references (e.g.${user.home}) are to be resolved when loading the properties
	 */
	public MyProperties (Properties defaults, boolean resolveSystemPropertyReferences) {
		super(defaults);
		load(this);
		if (resolveSystemPropertyReferences) {
			resolveSystemPropertyRefs();
		}
	}

	/**
	 * Constructor that skips the automatic loading process.
	 * 
	 * @param skipAutoLoad
	 *            not used, except to distinguish this constructor from others
	 */
	public MyProperties (SKIPAUTOLOAD skipAutoLoad) {
		skipAutoLoad.getClass();
	}

	/**
	 * 
	 */
	private void resolveSystemPropertyRefs ( ) {
		for (Entry<Object, Object> me : entrySet()) {
			setProperty((String) me.getKey(), resolveSystemPropertyRefs((String) me.getValue()));
		}
	}
}
