/**
 * 
 */
package be.erwinsmout;

import java.io.BufferedInputStream;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.InputStream;

/**
 * @author Erwin
 */
public abstract class MyResource {

	/**
	 * Gets the base name for the resource of the indicated type corresponding to the given base name. This is as follows (first rule that applies) :
	 * <OL>
	 * <LI>resourceName does not end with '&lt;resourceTypeName&gt;' (case insensitive) ===> resourceName, suffixed with '.&lt;resourceTypeName&gt;'</LI>
	 * <LI>resourceName ends with '.&lt;resourceTypeName&gt;', or is equal to '&lt;resourceTypeName&gt;' (both case insensitive) ===> resourceName</LI>
	 * <LI>resourceName ends with '&lt;resourceTypeName&gt;', not preceded by a dot (i.e. X.Y.Z&lt;resourceTypeName&gt;) ===> X.Y.Z, with either of dots and forward slashes as separators, suffixed with '.&lt;resourceTypeName&gt;' (e.g. "com.foo.Bar.properties" and "com/foo/Bar.properties")</LI>
	 * </OL>
	 * 
	 * @param resourceTypeName
	 * @param resourceName
	 * @return
	 */
	private static String getResourceBaseName (String resourceTypeName, String resourceName) {
		String lowerCaseResourceTypeName = resourceTypeName.toLowerCase();
		String dotSuffix = '.' + lowerCaseResourceTypeName;

		if (resourceName.toLowerCase().endsWith(lowerCaseResourceTypeName)) {
			if (resourceName.length() > resourceTypeName.length() && !(resourceName.charAt(resourceName.length() - dotSuffix.length()) == '.')) {
				return resourceName.substring(0, resourceName.length() - resourceTypeName.length()) + dotSuffix;
			} else {
				// classname is "properties" or is already of the form "xxx.properties"
				return resourceName;
			}
		} else {
			return resourceName + dotSuffix;
		}
	}

	/**
	 * @param resourceName
	 * @return
	 * @throws NotFoundException
	 */
	private static InputStream tryInUserHomeDirectory (String resourceName) throws NotFoundException {
		try {
			return new BufferedInputStream(new FileInputStream(new File(System.getProperty("user.home") + File.separatorChar + resourceName))); //$NON-NLS-1$
		} catch (FileNotFoundException e2) {
			throw new NotFoundException();
		} catch (SecurityException e2) {
			throw new NotFoundException();
		}
	}

	/**
	 * Gets the base name for the resource of the indicated type corresponding to the package of the given class. This is as follows (first rule that applies) :
	 * <OL>
	 * <LI>full packagename does not end with '&lt;resourceTypeName&gt;' (case insensitive) ===> full packagename, with either of dots and forward slashes as separators, suffixed with '.&lt;resourceTypeName&gt;' (e.g. "com.foo.Bar.properties" and "com/foo/Bar.properties")</LI>
	 * <LI>full packagename ends with '.&lt;resourceTypeName&gt;', or is equal to '&lt;resourceTypeName&gt;' (both case insensitive) ===> full packagename, with either of dots and forward slashes as separators (e.g. "com.foo.Bar.properties" and "com/foo/Bar.properties", or just "properties")</LI>
	 * <LI>full packagename ends with '&lt;resourceTypeName&gt;', not preceded by a dot (i.e. X.Y.Z&lt;resourceTypeName&gt;) ===> X.Y.Z, with either of dots and forward slashes as separators, suffixed with '.&lt;resourceTypeName&gt;' (e.g. "com.foo.Bar.properties" and "com/foo/Bar.properties")</LI>
	 * </OL>
	 * 
	 * @param clazz
	 *            the class for which the corresponding properties resource name is to be returned
	 * @param resourceTypeName
	 *            indication of the type of resource, e.g. 'properties' or 'keystore'
	 * @return the base name for the resource of the indicated type corresponding to the given class.
	 * 
	 */
	public static String getPackageResourceBaseName (Class<? extends Object> clazz, String resourceTypeName) {
		return getResourceBaseName(resourceTypeName, clazz.getPackage().getName());
	}

	/**
	 * Gets the base name for the resource of the indicated type corresponding to the given class. This is as follows (first rule that applies) :
	 * <OL>
	 * <LI>full classname does not end with '&lt;resourceTypeName&gt;' (case insensitive) ===> full classname, suffixed with '.&lt;resourceTypeName&gt;' (e.g. "com.foo.Bar.properties" and "com/foo/Bar.properties")</LI>
	 * <LI>full classname ends with '.&lt;resourceTypeName&gt;', or is equal to '&lt;resourceTypeName&gt;' (both case insensitive) ===> full classname, with either of dots and forward slashes as separators (e.g. "com.foo.Bar.properties" and "com/foo/Bar.properties", or just "properties")</LI>
	 * <LI>full classname ends with '&lt;resourceTypeName&gt;', not preceded by a dot (i.e. X.Y.Z&lt;resourceTypeName&gt;) ===> X.Y.Z, with either of dots and forward slashes as separators, suffixed with '.&lt;resourceTypeName&gt;' (e.g. "com.foo.Bar.properties" and "com/foo/Bar.properties")</LI>
	 * </OL>
	 * 
	 * @param clazz
	 *            the class for which the corresponding properties resource name is to be returned
	 * @param resourceTypeName
	 *            indication of the type of resource, e.g. 'properties' or 'keystore'
	 * @return the base name for the resource of the indicated type corresponding to the given class.
	 * 
	 */
	public static String getResourceBaseName (Class<? extends Object> clazz, String resourceTypeName) {
		return getResourceBaseName(resourceTypeName, clazz.getName());
	}

	/**
	 * Attempts to find a resource of the given name, and returns an input stream for reading that resource if it can be found. The search path for finding the resource is as follows :
	 * <ol>
	 * <li>The classpath that is available to the ClassLoader of the clazz class. That path is searched using the given class' getResourceAsStream() method, using the modified resourcename. The 'modified resourcename' is different from the resourcename itself only if the resourcename starts with the same tokens as the clazz' package name (in which case the dot-separated package name prefix in the resourcename is replaced with a forward-slash-separated '/' name, and a forward slash is prepended. Following cases are thus possible (clazz is com.foo.Bar in each case):
	 * <ul>
	 * <li>resourcename com.foo.Bar - classpath search is for /com/foo/Bar</li>
	 * <li>resourcename com.foo.Bar.properties - classpath search is for /com/foo/Bar.properties</li>
	 * <li>resourcename net.foo.Bar.properties - classpath search is for net.foo.Bar.properties (implying that java will conduct a search for /com/foo/net.foo.Bar.properties)</li>
	 * <li>resourcename Bar.properties - classpath search is for Bar.properties (implying that java will conduct a search for /com/foo/Bar.properties)</li>
	 * </ul>
	 * </li>
	 * <li>A file of the same name as the requested resource in the directory pointed to by the user.dir property.</li>
	 * <li>A file of the same name as the requested resource in the directory pointed to by the user.home property.</li>
	 * </ol>
	 * 
	 * @param clazz
	 *            The class object representing the class of the object for which the resource is to be located
	 * @param resourceName
	 *            The name of the resource to be searched
	 * @return An inputstream for reading the requested resource
	 * @throws NotFoundException
	 *             If no resource of the indicated name could be found
	 */
	public static InputStream getResourceInputStream (Class<?> clazz, String resourceName) throws NotFoundException {

		String modifiedResourceName;
		final String clazzPackageName = clazz.getPackage().getName();
		if (resourceName.startsWith(clazzPackageName)) {
			modifiedResourceName = '/' + clazzPackageName.replace('.', '/') + '/' + resourceName.substring(clazzPackageName.length() + 1);
		} else {
			modifiedResourceName = resourceName;
		}

		InputStream inputStream = clazz.getResourceAsStream(modifiedResourceName);

		if (inputStream == null) {
			inputStream = getResourceInputStream(resourceName);
//			System.out.println("  ClassLoader failed to load resource " + modifiedResourceName);
		}

		return inputStream;
	}

	/**
	 * Attempts to find a resource of the given name, and returns an input stream for reading that resource if it can be found. The search path for finding the resource is as follows :
	 * <ol>
	 * <li>A file of the same name as the requested resource in the directory pointed to by the user.dir property.</li>
	 * <li>A file of the same name as the requested resource in the directory pointed to by the user.home property.</li>
	 * </ol>
	 * 
	 * @param resourceName
	 *            The name of the resource to be searched
	 * @return An inputstream for reading the requested resource
	 * @throws NotFoundException
	 *             If no resource of the indicated name could be found
	 */
	public static InputStream getResourceInputStream (String resourceName) throws NotFoundException {
		InputStream inputStream = null;
		if (resourceName.charAt(0) != '/') {
			File resourceFile = new File(System.getProperty("user.dir") + File.separatorChar + resourceName); //$NON-NLS-1$
			try {
				inputStream = new BufferedInputStream(new FileInputStream(resourceFile));
			} catch (FileNotFoundException e) {
				inputStream = tryInUserHomeDirectory(resourceName);
			} catch (SecurityException e) {
				inputStream = tryInUserHomeDirectory(resourceName);
			}
		}

		if (inputStream == null) {
			//				Logger.getLogger(MyResource.class.getName()).info(MyMessageFormat.format("No resource found for {0}", new String[] { resourceName })); //$NON-NLS-1$
			throw new NotFoundException();
		}
		return inputStream;
	}

	/**
	 * Prevent instantiation
	 * 
	 */
	private MyResource ( ) {

	}
}
