package be.WAAR.PresentationLayer;

import java.io.*;
import java.util.*;
import java.util.Map.Entry;

/**
 * The Architecture singleton is instantiated when the first Architecture Servlet is initialized. That servlet maintains a reference to the Architecture object, thus ensuring that the Architecture object cannot be finalized as long as there is any Architecture Servlet that is still accessible to the JVM. The Architecture object maintains references to several pools, such as the user session pool. Doing so ensures us that all the resources needed by the WAAR architecture will exist and be consistent as long as any Architecture Servlet is kept active by the Application server.
 * 
 * @author Erwin
 */
public class Architecture {

	/**
	 * The AttributeTypePool is the map holding all the Presentation Type objects (mapped to their name).
	 * 
	 * @author Erwin
	 */

	public static class AttributeTypePool {

		/**
		 * The instance of the pool
		 */
		static AttributeTypePool attributeTypePoolInstance = new AttributeTypePool();

		/**
		 * The pool of Types for presentation attributes
		 */
		private HashMap<String, PresentationType> pool = new HashMap<String, PresentationType>();

		/**
		 * Creates the pool
		 */
		private AttributeTypePool ( ) {

		}

		/**
		 * Attempts to instantiate an object of a PresentationAttributeType class
		 * 
		 * @param typeName
		 *            The (fully qualified) name of a PresentationAttributeType that is to be instantiated
		 * @param presentationFunctionClass
		 *            The Class object of the presentation function for which the type is needed. This object acts as a handle to its classLoader, which might be needed to gain access to package-defined presentation types (such as e.g. enumeration types)
		 * @return An object of the Classname that is constructed by concatenating the given package name and type Name
		 * @throws NotAPresentationAttributeTypeException
		 */
		private PresentationType loadPresentationTypeFromPackage (String typeName, Class<? extends OnLinePresentationFunction> presentationFunctionClass) throws NotAPresentationAttributeTypeException {
			Object object;
			try {
				//				object = Class.forName(packageName + "." + typeName).newInstance(); //$NON-NLS-1$
				//				object = getClass().getClassLoader().loadClass(packageName + "." + typeName).newInstance(); //$NON-NLS-1$
				object = presentationFunctionClass.getClassLoader().loadClass(typeName).newInstance();
			} catch (InstantiationException e) {
				throw new NotAPresentationAttributeTypeException(e);
			} catch (IllegalAccessException e) {
				throw new NotAPresentationAttributeTypeException(e);
			} catch (ClassNotFoundException e) {
				throw new NotAPresentationAttributeTypeException(e);
			}
			if (object instanceof PresentationType) {
				return (PresentationType) object;
			} else {
				throw new NotAPresentationAttributeTypeException();
			}
		}

		/**
		 * Gets the PresentationType object associated with a name. The method first looks in the pool for a Presentation type of the given name. If no such type is present in the pool, then an attempt is made to instantiate a Class of the given typename, prefixed with the java package name of the Architecture. If that fails, an attempt is made to instantiate a Class of the given typename, prefixed with the given application package name. The Class must be a subClass of PresentationAttributeType.
		 * 
		 * @param typeName
		 *            The name of the attribute Type to be obtained
		 * @param presentationFunctionClass
		 *            The Class object of the presentation function for which the type is needed. This object acts as a handle to its classLoader, which might be needed to gain access to package-defined presentation types (such as e.g. enumeration types)
		 * @param locale
		 *            The locale in which to set an exception message text if needed
		 * @return The requested AttributeType object
		 * @throws FailedToCreatePresentationTypeException
		 *             if the Type object for the given name cannot be instantiated
		 */
		PresentationType get (String typeName, Class<? extends OnLinePresentationFunction> presentationFunctionClass, Locale locale) throws FailedToCreatePresentationTypeException {
			synchronized (pool) {
				PresentationType presentationType;
				if (pool.containsKey(typeName)) {
					presentationType = pool.get(typeName);
				} else {
					try {
						presentationType = loadPresentationTypeFromPackage(typeName, presentationFunctionClass);
					} catch (NotAPresentationAttributeTypeException e) {
						try {
							presentationType = loadPresentationTypeFromPackage(typeName, presentationFunctionClass);
						} catch (NotAPresentationAttributeTypeException e1) {
							e1.printStackTrace(System.err);
							throw new FailedToCreatePresentationTypeException(e1, typeName, locale);
						}
					}
					pool.put(typeName, presentationType);
				}
				return presentationType;
			}
		}

		/**
		 * Gets the pool
		 * 
		 * @return the pool
		 */
		public HashMap<String, PresentationType> getPool ( ) {
			return pool;
		}

		/**
		 * Removes a Presentation Type from the pool
		 * 
		 * @param typeName
		 *            The name of the type whose entry is to be removed from the pool
		 */
		public void remove (String typeName) {
			synchronized (pool) {
				pool.remove(typeName);
			}
		}
	}

	/**
	 * The mapping table
	 */
	// private static Properties applicationPackageNames = new Properties();
	/**
	 * 
	 */
	private static final String ARCHITECTURE_HTML = "architecture.html"; //$NON-NLS-1$

	/**
	 * 
	 */
	private static final Locale architectureLocale = new Locale("EN"); //$NON-NLS-1$

	/**
	 * The pool of Presentation layer attributes
	 */
	private static AttributeTypePool attributeTypePool;

	/**
	 * The instance of the Architecture context
	 */
	private static Architecture instance = new Architecture();

	/**
	 * The Map of Presentation architecture fields
	 */
	private static PresentationArchitectureFieldMap presentationArchitectureFieldMap = PresentationArchitectureFieldMap.getInstance();

	/**
	 * The map of supported languages
	 */
	private static HashMap<String, String> supportedLanguages;

	/**
	 * 
	 */
	private static final String USERDATABASEFILENAME = "udb"; //$NON-NLS-1$

	/**
	 * The actual map holding all the currently active user sessions
	 */
	private static HashMap<Long, UserSession> userSessions = new HashMap<Long, UserSession>();

	/**
	 * The map holding the registered WAAR users and properties
	 */
	private static HashMap<String, UserProperties> waarUsers;

	/**
	 * The common html for all architecture output
	 */
	static String architectureHtml;

	/**
	 * The home directory of the WAAR application
	 */
	static String waarHomeDirectory;

	/**
	 * 
	 */
	private static void writeUDBFile ( ) {
		File userDatabaseFile = new File(waarHomeDirectory + File.separatorChar + USERDATABASEFILENAME);
		try {
			DataOutputStream userDatabaseOutputStream = new DataOutputStream(new UDBOutputStream(userDatabaseFile));

			try {
				Iterator<Entry<String, UserProperties>> i_waarUsers = waarUsers.entrySet().iterator();
				while (i_waarUsers.hasNext()) {
					Entry<String, UserProperties> me = i_waarUsers.next();
					userDatabaseOutputStream.writeBytes(me.getKey() + '=' + me.getValue().toString() + "\r\n"); //$NON-NLS-1$
				}
			} catch (IOException e) {

			} finally {
				try {
					userDatabaseOutputStream.close();
				} catch (IOException e2) {

				}
			}
		} catch (FileNotFoundException e) {

		}
	}

	/**
	 * @param userID
	 * @param userPassword
	 * @param newPassword
	 * @param confirmNewPassword
	 * @throws InvalidPasswordException
	 * @throws UnknownUserIDException
	 * @throws InvalidNewPasswordException
	 * @throws PasswordUnconfirmedException
	 * @throws MaximumInvalidPasswordsForUserExceededException
	 * @throws UserIDDisabledException
	 */
	static void checkPassword (String userID, String userPassword, String newPassword, String confirmNewPassword) throws InvalidPasswordException, UnknownUserIDException, InvalidNewPasswordException, PasswordUnconfirmedException, MaximumInvalidPasswordsForUserExceededException, UserIDDisabledException {
		synchronized (waarUsers) {
			if (waarUsers.containsKey(userID)) {
				UserProperties userProperties = waarUsers.get(userID);
				if (userProperties.isDisabled()) {
					throw new UserIDDisabledException(userID, getDefaultLocale());
				}
				if (userProperties.getPassword().equals(userPassword)) {
					if (userProperties.isPasswordToBeChangedAtNextLogon()) {
						if (newPassword.length() < 8 || newPassword.equals(userPassword)) {
							throw new InvalidNewPasswordException(getDefaultLocale());
						}
						if (newPassword.equals(confirmNewPassword)) {
							waarUsers.put(userID, new UserProperties(newPassword + "#@#false#@#" + userProperties.getSuspiciousLogonAttemptTimes() + "#@#" + userProperties.getLanguage())); //$NON-NLS-1$ //$NON-NLS-2$
							writeUDBFile();
						} else {
							throw new PasswordUnconfirmedException(getDefaultLocale());
						}
					}
				} else {
					try {
						userProperties.registerSuspiciousLogonAttempt();
						throw new InvalidPasswordException(getDefaultLocale());
					} catch (MaximumInvalidPasswordsExceededException e) {
						throw new MaximumInvalidPasswordsForUserExceededException(userID, getDefaultLocale());
					} finally {
						writeUDBFile();
					}
				}
			} else {
				throw new UnknownUserIDException(userID, getDefaultLocale());
			}
		}
	}

	/**
	 * Gets the String object indicated by the given String
	 * 
	 * @param userLanguage
	 *            a valid 2-char ISO language code
	 * @return The String corresponding to the given language code
	 * @throws InvalidAttributeValueException
	 */
	static String checkUserLanguage (String userLanguage) throws InvalidAttributeValueException {
		String userLanguage_uc = userLanguage.toUpperCase();
		if (supportedLanguages.containsKey(userLanguage_uc)) {
			return userLanguage_uc;
		}
		throw new InvalidAttributeValueException(userLanguage, "Is not a supported language", getDefaultLocale()); //$NON-NLS-1$
	}

	/**
	 * Removes a session from the pool
	 * 
	 * @param sessionID
	 *            The identification number of the session to be removed
	 */
	static void dropUserSession (long sessionID) {
		synchronized (userSessions) {
			userSessions.remove(new Long(sessionID));
		}
	}

	/**
	 * Obtains the value for some field of the Presentation Architecture, as it applies to some given user function.
	 * 
	 * @param fieldName
	 *            The name of the Presentation Architecture field whose value is to be obtained
	 * @param userFunction
	 *            The user Function for which the field value is requested. (This is relevant for fields such as the function's application name, the system date (needs formatting info that resides in the function's user data).)
	 * @return The value of the architecture field in string format
	 * @throws NotAnArchitectureFieldException
	 *             if the given fieldName is not a known fieldName of the Presentation architecture
	 */
	static String getArchitectureFieldValue (String fieldName, OnLinePresentationFunction userFunction) throws NotAnArchitectureFieldException {
		if (presentationArchitectureFieldMap.containsKey(fieldName)) {
			ArchitectureValueGetter valueGetter = presentationArchitectureFieldMap.get(fieldName);
			return valueGetter.getValue(userFunction);
		} else {
			throw new NotAnArchitectureFieldException(fieldName, userFunction.getUserLocale());
		}
	}

	/**
	 * Gets the AttributeType object associated with a name.
	 * 
	 * @param typeName
	 *            The name of the attribute Type to be obtained
	 * @param presentationFunctionClass
	 *            The Class object of the presentation function for which the type is needed. This object acts as a handle to its classLoader, which might be needed to gain access to package-defined presentation types (such as e.g. enumeration types)
	 * @param locale
	 *            The locale in which to set an exception message text if needed
	 * @return The requested AttributeType object
	 * @throws FailedToCreatePresentationTypeException
	 */
	static PresentationType getAttributeType (String typeName, Class<? extends OnLinePresentationFunction> presentationFunctionClass, Locale locale) throws FailedToCreatePresentationTypeException {
		return attributeTypePool.get(typeName, presentationFunctionClass, locale);
	}

	/**
	 * Gets the AttributeTypePool
	 * 
	 * @return the AttributeTypePool
	 */
	public static AttributeTypePool getAttributeTypePool ( ) {
		return attributeTypePool;
	}

	/**
	 * Gets The default language of the installation
	 * 
	 * @return The default language of the installation
	 * @deprecated - replace with getDefaultLocale()
	 */
	static GenericStringValue getDefaultLanguage ( ) {
		return new GenericStringValue("EN"); //$NON-NLS-1$
	}

	/**
	 * Gets The instance of the Architecture
	 * 
	 * @return The instance of the Architecture
	 */
	static Architecture getInstance ( ) {
		return instance;
	}

	/**
	 * @param resourceName
	 * @param presentationFunctionClass
	 * @return
	 * @throws ResourceNotFoundException
	 */
	static InputStream getResourceInputStream (String resourceName, Class<? extends OnLinePresentationFunction> presentationFunctionClass) throws ResourceNotFoundException {
		return instance.resourceInputStream(resourceName, presentationFunctionClass);
	}

	/**
	 * @param resourceName
	 * @param presentationFunctionClass
	 * @return
	 * @throws ResourceNotFoundException
	 */
	static BufferedReader getResourceReader (String resourceName, Class<? extends OnLinePresentationFunction> presentationFunctionClass) throws ResourceNotFoundException {
		return instance.resourceReader(resourceName, presentationFunctionClass);
	}

	/**
	 * Gets the user session pool. Method should never be invoked. The Architecture just needs to hold a reference to this pool in order to prevent it from being finalized.
	 * 
	 * @param sessionID
	 *            The identification number of the user session to retrieve
	 * @return the requested user session, or null if such a session does not exist
	 */
	static UserSession getUserSession (long sessionID) {
		synchronized (userSessions) {
			return userSessions.get(new Long(sessionID));
		}
	}

	/**
	 * @return
	 */
	static Map<String, String> supportedLanguages ( ) {
		return supportedLanguages;
	}

	/**
	 * @param userID
	 * @param newLanguage
	 * @throws UnknownUserIDException
	 */
	public static void checkLanguage (String userID, String newLanguage) throws UnknownUserIDException {
		synchronized (waarUsers) {
			if (waarUsers.containsKey(userID)) {
				UserProperties userProperties = waarUsers.get(userID);
				if (supportedLanguages.containsKey(newLanguage)) {
					waarUsers.put(userID, new UserProperties(userProperties.getPassword() + "#@#" + userProperties.isPasswordToBeChangedAtNextLogon() + "#@#" + userProperties.getSuspiciousLogonAttemptTimes() + "#@#" + newLanguage)); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
					writeUDBFile();
				}
			} else {
				throw new UnknownUserIDException(userID, getDefaultLocale());
			}
		}
	}

	/**
	 * Gets The default locale of the installation
	 * 
	 * @return The default locale of the installation
	 */
	public static Locale getDefaultLocale ( ) {
		return architectureLocale;
	}

	/**
	 * Gets the UserProperties object holding the properties of the given user ID
	 * 
	 * @param userID
	 *            The user ID whose properties are to be obtained
	 * @param locale
	 * @return The UserProperties object holding the properties of the given user ID
	 * @throws UserIDUnknownException
	 */
	public static UserProperties getUserProperties (String userID, Locale locale) throws UserIDUnknownException {
		synchronized (waarUsers) {
			if (waarUsers.containsKey(userID.toLowerCase())) {
				return waarUsers.get(userID.toLowerCase());
			}
		}
		throw new UserIDUnknownException(userID, locale);
	}

	/**
	 * Gets The map holding the registered WAAR users and properties
	 * 
	 * @return The map holding the registered WAAR users and properties
	 */
	public static HashMap<String, UserProperties> getWaarUsers ( ) {
		return waarUsers;
	}

	/**
	 * @param userID
	 * @param password
	 * @param userLanguage
	 * @param locale
	 * @throws UserAlreadyExistsException
	 */
	public static void newWaarUser (String userID, String password, String userLanguage, Locale locale) throws UserAlreadyExistsException {
		String lowerCaseUserID = userID.toLowerCase();
		synchronized (waarUsers) {
			if (waarUsers.containsKey(lowerCaseUserID)) {
				throw new UserAlreadyExistsException(userID, locale);
			}

			waarUsers.put(lowerCaseUserID, new UserProperties(password + "#@#true#@#[]#@#" + userLanguage)); //$NON-NLS-1$
			writeUDBFile();
		}
	}

	/**
	 * @param userID
	 * @param locale
	 *            The locale in which to set the exception message texts
	 * @throws AdministratorCannotBeRemovedException
	 * @throws UserIDUnknownException
	 */
	public static void removeWaarUser (String userID, Locale locale) throws AdministratorCannotBeRemovedException, UserIDUnknownException {
		String lowerCaseUserID = userID.toLowerCase();
		synchronized (waarUsers) {
			if (!waarUsers.containsKey(lowerCaseUserID)) {
				throw new UserIDUnknownException(userID, locale);
			}
			if (lowerCaseUserID.equals("waaradministrator")) { //$NON-NLS-1$
				throw new AdministratorCannotBeRemovedException(locale);
			}

			waarUsers.remove(lowerCaseUserID);
			writeUDBFile();
		}
	}

	/**
	 * The last assigned session ID
	 */
	private long userSessionIDLastAssigned;

	/**
	 * The Architecture is instantiated at the time the first Architecture Servlet is initialized. It will not be destroyed as long as there are any Architecture Servlets "active" in the JVM in which the Application server is running.
	 */
	private Architecture ( ) {
		
		waarHomeDirectory = System.getProperty("user.home") + (File.separatorChar == '\\' ? (File.separatorChar + "Application Data") : "") + File.separatorChar + "WAAR"; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
		new File(waarHomeDirectory).mkdirs();

		attributeTypePool = AttributeTypePool.attributeTypePoolInstance;
		// applicationNameToPackageNameMapping = ApplicationNameToPackageNameMapping.applicationNameToPackageNameMappingInstance;
		// refreshApplicationPackageNames();

		// Prepare the tmp properties
		Properties p = new Properties();
		boolean writeUDBFile = false;
		try {
			File userDatabaseFile = new File(waarHomeDirectory + File.separatorChar + USERDATABASEFILENAME);
			FilterInputStream userDatabaseFileInputStream = new UDBInputStream(userDatabaseFile);
			try {
				p.load(userDatabaseFileInputStream);
			} catch (IOException e) {

			} finally {
				try {
					userDatabaseFileInputStream.close();
				} catch (IOException e21) {

				}
			}
		} catch (FileNotFoundException e11) {
			p.setProperty("waaradministrator", "WaarAdministrator"); //$NON-NLS-1$ //$NON-NLS-2$
			// (other properties changepassword, invalidlogonattemptcount and language have appropriate defaults in UserProperties)
			// p.setProperty("", "#@#false#@#[]"); //$NON-NLS-1$ //$NON-NLS-2$
			writeUDBFile = true;
		}

		if (waarUsers == null) {
			waarUsers = new HashMap<String, UserProperties>();
		}
		synchronized (waarUsers) {
			waarUsers.clear();

			Iterator<Entry<Object, Object>> i_p = p.entrySet().iterator();
			while (i_p.hasNext()) {
				Entry<Object, Object> me = i_p.next();
				String userID = (String) me.getKey();
				String userData_tx = (String) me.getValue();
				UserProperties userProperties = new UserProperties(userData_tx);
				waarUsers.put(userID, userProperties);
			}

			if (writeUDBFile) {
				writeUDBFile();
			}
		}

		StringBuffer htmlTemplateText = new StringBuffer(8192);
		String w;

		try {
			BufferedReader in = resourceReader(ARCHITECTURE_HTML, Architecture.class);
			try {
				while ((w = in.readLine()) != null) {
					htmlTemplateText.append(w);
				}
				architectureHtml = htmlTemplateText.toString();
			} catch (IOException e1) {
				architectureHtml = "<HTML><BODY>Error while building architecture page template : " + e1.getMessage() + "</BODY></HTML>"; //$NON-NLS-1$ //$NON-NLS-2$
			} finally {
				try {
					in.close();
				} catch (IOException e2) {

				}
			}
		} catch (ResourceNotFoundException e3) {
			architectureHtml = "<HTML><BODY>Architecture HTML file " + ARCHITECTURE_HTML + "not found.</BODY></HTML>"; //$NON-NLS-1$ //$NON-NLS-2$
		}

		// Set up the languages
		supportedLanguages = new HashMap<String, String>();
		supportedLanguages.put("NL", "Nederlands"); //$NON-NLS-1$//$NON-NLS-2$
		supportedLanguages.put("FR", "Franais"); //$NON-NLS-1$//$NON-NLS-2$
		supportedLanguages.put("DE", "Deutsch"); //$NON-NLS-1$//$NON-NLS-2$
		supportedLanguages.put("EN", "English"); //$NON-NLS-1$//$NON-NLS-2$
		supportedLanguages.put("ES", "Espaol"); //$NON-NLS-1$//$NON-NLS-2$
		supportedLanguages.put("PT", "Portugues"); //$NON-NLS-1$//$NON-NLS-2$
		supportedLanguages.put("IT", "Italiano"); //$NON-NLS-1$ //$NON-NLS-2$
		supportedLanguages.put("NO", "Norsk"); //$NON-NLS-1$//$NON-NLS-2$
		supportedLanguages.put("DK", "Dansk"); //$NON-NLS-1$//$NON-NLS-2$
		supportedLanguages.put("SE", "Svenska"); //$NON-NLS-1$ //$NON-NLS-2$
		supportedLanguages.put("FI", "Suomen"); //$NON-NLS-1$ //$NON-NLS-2$
	}

	/**
	 * @param resourceName
	 * @param resourceClass
	 * @return
	 * @throws ResourceNotFoundException
	 */
	private InputStream resourceInputStream (String resourceName, Class<?> resourceClass) throws ResourceNotFoundException {
		InputStream resourceInputStream = resourceClass.getResourceAsStream(resourceName);
		if (resourceInputStream == null) {
			resourceInputStream = resourceClass.getResourceAsStream("/" + resourceName); //$NON-NLS-1$
			if (resourceInputStream == null) {
				// String presentationFileName = Architecture.waarHomeDirectory + File.separatorChar + resourceClass + (File.separatorChar + Architecture.WEB_CONTENT + File.separatorChar + resourceName);
				// File presentationFile = new File(presentationFileName);
				// try {
				// in = new BufferedReader(new FileReader(presentationFile));
				// } catch (FileNotFoundException e2) {
				throw new ResourceNotFoundException();
				// }
			}
		}
		return resourceInputStream;
	}

	/**
	 * @param resourceName
	 * @param resourceClass
	 * @return
	 * @throws ResourceNotFoundException
	 */
	private BufferedReader resourceReader (String resourceName, Class<?> resourceClass) throws ResourceNotFoundException {
		return new BufferedReader(new InputStreamReader(resourceInputStream(resourceName, resourceClass)));
	}

	/**
	 * Gets a new UserSession
	 * 
	 * @return a new UserSession
	 */
	UserSession newUserSession ( ) {
		UserSession userSession;
		synchronized (userSessions) {
			userSessionIDLastAssigned++;
			userSession = new UserSession(userSessionIDLastAssigned);
			userSessions.put(new Long(userSessionIDLastAssigned), userSession);
		}
		return userSession;
	}
}
