package be.WAAR.PresentationLayer;

import java.io.IOException;
import java.io.UnsupportedEncodingException;

import javax.servlet.*;
import javax.servlet.http.*;

/**
 * @version 1.0
 * @author Erwin
 */
public abstract class HTTPArchitectureServlet extends HttpServlet implements Servlet {

	/**
	 * 
	 */
	private static final long serialVersionUID = -7925812282659471814L;

	/**
	 * the Architecture object (instance). This reference is maintained to ensure that the Architecture singleton remains in the JVM at least as long as any HTTPArchitectureServlet is maintained by the Application Server. It should not normally be referred to by any method.
	 */
	private Architecture architecture;

	/**
	 * A Request has been received that contained an Action= parameter. This means that some action button has been pushed in the current presentation function.
	 * 
	 * @param req
	 *            The request
	 * @param resp
	 *            The object through which the response is returned
	 * @param userSession
	 *            The user session in whose context the function is invoked
	 * @param action
	 *            The value of the Action= parameter
	 * @return The presentation function that will write the output
	 * @throws ServletException
	 */
	private OnLinePresentationFunction processAction (HttpServletRequest req, HttpServletResponse resp, UserSession userSession, String action) throws ServletException {
		PresentationFunctionStack functionStack = userSession.getFunctionStack();
		// action=... ; process requested action
		OnLinePresentationFunction userFunction = (OnLinePresentationFunction) functionStack.getLast();
		try {
			if (action.equalsIgnoreCase("OK")) { //$NON-NLS-1$
				userFunction.fillInPresentationValuesIgnoringDefaults(req);
				userFunction.setPresentationMessage(""); //$NON-NLS-1$
				userFunction.doPresentationLogic();
				userFunction.endPresentationLogic();
				if (userFunction.leaveFunctionOnExit()) {
					userFunction = processReturn(req, resp, userSession, functionStack);
				} else {
					// stay on same function after OK ===> do nothing
				}
			} else {
				if (action.equalsIgnoreCase("Cancel")) { //$NON-NLS-1$
					userFunction.doCancel();
					userFunction = processReturn(req, resp, userSession, functionStack);
				} else {
					if (action.equalsIgnoreCase("Help")) { //$NON-NLS-1$
						userFunction.setPresentation(userFunction.getPresentation().getName() + "Help"); //$NON-NLS-1$
					} else {
						if (action.equalsIgnoreCase("Main")) { //$NON-NLS-1$
							while (functionStack.size() > 0) {
								userFunction.doCancel();
								userFunction = (OnLinePresentationFunction) functionStack.removeLast();
							}
							if (userFunction.isSuspended()) { // Should always be true, but safety rules ...
								userFunction.unSuspend();
							}
							functionStack.addLast(userFunction);
						} else {
							if (action.equalsIgnoreCase("Logoff")) { //$NON-NLS-1$
								do {
									userFunction.doCancel();
									if (functionStack.size() > 0) {
										userFunction = (OnLinePresentationFunction) functionStack.removeLast();
									} else {
										userFunction = null;
									}
								} while (userFunction != null);
							} else {
								userFunction.doActionLogic(action);
							}
						}
					}
				}
			}
		} catch (WaarException e1) {
			log(e1.getMessage(), e1);
			userFunction.setPresentationMessage(e1);
		}
		return userFunction;
	}

	/**
	 * A Request has been received that did not contain an Action= parameter. This means that either a new function is invoked, or else that we are returning from an invoked function to the function that was current when the invoke occurred.
	 * 
	 * @param req
	 *            The request
	 * @param applicationName
	 *            The application name
	 * @param functionName
	 *            The name of the user function in case it is a new invocation
	 * @param userSession
	 *            The user session in whose context
	 * @param userData
	 *            The userData object from the user session
	 * @return The presentation function that will write the output
	 * @throws ServletException
	 */
	private OnLinePresentationFunction processInitOrResume (HttpServletRequest req, String applicationName, String functionName, UserSession userSession, UserData userData) throws ServletException {
		PresentationFunctionStack functionStack = userSession.getFunctionStack();
		OnLinePresentationFunction userFunction = null;

		// processing if there is no action : either function startup or function resume
		String returnPath = "", returnParms = ""; //$NON-NLS-1$ //$NON-NLS-2$
		if (functionStack.size() > 0) {
			userFunction = (OnLinePresentationFunction) functionStack.getLast();
			if (userFunction.isSuspended()) {
				// This means that there has been a function atop of us, but since we are back again the top function now, we can conclude that that function
				// has done a doCancel(), thereby removing itself from the function stack, followed by a redirect to our URL. So we must now reactivate
				// (=unsuspend) ourself.
				userFunction.unSuspend();
				return userFunction;
			}

			// This means we have to suspend ourselves and invoke a new function atop of us. That function must be given our URL so that it knows where to return to
			// (using redirect) when it gets cancelled.
			returnPath = userFunction.getServletPath();
			returnParms = userFunction.getQueryString();
		}

		// Create the presentationFunction object
		// String prefix = Architecture.getApplicationPackageName(applicationName);
		//		String userFunctionName = prefix + "." + functionName; //$NON-NLS-1$
		// Get the package name from the Servlet's Class. Each application should create its own Servlet than extends HTTPArchitectureServlet.
		String prefix = this.getClass().getPackage().getName();
		String userFunctionName = prefix + "." + functionName; //$NON-NLS-1$
		OnLinePresentationFunction newUserFunction;
		try {
			// newUserFunction = (OnLinePresentationFunction) Class.forName(userFunctionName).newInstance();
			newUserFunction = (OnLinePresentationFunction) getClass().getClassLoader().loadClass(userFunctionName).newInstance();
		} catch (InstantiationException e) {
			log(e.getMessage(), e);
			req.getSession().invalidate();
			Architecture.dropUserSession(userSession.getSessionID());
			throw new ServletException(e);
		} catch (IllegalAccessException e) {
			log(e.getMessage(), e);
			req.getSession().invalidate();
			Architecture.dropUserSession(userSession.getSessionID());
			throw new ServletException(e);
		} catch (ClassNotFoundException e) {
			log(e.getMessage(), e);
			req.getSession().invalidate();
			Architecture.dropUserSession(userSession.getSessionID());
			throw new ServletException(e);
		}
		// Initialize the presentationfunction
		newUserFunction.initialize(applicationName, req.getQueryString(), userData, returnPath, returnParms, functionName, req, this);

		// Suspend ourselves (iff the user function existed - which is not the case for the very first function invoked)
		if (userFunction != null) {
			userFunction.suspend();
		}

		// Update the stack
		String functionStartupMode = req.getParameter("FUNCTIONSTARTUPMODE"); //$NON-NLS-1$
		if (functionStartupMode != null && functionStartupMode.equalsIgnoreCase("XFER") && functionStack.size() > 0) { //$NON-NLS-1$
			functionStack.removeLast();
		}
		functionStack.addLast(newUserFunction);

		return newUserFunction;
	}

	/**
	 * Does the returning to a previous function
	 * 
	 * @param req
	 *            The request that was handled and whose corresponding function is now returning
	 * @param resp
	 *            The response object through which the return must be handled
	 * @param userSession
	 *            The user session
	 * @param functionStack
	 *            The (user session's) function stack
	 * @return The user function that will receive control after the cancel, or null if there is no such function (meaning return is to the application main menu)
	 * @throws ServletException
	 */
	private OnLinePresentationFunction processReturn (HttpServletRequest req, HttpServletResponse resp, UserSession userSession, PresentationFunctionStack functionStack) throws ServletException {
		OnLinePresentationFunction abandonedUserFunction = (OnLinePresentationFunction) functionStack.removeLast();
		OnLinePresentationFunction returnToUserFunction;
		if (functionStack.size() == 0) {
			returnToUserFunction = null;
		} else {
			returnToUserFunction = (OnLinePresentationFunction) functionStack.getLast();
			returnToUserFunction.saveInFunctionContext("__WRETURNFROM", abandonedUserFunction.getClass().getSimpleName()); //$NON-NLS-1$
			String backto = abandonedUserFunction.getReturnPath();
			String backparms = abandonedUserFunction.getReturnParms();
			if (backparms.indexOf("USERSESSIONID=") < 0) { //$NON-NLS-1$
				if (backparms.length() > 0) {
					backparms = backparms + "&USERSESSIONID=" + userSession.getSessionID() + "&USERSESSIONTOKEN=" + userSession.getSessionVerificationToken(); //$NON-NLS-1$ //$NON-NLS-2$
				} else {
					backparms = "USERSESSIONID=" + userSession.getSessionID() + "&USERSESSIONTOKEN=" + userSession.getSessionVerificationToken(); //$NON-NLS-1$ //$NON-NLS-2$
				}
			}
			String erdurl = resp.encodeRedirectURL("/" + backto + "?" + backparms); //$NON-NLS-1$ //$NON-NLS-2$
			try {
				resp.sendRedirect(erdurl);
			} catch (IOException e2) {
				log(e2.getMessage(), e2);
				req.getSession().invalidate();
				Architecture.dropUserSession(userSession.getSessionID());
				throw new ServletException(e2.getMessage());
			}
		}
		return returnToUserFunction;
	}

	/**
	 * @see javax.servlet.http.HttpServlet#doGet(javax.servlet.http.HttpServletRequest, javax.servlet.http.HttpServletResponse)
	 */
	public final void doGet (HttpServletRequest req, HttpServletResponse resp) throws ServletException {
		process(req, resp);
	}

	/**
	 * @see javax.servlet.http.HttpServlet#doPost(javax.servlet.http.HttpServletRequest, javax.servlet.http.HttpServletResponse)
	 */
	public final void doPost (HttpServletRequest req, HttpServletResponse resp) throws ServletException {
		process(req, resp);
	}

	/**
	 * @see javax.servlet.GenericServlet#getServletInfo()
	 */
	public final String getServletInfo ( ) {
		return this.getClass().getName() + " - " + this.toString(); //$NON-NLS-1$
	}

	/**
	 * Initializes the Servlet. The WAAR architecture is instantiated, and a reference to it is kept as long as the servlet object is kept by the Application server and the JVM. ServerInfo is obtained from the servlet context and copied into the architecture object for future reference by the
	 * 
	 * @see javax.servlet.GenericServlet#init()
	 */
	public final void init ( ) throws ServletException {
		super.init();
		architecture = Architecture.getInstance();
	}

	/**
	 * Processes a request.
	 * 
	 * @param req
	 *            The request
	 * @param resp
	 *            The object through which the response is returned
	 * @throws ServletException
	 */
	private void process (HttpServletRequest req, HttpServletResponse resp) throws ServletException {
		resp.setContentType("text/html;charset=" + ArchitectureValueGetterForCharSet.UTF8); //$NON-NLS-1$
		resp.setCharacterEncoding(ArchitectureValueGetterForCharSet.UTF8);
		//		resp.setHeader("accept-charset", ArchitectureValueGetterForCharSet.UTF8); //$NON-NLS-1$

		if (req.getCharacterEncoding() == null) {
			try {
				req.setCharacterEncoding(ArchitectureValueGetterForCharSet.UTF8); // ??????????????? The server deciding that the client has sent something in UTF-8 ????????????????????????????
			} catch (UnsupportedEncodingException e1) {
				e1.printStackTrace(System.err);
			}
		}

		String applicationName = req.getContextPath().substring(1); // "/WAAR" ===> "WAAR"
		String functionName = req.getServletPath().substring(1);
		if (functionName.endsWith(".wfn")) { //$NON-NLS-1$
			functionName = functionName.substring(0, functionName.length() - 4);
		}
		String userSessionIDText = req.getParameter("USERSESSIONID"), sessionToken = req.getParameter("USERSESSIONTOKEN"); //$NON-NLS-1$ //$NON-NLS-2$

		UserSession userSession;
		if (userSessionIDText == null) {
			userSession = architecture.newUserSession();
		} else {
			long userSessionID = Long.parseLong(userSessionIDText);
			userSession = Architecture.getUserSession(userSessionID);
			if (userSession == null) {
				log("User Session no longer active : " + req.getRequestURL()); //$NON-NLS-1$
				req.getSession().invalidate();
				throw new ServletException("User Session no longer active."); //$NON-NLS-1$
			}
			if (Long.parseLong(sessionToken) != userSession.getSessionVerificationToken()) {
				log("Session token verification failed : " + req.getRequestURL()); //$NON-NLS-1$
				req.getSession().invalidate();
				Architecture.dropUserSession(userSession.getSessionID());
				throw new ServletException("Session token verification failed."); //$NON-NLS-1$
			}
		}

		try {
			UserData userData = userSession.getUserdata();
			if (userData == null) {
				String erdurl = resp.encodeRedirectURL("/" + applicationName + "/Logon?USERSESSIONID=" + userSession.getSessionID() + "&return=" + req.getRequestURL()); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
				resp.sendRedirect(erdurl);
			} else {
				OnLinePresentationFunction userFunction = null;
				String action = req.getParameter("Action"); //$NON-NLS-1$
				if (action == null) {
					userFunction = processInitOrResume(req, applicationName, functionName, userSession, userData);
				} else {
					userFunction = processAction(req, resp, userSession, action);
				}

				if (userFunction != null) {
					resp.getWriter().write(userFunction.getHTMLOutput());
				} else {
					Architecture.dropUserSession(userSession.getSessionID());
					String erdurl = resp.encodeRedirectURL("/" + applicationName); //$NON-NLS-1$
					resp.sendRedirect(erdurl);
				}
			}
		} catch (IOException e) {
			log(e.getMessage(), e);
			req.getSession().invalidate();
			Architecture.dropUserSession(userSession.getSessionID());
			throw new ServletException(e);
		} catch (RuntimeException e) {
			log(e.getMessage(), e);
			req.getSession().invalidate();
			Architecture.dropUserSession(userSession.getSessionID());
			throw new ServletException(e);
		}
	}
}