package be.WAAR.PresentationLayer;

import java.util.HashMap;
import java.util.LinkedList;
import java.util.Locale;

import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;

/**
 * @author Erwin
 */
public abstract class OnLinePresentationFunction extends PresentationFunction {

	/**
	 * The calling Servlet
	 */
	private HttpServlet callingServlet;

	/**
	 * The map where user context can be saved
	 */
	private HashMap<String, Object> functionUserContext = new HashMap<String, Object>();

	/**
	 * Comment for <code>htmlHandler</code>
	 */
	private HTMLPresentationHandler htmlHandler;

	/**
	 * The function's current Presentation
	 */
	private Presentation presentation;

	/**
	 * Comment for <code>presentationStack</code>
	 */
	private LinkedList<Presentation> presentationStack = new LinkedList<Presentation>();

	/**
	 * Comment for <code>queryString</code>
	 */
	private String queryString;

	/**
	 * Comment for <code>returnParms</code>
	 */
	private String returnParms;

	/**
	 * Comment for <code>returnPath</code>
	 */
	private String returnPath;

	/**
	 * Comment for <code>servletPath</code>
	 */
	private String servletPath;

	/**
	 * Comment for <code>suspended</code>
	 */
	private boolean suspended = false;

	/**
	 * Creates the presentation function
	 */
	public OnLinePresentationFunction ( ) {

	}

	/**
	 * @param presentation
	 */
	private void setPresentation (Presentation presentation) {
		this.presentation = presentation;
		htmlHandler = new HTMLPresentationHandler(this);
	}

	/**
	 * Finalizes the execution of the presentation logic
	 */
	final void endPresentationLogic ( ) {
		// if (getDbLink() != null) {
		// getDbLink().close();
		// }
	}

	/**
	 * Fills in all values that appear in the request's parameters
	 * 
	 * @param req
	 *            The HTTP request from which the presentation values have to be obtained
	 * @throws InvalidAttributeValueException
	 *             If a textual representation of a value is not valid for the type of field it is assigned to.
	 * @throws MandatoryFieldMissingException
	 */
	void fillInPresentationValuesIgnoringDefaults (HttpServletRequest req) throws InvalidAttributeValueException, MandatoryFieldMissingException {
		presentation.setPresentationValuesIgnoringDefaults(req, getUserData());
	}

	/**
	 * @param req
	 * @throws InvalidAttributeValueException
	 */
	void fillInPresentationValuesIncludingDefaults (HttpServletRequest req) throws InvalidAttributeValueException {
		presentation.setPresentationValuesIncludingDefaults(req, getUserData());
	}

	/**
	 * Gets The Servlet object that invoked this function.
	 * 
	 * @return The Servlet object that invoked this function.
	 */
	HttpServlet getCallingServlet ( ) {
		return callingServlet;
	}

	/**
	 * @return the html output to be displayed
	 */
	String getHTMLOutput ( ) {
		return htmlHandler.createOutputData(getUserLocale());
	}

	/**
	 * @return the querystring
	 */
	String getQueryString ( ) {
		return queryString;
	}

	/**
	 * @return the return parms
	 */
	String getReturnParms ( ) {
		return returnParms;
	}

	/**
	 * @return the return url
	 */
	String getReturnPath ( ) {
		return returnPath;
	}

	/**
	 * @return the servletpath
	 */
	String getServletPath ( ) {
		return servletPath;
	}

	/**
	 * @param applicationName
	 *            The application name
	 * @param queryString_p
	 *            The query String from the HttpServletRequest
	 * @param userData
	 *            The userData object holding all data regarding the user
	 * @param returnPath_p
	 *            The path to return to once this function is completed
	 * @param returnParms_p
	 *            The parms to give when returning
	 * @param functionName
	 *            The name of this function
	 * @param req
	 *            The HttpServletRequest object
	 * @param servlet
	 *            The invoking servlet (to which our log requests must be delegated)
	 */
	void initialize (String applicationName, String queryString_p, UserData userData, String returnPath_p, String returnParms_p, String functionName, HttpServletRequest req, HttpServlet servlet) {
		setApplicationName(applicationName);
		this.servletPath = applicationName + req.getServletPath();
		this.queryString = queryString_p;
		setUserData(userData);
		this.returnPath = returnPath_p;
		this.returnParms = returnParms_p;
		this.callingServlet = servlet;
		String presentationName = getInitialPresentationName();
		if (presentationName == null) {
			presentationName = functionName;
		}
		try {
			setPresentation(presentationName);
			fillInPresentationValuesIncludingDefaults(req);
			doInitialLogic();
		} catch (WaarException e) {
			// If the requested presentation could not be set for whatever reason, an "emergency escape" presentation is set instead,
			// so that a presentation message will not be set on a null presenetation
			setPresentationMessage(e);
		}
	}

	/**
	 * Gets the suspended flag
	 * 
	 * @return the suspended flag
	 */
	boolean isSuspended ( ) {
		return suspended;
	}

	// /**
	// * @param servlet
	// */
	// final void setCallingServlet(HTTPArchitectureServlet servlet) {
	// }

	/**
	 * Sets the presentation message text to the message text of the given WaarException, adapted to the user's national language
	 * 
	 * @param e
	 *            The WaarException of whose appearance the user is to be informed in his own language
	 */
	final void setPresentationMessage (WaarException e) {
		// presentation.setMessage(e.getMessageText(getApplicableUserLanguage()));
		presentation.setMessage(e.getMessage());
	}

	/**
	 * Suspends this function until the newly invoked function returns
	 */
	void suspend ( ) {
		suspended = true;
	}

	/**
	 * 
	 */
	void unSuspend ( ) {
		suspended = false;
		try {
			doResumeLogic();
		} catch (WaarException e) {
			setPresentationMessage(e);
		}
	}

	/**
	 * Performs an action other than 'OK', 'HELP', 'CANCEL', 'MAIN' or 'LOGOFF'
	 * 
	 * @param action
	 *            the name of the requested action
	 * @throws ActionUnknownException
	 * @throws WaarException
	 */
	public void doActionLogic (String action) throws WaarException {
		throw new ActionUnknownException(action, getUserLocale());
	}

	/**
	 * Performs the 'cancel' action
	 */
	public void doCancel ( ) {

	}

	/**
	 * Function is called when the presentation is initialized, after the presentation object has been filled.
	 * 
	 * @throws WAARApplicationException
	 * @throws WaarException
	 */
	public abstract void doInitialLogic ( ) throws WaarException;

	/**
	 * Performs the presentation logic
	 * 
	 * @throws WAARApplicationException
	 * @throws WaarException
	 */
	public abstract void doPresentationLogic ( ) throws WaarException;

	/**
	 * Performs the logic when the function is returned to from another invoked function
	 * 
	 * @throws WAARApplicationException
	 * @throws WaarException
	 */
	public abstract void doResumeLogic ( ) throws WaarException;

	/**
	 * Gets an object from the function context
	 * 
	 * @param name
	 *            The name under which the object was previuosly stored in the function context
	 * @return the object saved in the functioncontext under the given name, or null if no object was saved under the given name.
	 * @throws FunctionContextDoesNotHoldThisPropertyException
	 *             If the function context does not contain the named element
	 */
	public final Object getFromFunctionContextMandatory (String name) throws FunctionContextDoesNotHoldThisPropertyException {
		if (!functionUserContext.containsKey(name)) {
			throw new FunctionContextDoesNotHoldThisPropertyException(name, getUserLocale());
		}
		return functionUserContext.get(name);
	}

	/**
	 * Gets an object from the function context
	 * 
	 * @param name
	 *            The name under which the object was previuosly stored in the function context
	 * @return the object saved in the functioncontext under the given name, or null if no object was saved under the given name.
	 * @throws NotFoundException 
	 *             If the function context does not contain the named element
	 */
	public final Object getFromFunctionContext (String name) throws NotFoundException {
		if (!functionUserContext.containsKey(name)) {
			throw new NotFoundException();
		}
		return functionUserContext.get(name);
	}

	/**
	 * Retrieves an object from the user session context
	 * 
	 * @param name
	 *            The name under which something was saved in the session context
	 * @return The object saved under the given name
	 */
	public final Object getFromSessionContext (String name) {
		return getUserData().getFromSessionContext(name);
	}

	/**
	 * Gets the name of the initial presentation to be used with the function.
	 * 
	 * @return The name of the initial presentation to be used with the function. If null is returned, the system will use a presentation of the same name as the Business function (i.e. the servletname in http).
	 */
	public abstract String getInitialPresentationName ( );

	/**
	 * Gets the current presentation
	 * 
	 * @return The current presentation
	 */
	public final Presentation getPresentation ( ) {
		return presentation;
	}

	/**
	 * Gets the Presentation Function Title to be displayed
	 * 
	 * @param language
	 *            The two-character ISO code of the language in which the description is requested
	 * @return the Presentation Function Title to be displayed
	 */
	public abstract String getPresentationFunctionDescription (String language);

	/**
	 * Gets the value of the the named field on the presentation
	 * 
	 * @param presentationFieldName
	 *            The name of the presentation field whose value is to be retrieved
	 * @return The value of the field
	 */
	public final WaarValue getPresentationValue (String presentationFieldName) {
		return presentation.getPresentationValues().get(presentationFieldName);
	}

	/**
	 * Checks whether the function has to retain the "focus" after the presentation logic has been performed, or whether it can be "left".
	 * 
	 * @return true if the function is no longer to be in control after the processing logic has been performed, false if the function is to remain in control.
	 */
	public boolean leaveFunctionOnExit ( ) {
		return true;
	}

	/**
	 * Method to log a message in the application server's log.
	 * 
	 * @param msg
	 *            The message to be logged by the application server
	 */
	public final void log (String msg) {
		callingServlet.log(msg);
	}

	/**
	 * Logs a throwable in the Application Server's log
	 * 
	 * @param e
	 *            The throwable to log
	 */
	public final void log (Throwable e) {
		callingServlet.log(e.getMessage(), e);
	}

	/**
	 * @throws NoStackedPresentationException
	 */
	public final void popoffPresentation ( ) throws NoStackedPresentationException {
		if (presentationStack.size() > 0) {
			Presentation revertToPresentation = presentationStack.removeLast();
			setPresentation(revertToPresentation);
		} else {
			throw new NoStackedPresentationException(getUserLocale());
		}
	}

	/**
	 * Pops up another presentation, while retaining the current presentation on a stack
	 * 
	 * @param presentationName
	 *            The name of the new presentation to pop "above" the current
	 * @throws PresentationNotSetException
	 */
	public final void popupPresentation (String presentationName) throws PresentationNotSetException {
		presentationStack.addLast(presentation);
		setPresentation(presentationName);
	}

	/**
	 * Stores an object under the given name in the function context for later retrieval
	 * 
	 * @param name
	 *            The name under which the object is to be stored in the function context
	 * @param object
	 *            The object to be stored in the function context
	 */
	public final void saveInFunctionContext (String name, Object object) {
		functionUserContext.put(name, object);
	}

	/**
	 * Saves an object in the user session context
	 * 
	 * @param name
	 *            The name under which something should be saved in the context
	 * @param o
	 *            The object to save under the given name
	 */
	public final void saveInSessionContext (String name, Object o) {
		getUserData().saveInSessionContext(name, o);
	}

	/**
	 * Sets the presentation currently to be associated with this presentation function
	 * 
	 * @param name
	 *            The name of the presentation to be set current
	 * @throws PresentationNotSetException
	 */
	public final void setPresentation (String name) throws PresentationNotSetException {
		Locale locale = getUserLocale();
		try {
			presentation = new Presentation(this.getClass(), name, locale);
		} catch (WaarException e) {
			log(e);
			try {
				presentation = new Presentation(OnLinePresentationFunction.class, "PresentationProblem", locale); //$NON-NLS-1$
			} catch (WaarException e1) {
				throw new RuntimeException("Neither the application's requested presentation, nor the architecture's presentation reporting this problem, could be set.  Problem with application's presentation was : " + e.getMessage() + ".  Problem with architecture's presentation was : " + e1.getMessage(), e); //$NON-NLS-1$ //$NON-NLS-2$
			}
			htmlHandler = new HTMLPresentationHandler(this);
			throw new PresentationNotSetException(name, getApplicationName(), e.getMessage(), locale);

		}
		htmlHandler = new HTMLPresentationHandler(this);
	}

	/**
	 * Sets the presentation message text in the language of the user.
	 * 
	 * @param localizedMessage
	 *            The localized message text to display to the user
	 */
	public final void setPresentationMessage (String localizedMessage) {
		presentation.setMessage(localizedMessage);
	}

	/**
	 * Sets a value on the presentation where that value is given in String format
	 * 
	 * @param presentationFieldName
	 * @param v
	 * @throws WaarException
	 */
	public final void setPresentationValue (String presentationFieldName, boolean v) throws WaarException {
		setPresentationValue(presentationFieldName, getPresentation().getPresentationField(presentationFieldName, getUserLocale()).getType().getWaarValue(v, getUserLocale()));
	}

	/**
	 * Sets a value on the presentation where that value is given in String format
	 * 
	 * @param presentationFieldName
	 * @param v
	 * @throws WaarException
	 */
	public final void setPresentationValue (String presentationFieldName, char v) throws WaarException {
		setPresentationValue(presentationFieldName, Character.toString(v));
	}

	/**
	 * Sets a value on the presentation where that value is given in String format
	 * 
	 * @param presentationFieldName
	 * @param v
	 * @throws WaarException
	 */
	public final void setPresentationValue (String presentationFieldName, double v) throws WaarException {
		setPresentationValue(presentationFieldName, Double.toString(v));
	}

	/**
	 * Sets a value on the presentation where that value is given in String format
	 * 
	 * @param presentationFieldName
	 * @param v
	 * @throws WaarException
	 */
	public final void setPresentationValue (String presentationFieldName, float v) throws WaarException {
		setPresentationValue(presentationFieldName, Float.toString(v));
	}

	/**
	 * Sets a value on the presentation where that value is given in String format
	 * 
	 * @param presentationFieldName
	 * @param v
	 * @throws WaarException
	 */
	public final void setPresentationValue (String presentationFieldName, int v) throws WaarException {
		setPresentationValue(presentationFieldName, getPresentation().getPresentationField(presentationFieldName, getUserLocale()).getType().getWaarValue(v, getUserLocale()));
	}

	/**
	 * Sets a value on the presentation where that value is given in String format
	 * 
	 * @param presentationFieldName
	 * @param v
	 * @throws WaarException
	 */
	public final void setPresentationValue (String presentationFieldName, long v) throws WaarException {
		setPresentationValue(presentationFieldName, getPresentation().getPresentationField(presentationFieldName, getUserLocale()).getType().getWaarValue(v, getUserLocale()));
	}

	/**
	 * Sets a value on the presentation where that value is given in String format
	 * 
	 * @param presentationFieldName
	 * @param v
	 * @throws WaarException
	 */
	public final void setPresentationValue (String presentationFieldName, String v) throws WaarException {
		presentation.getPresentationValues().put(presentationFieldName, presentation.getPresentationField(presentationFieldName, getUserLocale()).getType().encode(v, getUserData(), getUserLocale()));
	}

	/**
	 * Sets a value on the presentation (except if the presentation is the architecure's rescue presentation, which means that the presentation that was actually requested by the application could not be set in the first place, which will certainly prohibit to set any fields on them). This is a bit of a kludge. We should not even end up in the application anymore in that case.
	 * 
	 * @param presentationFieldName
	 * @param value
	 * @throws WaarException
	 */
	public final void setPresentationValue (String presentationFieldName, WaarValue value) throws WaarException {
		WaarValue checkedValue = presentation.getPresentationField(presentationFieldName, getUserLocale()).checkValidWaarValue(value, getUserData());
//		if (presentation.getPresentationFields().containsKey(presentationFieldName)) {
			presentation.getPresentationValues().put(presentationFieldName, checkedValue);
//		} else {
//			throw new PresentationFieldMissingException(presentationFieldName, presentation.getName(), getUserLocale());
//		}
	}
}