package be.WAAR.PresentationLayer;

import java.io.BufferedReader;
import java.io.IOException;
import java.util.HashMap;
import java.util.Locale;

/**
 * @author Erwin
 */
public class PresentationField {

	/**
	 * The number of attributes that are recorded for presentation fields
	 */
	private static int QY_ATTRIBUTES_MAX = 8;

	/**
	 * 
	 */
	private static final String TABLEROWHTML = ".tablerowhtml"; //$NON-NLS-1$

	/**
	 * The field's default value
	 */
	private WaarValue defaultValue;

	/**
	 * 
	 */
	private boolean inputEnabled;

	/**
	 * The list of columns that make up this field if this field happens to be a list.
	 */
	private HashMap<String, PresentationField> listFieldMap;

	/**
	 * Indicates whether the field is mandatory upon presentation input
	 */
	private boolean mandatory;

	/**
	 * The name of the field
	 */
	private String name = ""; //$NON-NLS-1$

	/**
	 * The renderer for the field
	 */
	private Renderer presentationRenderer;

	/**
	 * 
	 */
	private String presentationSizeSpec;

	/**
	 * Comment for <code>rowTemplateText</code>
	 */
	private StringBuffer rowTemplateText;

	/**
	 * Comment for <code>tabindex</code>
	 */
	private int tabIndex = -1;

	/**
	 * The type of the data field
	 */
	private PresentationType type;

	/**
	 * Parses the text for values of all info describing a presentation field : name,type,defaultvalue,mandatoryflag
	 * 
	 * @param txt
	 *            A comma-separated String stating the attributes of a presentation field
	 * @param presentationFunctionClass
	 *            The Class object of the presentation function for which the type is needed. This object acts as a handle to its classLoader, which might be needed to gain access to package-defined presentation types (such as e.g. enumeration types)
	 * @param locale
	 * @throws WaarException
	 */
	PresentationField (String txt, Class<? extends OnLinePresentationFunction> presentationFunctionClass, Locale locale) throws WaarException {
		int commaPos, ord = 0;
		String remain = txt, keyWord;

		do {
			commaPos = remain.indexOf(","); //$NON-NLS-1$
			if (commaPos >= 0) {
				keyWord = remain.substring(0, commaPos).trim();
				remain = remain.substring(commaPos + 1);
			} else {
				keyWord = remain.trim();
				remain = ""; //$NON-NLS-1$
			}
			switch (ord) {
				case 0: {
					name = keyWord;
					break;
				}
				case 1: {
					type = Architecture.getAttributeType(keyWord, presentationFunctionClass, locale);
					if (type instanceof List || type instanceof DynamicColumnList) {
						listFieldMap = new HashMap<String, PresentationField>();
					}
					break;
				}
				case 2: {
					if (keyWord.length() > 0) {
						defaultValue = type.encode(keyWord, null, locale); // No userdata, meaning we can't use the presentation-level defaults to define default dates or times (because the formatting of those depends on having user data available)
					}
					break;
				}
				case 3: {
					// inputmandatory ?
					// List fields are always output, and therefore always optional
					if (type instanceof List || type instanceof DynamicColumnList || keyWord.equals("0") || keyWord.equalsIgnoreCase("OPTIONAL")) { //$NON-NLS-1$ //$NON-NLS-2$
						mandatory = false;
					} else {
						if (keyWord.equals("") || keyWord.equals("1") || keyWord.equalsIgnoreCase("MANDATORY")) { //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
							mandatory = true;
						} else {
							throw new IllegalArgumentException(keyWord);
						}
					}
					break;
				}
				case 4: {
					// tabindex
					if (keyWord.length() > 0) {
						tabIndex = Integer.parseInt(keyWord);
					} else {
						tabIndex = -1;
					}
					break;
				}
				case 5: {
					// input mode enabled ?
					// List fields are always output-only
					if (type instanceof List || type instanceof DynamicColumnList) {
						inputEnabled = false;
					} else {
						if (keyWord.length() == 0 || keyWord.equalsIgnoreCase("INPUT")) { //$NON-NLS-1$
							inputEnabled = true;
						} else {
							if (keyWord.equalsIgnoreCase("OUTPUT")) { //$NON-NLS-1$
								inputEnabled = false;
							} else {
								throw new IllegalArgumentException(keyWord);
							}
						}
					}
					break;
				}
				case 6: {
					// presentationRenderer
					if (keyWord.length() > 0) {
						presentationRenderer = type.getPresentationRenderingStyle(keyWord);
					} else {
						presentationRenderer = type.getDefaultPresentationRenderingStyle();
					}
					break;
				}
				case 7: {
					// presentationSizeSpec
					if (keyWord.length() > 0) {
						presentationSizeSpec = keyWord;
					} else {
						presentationSizeSpec = ""; //$NON-NLS-1$
					}
					break;
				}
			}
			ord++;
			// } while (remain.length() > 0 && ord < QY_ATTRIBUTES_MAX);
		} while (ord < QY_ATTRIBUTES_MAX);

	}

	/**
	 * @return
	 */
	int getColumnCount ( ) {
		return listFieldMap == null ? 0 : listFieldMap.size();
	}

	/**
	 * Gets the field's default value
	 * 
	 * @return The field's default value, or null if there is no default value defined for this field at the presentation level
	 */
	WaarValue getDefaultValue ( ) {
		return defaultValue;
	}

	/**
	 * Gets presentationRenderer
	 * 
	 * @return presentationRenderer.
	 */
	final Renderer getPresentationRenderer ( ) {
		return presentationRenderer;
	}

	/**
	 * Gets The tabIndex for this field
	 * 
	 * @return The tabIndex for this field
	 */
	int getTabIndex ( ) {
		return tabIndex;
	}

	/**
	 * Gets inputEnabled
	 * 
	 * @return inputEnabled.
	 */
	final boolean isInputEnabled ( ) {
		return inputEnabled;
	}

	/**
	 * Gets the 'mandatory' indicator
	 * 
	 * @return true if the field is mandatory on presentation input
	 */
	boolean isMandatory ( ) {
		return mandatory;
	}

	/**
	 * @param mode
	 */
	final void setInOutMode (InOutMode mode) {
		inputEnabled = mode == InOutMode.IN;
	}

	/**
	 * @param presentationName
	 * @param presentationFunctionClass
	 */
	void setTableRowHtml (String presentationName, Class<? extends OnLinePresentationFunction> presentationFunctionClass) {
		rowTemplateText = new StringBuffer(8192);
		String tableRowHTMLResourceName = presentationName + "." + name + TABLEROWHTML; //$NON-NLS-1$

		try {
			BufferedReader in = Architecture.getResourceReader(tableRowHTMLResourceName, presentationFunctionClass);
			try {
				String w;
				while ((w = in.readLine()) != null) {
					rowTemplateText.append(w);
				}
			} catch (IOException e1) {
				rowTemplateText = new StringBuffer("Error while building table row template : " + e1.getMessage()); //$NON-NLS-1$
			} finally {
				try {
					in.close();
				} catch (IOException e2) {
					e2.printStackTrace(System.err);
				}
			}
		} catch (ResourceNotFoundException e) {
			rowTemplateText = new StringBuffer("Table row template HTML resource " + tableRowHTMLResourceName + "not found."); //$NON-NLS-1$ //$NON-NLS-2$
		}
	}

	/**
	 * Registers a column in a Presentation field
	 * 
	 * @param presentationField
	 *            The Presentation field defining the column
	 */
	public void addColumn (PresentationField presentationField) {
		listFieldMap.put(presentationField.getName(), presentationField);
	}

	/**
	 * Gets the Presentation Field describing the named column of a list field
	 * 
	 * @param columnName
	 *            The name of the requested column
	 * @param locale
	 * @return The requested column; null if such a column does not exist
	 * @throws NoColumnsInNonListFieldsException
	 */
	public PresentationField getColumn (String columnName, Locale locale) throws NoColumnsInNonListFieldsException {
		if (listFieldMap == null) {
			throw new NoColumnsInNonListFieldsException(name, type.getClass().getName(), columnName, locale);
		} else {
			return listFieldMap.get(columnName);
		}
	}

	/**
	 * Gets the name of the field
	 * 
	 * @return The name of the field
	 */
	public String getName ( ) {
		return name;
	}

	/**
	 * @return
	 */
	public String getPresentationSizeSpec ( ) {
		return presentationSizeSpec;
	}

	/**
	 * @return The row template text of a list field
	 */
	public StringBuffer getRowTemplateText ( ) {
		return rowTemplateText;
	}

	/**
	 * Gets the type of the data field
	 * 
	 * @return The type of the data field
	 */
	public PresentationType getType ( ) {
		return type;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.lang.Object#toString()
	 */
	public String toString ( ) {
		return name + " " + type; //$NON-NLS-1$
	}

	/**
	 * @param value
	 * @param userData 
	 * @return 
	 * @throws InvalidAttributeValueException 
	 */
	public WaarValue checkValidWaarValue (WaarValue value, UserData userData) throws InvalidAttributeValueException {
		return type.checkValidWaarValue(value, userData);
	}
}
