package be.WAAR.PresentationLayer;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Locale;
import java.util.Map;

import be.erwinsmout.MyMessageFormat;

/**
 * TextsEnumerationType is the presentation type for enumerated text values
 * 
 * @author Erwin
 */
public abstract class TextsEnumerationType extends GenericEnumerationType {

	/**
	 * The textual values of all the enumerated possibilities, with the keys in case-insensitive canonical form
	 */
	private HashMap<String, String> caseInsensitiveValuesAndLabels;

	/**
	 * 
	 */
	private boolean caseSensitive = false;

	/**
	 * The textual values of all the enumerated possibilities
	 */
	private Map<String, String> valuesAndLabels = new HashMap<String, String>();

	/**
	 * Creates the Enumeration type instance
	 * 
	 * @param valuesAndLabels
	 *            The valid values for the enumeration
	 */
	public TextsEnumerationType (Map<String, String> valuesAndLabels) {
		this.valuesAndLabels.putAll(valuesAndLabels);

		HashSet<String> s = new HashSet<String>();
		for (String value : valuesAndLabels.keySet()) {
			String ucValue = value.toUpperCase();
			if (s.contains(ucValue)) {
				caseSensitive = true;
			} else {
				s.add(ucValue);
			}
		}

		if (!caseSensitive) {
			caseInsensitiveValuesAndLabels = new HashMap<String, String>();
			for (String value : valuesAndLabels.keySet()) {
				caseInsensitiveValuesAndLabels.put(value.toUpperCase(), value);
			}
		}
	}

	/**
	 * Gets valuesAndLabels
	 * 
	 * @return valuesAndLabels.
	 */
	final Map<String, String> getValuesAndLabels ( ) {
		return valuesAndLabels;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.waar.AttributeType#validate(java.lang.String)
	 */
	public final WaarValue encode (String val, UserData userData, Locale locale) throws InvalidAttributeValueException {
		if (caseSensitive) {
			String encodeKey = val;
			if (valuesAndLabels.containsKey(encodeKey)) {
				return new GenericStringValue(val);
			} else {
				throw new InvalidAttributeValueException(val, this.getClass().getSimpleName(), valuesAndLabels.keySet().toString(), locale);
			}
		} else {
			String encodeKey = val.toUpperCase();
			if (caseInsensitiveValuesAndLabels.containsKey(encodeKey)) {
				return new GenericStringValue(caseInsensitiveValuesAndLabels.get(encodeKey));
			} else {
				throw new InvalidAttributeValueException(val, this.getClass().getSimpleName(), valuesAndLabels.keySet().toString(), locale);
			}
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.WAAR.PresentationLayer.PresentationType#getDefaultPresentationRenderingStyle()
	 */
	@Override
	public final Renderer getDefaultPresentationRenderingStyle ( ) {
		return ComboBoxRenderer.getInstance();
	}

	public String getEmptyNakedStringRepresentation (PresentationField presentationField, Locale locale) {
		return ""; //$NON-NLS-1$
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.waar.AttributeType#validate(java.lang.String)
	 */
	public final String getNakedStringRepresentation (WaarValue waarValue, OnLinePresentationFunction userFunction, PresentationField presentationField, Locale locale) {
		String stringValue = ((GenericStringValue) waarValue).getString();
		if (caseSensitive) {
			String label = valuesAndLabels.get(stringValue);
			return (label == null ? "" : label); //$NON-NLS-1$
		} else {
			String label = caseInsensitiveValuesAndLabels.get(stringValue.toUpperCase());
			return (label == null ? "" : label); //$NON-NLS-1$
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.WAAR.PresentationLayer.PresentationType#getPresentationRenderingStyle(java.lang.String)
	 */
	@Override
	public final Renderer getPresentationRenderingStyle (String renderingStyleName) {
		if (renderingStyleName.equalsIgnoreCase("PASSWORD")) { //$NON-NLS-1$
			return PassWordRenderer.getInstance();
		}
		if (renderingStyleName.equalsIgnoreCase("TEXTBOX")) { //$NON-NLS-1$
			return TextBoxRenderer.getInstance();
		}
		return ComboBoxRenderer.getInstance();
	}

	@Override
	public final WaarValue getWaarValue (int v, Locale locale) {
		throw new UnsupportedOperationException();
	}

	@Override
	public final WaarValue getWaarValue (long v, Locale locale) {
		throw new UnsupportedOperationException();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.WAAR.PresentationLayer.PresentationType#getWaarValue(boolean, java.util.Locale)
	 */
	@Override
	public final WaarValue getWaarValue (boolean b, Locale locale) {
		throw new UnsupportedOperationException();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.WAAR.PresentationLayer.PresentationType#checkValidWaarValue(be.WAAR.PresentationLayer.WaarValue, be.WAAR.PresentationLayer.UserData)
	 */
	@Override
	public final WaarValue checkValidWaarValue (WaarValue value, UserData userData) throws InvalidAttributeValueException {
		if (!(value instanceof GenericStringValue)) {
			Locale locale = userData.getLocale();
			throw new InvalidAttributeValueException(value.toWAARString(userData), this.getClass().getSimpleName(), MyMessageFormat.format(I18N.getString(locale, "DynamicColumnList.InvalidValueClass"), new String[] { getClass().getName(), value.getClass().getName() }), locale); //$NON-NLS-1$
		}

		return encode(((GenericStringValue) value).getString(), userData, userData.getLocale());
		
	}
}
