package be.WAAR.PresentationLayer;

import java.text.SimpleDateFormat;
import java.util.Locale;

/**
 * UserData objects store all the information regarding a user that is connected to the system
 * 
 * @author Erwin
 */
public class UserData {

	/**
	 * The users's preferred date format
	 */
	private SimpleDateFormat preferredDateFormat;

	/**
	 * The user's preferred DateTime format, derived from his preferred DateFormat and his preferred TimeFormat
	 */
	private SimpleDateFormat preferredDateTimeFormat;

	/**
	 * The user's preferred time format as entered on the logon screen
	 */
	private SimpleDateFormat preferredTimeFormat;

	/**
	 * The User ID
	 */
	private String userID;

	/**
	 * The users's preferred language
	 */
	private GenericStringValue userLanguage;

	/**
	 * The session this user is currently using
	 */
	private UserSession userSession;

	/**
	 * 
	 */
	private Locale locale;

	/**
	 * Creates the UserData object, setting the user ID, the user's preferred date and time format, the user's preferred language and his password
	 * 
	 * @param userid
	 *            The user ID
	 * @param dateFormat
	 *            The users's preferred date format
	 * @param timeFormat
	 *            The user's preferred time format as entered on the logon screen
	 * @param userLanguage
	 *            The users's preferred language
	 * @throws InvalidAttributeValueException
	 *             if any of the given values are not valid
	 */
	UserData (String userid, String dateFormat, String timeFormat, GenericStringValue userLanguage) throws InvalidAttributeValueException {
		this.userID = userid;
		this.userLanguage = userLanguage;
		this.locale = new Locale(userLanguage.getString());
		try {
			preferredDateFormat = new SimpleDateFormat(dateFormat);
		} catch (IllegalArgumentException e) {
			throw new InvalidAttributeValueException(dateFormat, e.getMessage(), locale);
		}
		try {
			preferredTimeFormat = new SimpleDateFormat(timeFormat);
		} catch (IllegalArgumentException e) {
			throw new InvalidAttributeValueException(dateFormat, e.getMessage(), locale);
		}
		String dateTimeFormat = dateFormat + ' ' + timeFormat;
		try {
			preferredDateTimeFormat = new SimpleDateFormat(dateTimeFormat);
		} catch (IllegalArgumentException e) {
			throw new InvalidAttributeValueException(dateTimeFormat, e.getMessage(), locale);
		}
	}

	/**
	 * Creates the UserData object, setting the user ID, the user's preferred date and time format, the user's preferred language and his password, and the userSession
	 * 
	 * @param userid
	 *            The User ID
	 * @param dateFormat
	 *            The users's preferred date format
	 * @param timeFormat
	 *            The user's preferred time format as entered on the logon screen
	 * @param userLanguage
	 *            The users's preferred language
	 * @param userSession
	 *            The session this user is currently using
	 * @throws InvalidAttributeValueException
	 *             if any of the given values are not valid
	 */
	UserData (String userid, String dateFormat, String timeFormat, GenericStringValue userLanguage, UserSession userSession) throws InvalidAttributeValueException {
		this(userid, dateFormat, timeFormat, userLanguage);
		this.userSession = userSession;
	}

	/**
	 * Retrieves an object from the user session context
	 * 
	 * @param name
	 *            The name under which something was saved in the context
	 * @return The object saved under the given name
	 */
	Object getFromSessionContext (String name) {
		if (userSession != null) {
			return userSession.getFromContext(name);
		} else {
			return null;
		}
	}

	/**
	 * @return Gets preferredDateFormat.
	 */
	SimpleDateFormat getPreferredDateFormat ( ) {
		return preferredDateFormat;
	}

	/**
	 * Gets The user's preferred DateTime format
	 * 
	 * @return The user's preferred DateTime format
	 */
	SimpleDateFormat getPreferredDateTimeFormat ( ) {
		return preferredDateTimeFormat;
	}

	/**
	 * @return the preferred time format
	 */
	SimpleDateFormat getPreferredTimeFormat ( ) {
		return preferredTimeFormat;
	}

	/**
	 * Gets The users's preferred language
	 * 
	 * @return The users's preferred language
	 */
	GenericStringValue getUserLanguage ( ) {
		return userLanguage;
	}

	/**
	 * Getss The session this user is currently using
	 * 
	 * @return The session this user is currently using
	 */
	UserSession getUserSession ( ) {
		return userSession;
	}

	/**
	 * Saves an object in the user session context
	 * 
	 * @param name
	 *            The name under which something should be saved in the context
	 * @param o
	 *            The object to save under the given name
	 */
	void saveInSessionContext (String name, Object o) {
		if (userSession != null) {
			userSession.saveInContext(name, o);
		}
	}

	/**
	 * Gets the User ID
	 * 
	 * @return The User ID
	 */
	public String getUserID ( ) {
		return userID;
	}

	/**
	 * @return
	 */
	public Locale getLocale ( ) {
		return locale;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString ( ) {
		return userID + "_" + userLanguage + "_" + preferredDateFormat.toPattern() + "_" + preferredTimeFormat.toPattern();   //$NON-NLS-1$//$NON-NLS-2$//$NON-NLS-3$
	}
}
