/*
 * Created on 16-aug-2008
 */
package be.WAAR.PresentationLayer;

import java.util.Iterator;
import java.util.TreeSet;

/**
 * UserProperties hold the password and the "need-to-change-password" flag for a User
 * 
 * @author Erwin Smout
 */
public class UserProperties {

	/**
	 * The time lapse in milliseconds that a userID must be considered as disabled after exceeding the maximum allowed invalid password logons
	 */
	private static final int DISABLEPERIODMILLIS = 3600000;

	/**
	 * The quantity of allowed invalid password logons before the userID is disabled
	 */
	private static final int INVALIDATEUSERATATTEMPT = 3;

	/**
	 * the user's default language
	 */
	private String language;

	/**
	 * The password
	 */
	private String password;

	/**
	 * flag indicating whether the password must be changed at the next logon
	 */
	private boolean passwordToBeChangedAtNextLogon;

	/**
	 * The set of times when a suspicious logon attempt was detected.
	 */
	private TreeSet<Long> suspiciousLogonAttemptTimes = new TreeSet<Long>();

	/**
	 * Creates the UserProperties from a String in the format of UserProperties.toString()
	 * 
	 * @param userData_tx
	 *            A String holding the relevant properties in the format produced by this class' toString() method
	 */
	UserProperties (String userData_tx) {
		String[] properties = userData_tx.split("#@#"); //$NON-NLS-1$
		this.password = properties[0].trim();

		String passwordChangeRequired = properties.length > 1 ? properties[1].trim() : "true"; //$NON-NLS-1$
		this.passwordToBeChangedAtNextLogon = Boolean.valueOf(passwordChangeRequired).booleanValue();

		String suspiciousLogonAttemptTimes_tx = properties.length > 2 ? properties[2].trim() : "[]"; //$NON-NLS-1$;
		suspiciousLogonAttemptTimes_tx = suspiciousLogonAttemptTimes_tx.substring(1, suspiciousLogonAttemptTimes_tx.length() - 1).trim();
		while (suspiciousLogonAttemptTimes_tx.length() > 0) {
			int i = suspiciousLogonAttemptTimes_tx.indexOf(',');
			String suspiciousLogonAttemptTime_tx;
			if (i == -1) {
				suspiciousLogonAttemptTime_tx = suspiciousLogonAttemptTimes_tx.trim();
				suspiciousLogonAttemptTimes_tx = ""; //$NON-NLS-1$
			} else {
				suspiciousLogonAttemptTime_tx = suspiciousLogonAttemptTimes_tx.substring(0, i).trim();
				suspiciousLogonAttemptTimes_tx = suspiciousLogonAttemptTimes_tx.substring(i + 1);
			}
			long suspiciousLogonAttemptTime = Long.parseLong(suspiciousLogonAttemptTime_tx);
			suspiciousLogonAttemptTimes.add(new Long(suspiciousLogonAttemptTime));
		}

		language = properties.length > 3 ? properties[3].trim() : "EN"; //$NON-NLS-1$

	}

	/**
	 * Removes suspicious logon attempts from the log that are older than the disable period
	 * 
	 * @return the pivot time
	 */
	private long removeOldPasswordFailures ( ) {
		long currentTimeMillis = System.currentTimeMillis();

		Iterator<Long> i_suspiciousLogonAttemptTimes = suspiciousLogonAttemptTimes.iterator();
		while (i_suspiciousLogonAttemptTimes.hasNext()) {
			Long suspiciousLogonAttemptTime = i_suspiciousLogonAttemptTimes.next();
			if (suspiciousLogonAttemptTime.longValue() < currentTimeMillis - DISABLEPERIODMILLIS) { // longer than an hour ago
				i_suspiciousLogonAttemptTimes.remove();
			}
		}
		return currentTimeMillis;
	}

	/**
	 * Gets The password
	 * 
	 * @return The password
	 */
	String getPassword ( ) {
		return password;
	}

	/**
	 * @throws MaximumInvalidPasswordsExceededException
	 */
	void registerSuspiciousLogonAttempt ( ) throws MaximumInvalidPasswordsExceededException {
		long currentTimeMillis = removeOldPasswordFailures();

		// Always add the time of the last attempt
		suspiciousLogonAttemptTimes.add(new Long(currentTimeMillis));
		if (suspiciousLogonAttemptTimes.size() > INVALIDATEUSERATATTEMPT) {
			throw new MaximumInvalidPasswordsExceededException();
		}
	}

	/**
	 * Gets the user's default language
	 * 
	 * @return the user's default language
	 */
	public String getLanguage ( ) {
		return language;
	}

	/**
	 * Gets The set of times when a suspicious logon attempt was detected.
	 * 
	 * @return The set of times when a suspicious logon attempt was detected.
	 */
	public TreeSet<Long> getSuspiciousLogonAttemptTimes ( ) {
		return suspiciousLogonAttemptTimes;
	}

	/**
	 * Checks whether this userID has been disabled
	 * 
	 * @return true if this userID has been disabled
	 */
	public boolean isDisabled ( ) {
		long currentTimeMillis = removeOldPasswordFailures();

		if (suspiciousLogonAttemptTimes.size() > INVALIDATEUSERATATTEMPT) {
			long lastSuspiciousLogonAttemptTime = suspiciousLogonAttemptTimes.last().longValue();
			if (currentTimeMillis < lastSuspiciousLogonAttemptTime + DISABLEPERIODMILLIS) {
				return true;
			}
		}
		return false;
	}

	/**
	 * Checks whether the password must be changed at the next logon
	 * 
	 * @return true if the password must be changed at the next logon
	 */
	public boolean isPasswordToBeChangedAtNextLogon ( ) {
		return passwordToBeChangedAtNextLogon;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.lang.Object#toString()
	 */
	public String toString ( ) {
		return password + "#@#" + passwordToBeChangedAtNextLogon + "#@#" + suspiciousLogonAttemptTimes + "#@#" + language; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
	}
}
