package be.SIRAPRISE.webclient;

import java.util.Arrays;
import java.util.LinkedList;

import be.SIRAPRISE.client.ConnectionClosedException;
import be.SIRAPRISE.client.DBConnection;
import be.SIRAPRISE.client.DBTransaction;
import be.SIRAPRISE.client.DDLCapture;
import be.SIRAPRISE.client.DmlAddCommand;
import be.SIRAPRISE.client.ErrorMessageException;
import be.SIRAPRISE.client.NAMES.*;
import be.SIRAPRISE.client.jsba.Index;
import be.SIRAPRISE.client.jsba.IndexComponent;
import be.SIRAPRISE.util.BracketParser;
import be.SIRAPRISE.util.NameValueResult;
import be.SIRAPRISE.util.NoClosingBracketException;
import be.SIRAPRISE.util.NoOpeningBracketException;
import be.WAAR.PresentationLayer.*;
import be.erwinsmout.MyMessageFormat;

/**
 * Function to create a new SIRA_PRISE index on a record type of a relvar
 * 
 * @author Erwin
 */
public class IndexCreate extends IndexWebClientFunction {

	/**
	 * Checks whether the storage space specified by the user matches the index type requested by the user
	 * 
	 * @param transaction
	 *            The connection on which to query the storage space type
	 * @param wv_indexType
	 *            The indextype specified by the user
	 * @param storageSpaceQuery
	 *            The query text already built (based on the index type specified by the user)
	 * @throws StorageSpaceSpecifiedDoesNotMatchIndexTypeSpecifiedException
	 *             If the index type specified by the user does not match the actual type of storage space in which the user wants the index stored
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server
	 * @throws ErrorMessageException
	 *             If the response obtained from the server is not a commit confirmation, but an error response instead.
	 */
	private void checkStorageSpace (DBTransaction transaction, NumberedEnumerationValue wv_indexType, String storageSpaceQuery) throws StorageSpaceSpecifiedDoesNotMatchIndexTypeSpecifiedException, ConnectionClosedException, ErrorMessageException {
		if (transaction.execQuery(storageSpaceQuery).size() == 0) {
			throw new StorageSpaceSpecifiedDoesNotMatchIndexTypeSpecifiedException(wv_indexType.getStringValue(), getUserLocale());
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.fgov.kszbcss.WAAR.OnLinePresentationFunction#doInitialLogic()
	 */
	public void doInitialLogic ( ) throws WaarException {
		setSiraPriseServerPortPresentationValues();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.fgov.kszbcss.WAAR.OnLinePresentationFunction#doPresentationLogic()
	 */
	public void doPresentationLogic ( ) throws WaarException {
		SiraPriseServer sirapriseServer = setSiraPriseServerPortPresentationValues();

		DBConnection dbc = getDBConnection(sirapriseServer);
		try {
			DBTransaction startTransaction = dbc.startTransaction(getUserData().getUserID(), true, "", new byte[] {}, DDLCapture.getDDLCapture(isDDLCapture())); //$NON-NLS-1$

			String recordTypeName = ((GenericStringValue) getPresentationValue(ATTRIBUTENAMES.RECORDTYPENAME)).getString().trim();
			String relvarName = ((GenericStringValue) getPresentationValue(ATTRIBUTENAMES.RELVARNAME)).getString().trim();
			String indexName = ((GenericStringValue) getPresentationValue(ATTRIBUTENAMES.INDEXNAME)).getString().trim();
			String fileName = ((GenericStringValue) getPresentationValue(ATTRIBUTENAMES.FILENAME)).getString().trim();
			int storageSpaceID = ((GenericIntegerValue) getPresentationValue(ATTRIBUTENAMES.STORAGESPACEID)).getInteger();

			Index index = new Index(relvarName, indexName, recordTypeName, fileName, storageSpaceID);
			LinkedList<DmlAddCommand> cmds = new LinkedList<DmlAddCommand>(Arrays.asList(new DmlAddCommand[] { index.dbAddCommand(RELVARNAMES.INDEX) }));

			// Check the user-specified index type against the storage space type
			NumberedEnumerationValue wv_indexType = (NumberedEnumerationValue) getPresentationValue("INDEXTYPE"); //$NON-NLS-1$
			int indexType = wv_indexType.getIntegerValue();
			String restrictCondition = "AND(EQ(" + ATTRIBUTENAMES.FILENAME + ",FILENAME(" + fileName + ")),EQ(" + ATTRIBUTENAMES.STORAGESPACEID + ",INT(" + storageSpaceID + ")))";   //$NON-NLS-1$//$NON-NLS-2$//$NON-NLS-3$ //$NON-NLS-4$ //$NON-NLS-5$
			switch (indexType) {
				case 1:
					// Storage space must be a Hashing or a Multihashing index space
					String storageSpaceQuery = "RESTRICT(UNION(" + RELVARNAMES.HASHINGINDEXSPACE + ",PROJECT(" + RELVARNAMES.MULTIHASHINGINDEXSPACE + ",(" + ATTRIBUTENAMES.FILENAME + "," + ATTRIBUTENAMES.STORAGESPACEID + ")))," + restrictCondition + ")"; //$NON-NLS-1$//$NON-NLS-2$//$NON-NLS-3$ //$NON-NLS-4$ //$NON-NLS-5$ //$NON-NLS-6$
					checkStorageSpace(startTransaction, wv_indexType, storageSpaceQuery);
					break;
				case 2:
					// Storage space must be a Treeindex space
					storageSpaceQuery = "RESTRICT(" + RELVARNAMES.TREEINDEXSPACE + "," + restrictCondition + ")"; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
					checkStorageSpace(startTransaction, wv_indexType, storageSpaceQuery);
					break;
				default:
					// TupleIndex ; storagespace must just be an index space (this will be verified by the database constraints)
					// TupleIndex tuple must be added
					cmds.addLast(index.dbAddCommand(RELVARNAMES.TUPLEINDEX));
			}

			GenericStringValue wv_indexComponentExpressions = (GenericStringValue) getPresentationValue("INDEXCOMPONENTS"); //$NON-NLS-1$
			if (wv_indexComponentExpressions != null) {
				int ordinal = 10;
				String indexComponentExpressions = wv_indexComponentExpressions.getString().trim();
				while (indexComponentExpressions.length() > 0) {
					int cp = indexComponentExpressions.indexOf(',');
					int bp = indexComponentExpressions.indexOf('(');
					String indexComponentExpression;
					if (bp < 0 || bp > cp) {
						// No brackets ; only possibility is that there's a comma somewhere, making it a list of attribute references
						// Brackets after no comma at all ; it's a single expression involving some ()
						// Brackets after a comma ; the first thing is an attribute reference, and an expression involving () follows somewhere
						if (cp < 0) {
							indexComponentExpression = indexComponentExpressions;
							indexComponentExpressions = ""; //$NON-NLS-1$
						} else {
							indexComponentExpression = indexComponentExpressions.substring(0, cp);
							indexComponentExpressions = indexComponentExpressions.substring(cp + 1);
						}
					} else {
						// bp >= 0 && bp < cp : first part is an expression involving () ; get that part regardless of where the first comma is
						try {
							NameValueResult nvr = BracketParser.getNameAndMandatoryValueFromNonEscaped(indexComponentExpressions, 0);
							indexComponentExpression = nvr.getNameValuePair().getValue();
							indexComponentExpressions = indexComponentExpressions.substring(nvr.getNextParsePos());
						} catch (NoOpeningBracketException e2) {
							throw new ExpectedBracketsMissingException(indexComponentExpressions, getUserLocale());
						} catch (NoClosingBracketException e2) {
							throw new ExpectedBracketsMissingException(indexComponentExpressions, getUserLocale());
						}
					}
					cmds.addLast(new IndexComponent(relvarName, indexName, ordinal, indexComponentExpression).dbAddCommand(RELVARNAMES.INDEXCOMPONENT));
					ordinal += 10;
				}
			}

			startTransaction.execMultipleStatementAndEndTransaction(cmds);
		} catch (ConnectionClosedException e) {
			log(e);
			throw new DBProblem(e, getUserLocale());
		} catch (ErrorMessageException e) {
			log(e);
			throw getWAARApplicationException(e);
		} finally {
			// Close the connection in case any WaarApplicationException gets thrown
			dbc.close();
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.fgov.kszbcss.WAAR.OnLinePresentationFunction#doResumeLogic()
	 */
	public void doResumeLogic ( ) {

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.fgov.kszbcss.WAAR.OnLinePresentationFunction#getInitialPresentationName()
	 */
	public String getInitialPresentationName ( ) {
		return null;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.fgov.kszbcss.WAAR.OnLinePresentationFunction#getPresentationFunctionDescription()
	 */
	public String getPresentationFunctionDescription (String language) {
		return MyMessageFormat.format(I18N.getString(getUserLocale(), "IndexCreate.Text"), new String[] {}); //$NON-NLS-1$
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.fgov.kszbcss.WAAR.OnLinePresentationFunction#leaveFunctionOnExit()
	 */
	public boolean leaveFunctionOnExit ( ) {
		return true;
	}
}