/*
 * Created on 30-sep-2008
 */
package be.SIRAPRISE.webclient;

import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.Locale;
import java.util.Properties;
import java.util.Set;
import java.util.Map.Entry;
import java.util.logging.Level;
import java.util.logging.Logger;

import be.SIRAPRISE.client.*;
import be.WAAR.PresentationLayer.*;
import be.SIRAPRISE.messages.*;
import be.SIRAPRISE.util.BracketParser;
import be.SIRAPRISE.util.IntersectableMap;
import be.SIRAPRISE.util.InvalidEscapedCharacterException;
import be.SIRAPRISE.util.MissingEscapedCharacterException;
import be.SIRAPRISE.util.WaarListFactory;

/**
 * Class that can be used to define methods that can commonly be used by all SIRA_PRISE web client functions
 * 
 * @author Erwin Smout
 */
abstract class SiraPriseWebClientFunction extends OnLinePresentationFunction {

	/**
	 * 
	 */
	static final String SIRAPRISESERVER = "SIRAPRISESERVER"; //$NON-NLS-1$

	/**
	 * 
	 */
	private static final String SIRAPRISEWEBCLIENT_PROPERTIES = "siraprisewebclient.properties"; //$NON-NLS-1$

	/**
	 * 
	 */
	private static final String SIGNERCLASSNAME = "SIGNERCLASSNAME"; //$NON-NLS-1$

	/**
	 * The Signer that will be used when communicating with a SIRA_PRISE server
	 */
	private static Signer siraPriseWebClientSigner;

	/**
	 * The webclient's configured client name for obtaining database connections
	 */
	final static String clientName = "SIRAPRISEWEBCLIENT"; //$NON-NLS-1$

	/**
	 * The set of signing algorithm names usable by the client
	 */
	static Set<String> clientSigningAlgorithmNames;

	/**
	 * The set of usable Cipher names
	 */
	static Set<String> cryptoAlgorithmNames;

	/**
	 * 
	 */
	static final String SERVER = "SERVER"; //$NON-NLS-1$

	/**
	 * 
	 */
	static {
		Properties properties = new Properties(DBConnectionProperties.getInstance());
		InputStream ris = SiraPriseWebClientFunction.class.getResourceAsStream("/" + SIRAPRISEWEBCLIENT_PROPERTIES); //$NON-NLS-1$
		if (ris != null) {
			try {
				properties.load(ris);
			} catch (IOException e) {
				Logger.getLogger(SiraPriseWebClientFunction.class.getName()).log(Level.SEVERE, "Failed to obtain SIRA_PRISE web client properties.", e); //$NON-NLS-1$
				properties.clear();
			}
			try {
				ris.close();
			} catch (IOException e1) {
			}
		}

		String clientSigningAlgorithmNameList = properties.getProperty(DBConnectionProperties.SIGNINGALGORITHMS, ""); //$NON-NLS-1$
		clientSigningAlgorithmNames = DBConnectionProperties.getSigningAlgorithmNameSet(clientSigningAlgorithmNameList);

		String cryptoAlgorithmNameList = properties.getProperty(DBConnectionProperties.CRYPTOALGORITHMS, ""); //$NON-NLS-1$
		cryptoAlgorithmNames = DBConnectionProperties.getCryptoAlgorithmNameSet(cryptoAlgorithmNameList);

		String siraPriseWebClientSignerClassName = properties.getProperty(SIGNERCLASSNAME, SiraPriseWebClientSigner.class.getName());
		try {
			Class<?> siraPriseWebClientSignerClass = SiraPriseWebClientFunction.class.getClassLoader().loadClass(siraPriseWebClientSignerClassName);
			siraPriseWebClientSigner = (Signer) siraPriseWebClientSignerClass.newInstance();
		} catch (Exception e) {
			Logger.getLogger(SiraPriseWebClientFunction.class.getName()).log(Level.SEVERE, "Failed to load Signer.", e); //$NON-NLS-1$
			siraPriseWebClientSigner = SiraPriseWebClientSigner.getInstance();
		}

		Logger.getLogger(SiraPriseWebClientFunction.class.getName()).info("SIRA_PRISE web client employing Signer " + siraPriseWebClientSigner.getClass().getName()); //$NON-NLS-1$
		Logger.getLogger(SiraPriseWebClientFunction.class.getName()).info("SIRA_PRISE web client employing signatures set " + clientSigningAlgorithmNames.toString()); //$NON-NLS-1$
		Logger.getLogger(SiraPriseWebClientFunction.class.getName()).info("SIRA_PRISE web client employing ciphers set " + cryptoAlgorithmNames.toString()); //$NON-NLS-1$
	}

	/**
	 * @param siraPriseServer
	 * @return
	 * @throws NoConnectionException
	 */
	final DBConnection getDBConnection (SiraPriseServer siraPriseServer) throws NoConnectionException {
		return getDBConnection(siraPriseServer.getServer(), siraPriseServer.getPort(), getUserLocale());
	}

	/**
	 * Gets a DBConnection connected to the given host and port
	 * 
	 * @param server
	 *            the host where a sira_prise server is supposed to be listening
	 * @param port
	 *            The port where a sira_prise server is supposed to be listening
	 * @param locale
	 *            The locale in which to set exception message texts, if needed
	 * @return A Monitor connection, connected to a server
	 * @throws NoConnectionException
	 */
	final DBConnection getDBConnection (String server, int port, Locale locale) throws NoConnectionException {
		try {
			return new OneShotDBConnection(server, port, clientSigningAlgorithmNames, clientName, cryptoAlgorithmNames, siraPriseWebClientSigner);
		} catch (IOException e) {
			log(e);
			throw new NoConnectionException(e, locale);
		} catch (CommunicationProtocolException e) {
			log(e);
			throw new NoConnectionException(e, locale);
		} catch (DBException e) {
			log(e);
			throw new NoConnectionException(e, locale);
		}
	}

	/**
	 * Gets a MonitorConnection connected to the given host and port
	 * 
	 * @param server
	 *            the host where a sira_prise server monitor is supposed to be listening
	 * @param monitorPort
	 *            The port where a sira_prise server monitor is supposed to be listening
	 * @return A Monitor connection, connected to a server monitor port
	 * @throws NoMonitorConnectionException
	 */
	final MonitorConnection getMonitorConnection (String server, int monitorPort) throws NoMonitorConnectionException {
		try {
			return new MonitorConnection(server, monitorPort, clientSigningAlgorithmNames, clientName, siraPriseWebClientSigner);
		} catch (IOException e) {
			log(e);
			throw new NoMonitorConnectionException(getUserLocale());
		} catch (CommunicationProtocolException e) {
			log(e);
			throw new NoMonitorConnectionException(getUserLocale());
		} catch (DBException e) {
			log(e);
			throw new NoMonitorConnectionException(getUserLocale());
		}
	}

	/**
	 * Gets A SiraPriseServer object identifying the server that is being managed by this session
	 * 
	 * @return A SiraPriseServer object identifying the server that is being managed by this session
	 */
	final SiraPriseServer getSiraPriseServer ( ) {
		// return new SiraPriseServer(((GenericStringValue) getFromSessionContext(SERVER)), ((GenericIntegerValue) getFromSessionContext(DBConnectionProperties.PORT)), ((GenericIntegerValue) getFromSessionContext(MONITORPORT)));
		return (SiraPriseServer) getFromSessionContext(SIRAPRISESERVER);
	}

	/**
	 * Gets the applicable presentation layer exception to report the given kind of server error to the user
	 * 
	 * @param e
	 *            The original error message read by the client
	 * @return The applicable WAAR exception object
	 */
	final WAARApplicationException getWAARApplicationException (ErrorMessageException e) {
		ServerErrorMessage serverErrorMessage = e.getServerErrorMessage();
		switch (serverErrorMessage.getMessageTypeID()) {
			case ConstraintViolatedErrorMessageType.MESSAGETYPEID: {
				return new DBConstraintViolationException(serverErrorMessage, getUserLocale());
			}
			case NonSiraPriseErrorMessageType.MESSAGETYPEID: {
				return new DBServerRuntimeException(serverErrorMessage, getUserLocale());
			}
			case SiraPriseErrorMessageType.MESSAGETYPEID: {
				return new DBSiraPriseRuntimeException(serverErrorMessage, getUserLocale());
			}
			default: {

			}
		}
		return new DBProblem(e, getUserLocale());
	}

	/**
	 * @param listFieldName
	 *            The name of the list field on the presentation to be filled from the given relation
	 * @param r
	 *            The relation holding the tuples to create the list table rows from
	 * @param escapeProcessingForColumnNames
	 *            The set of column names for which SIRA_PRISE escape character removal is to be applied
	 * @return A ListValue holding the information contained in the relationValue argument
	 * @throws WaarException
	 */
	final ListValue getWAARListValue (String listFieldName, AbstractRelation r, Set<String> escapeProcessingForColumnNames) throws WaarException {
		Locale locale = getUserLocale();
		PresentationField listField = getPresentation().getPresentationField(listFieldName, locale);
		ListValue listValue = new ListValue(listFieldName);

		for (Tuple tuple : r) {
			HashMap<String, WaarValue> m = new HashMap<String, WaarValue>();
			IntersectableMap<?, ?> tupleAttributeValues = tuple.values();
			for (Entry<?, ?> me : tupleAttributeValues.entrySet()) {
				String attributeName = (String) me.getKey();
				String attributeValue = (String) me.getValue();
				// BracketParseResult attributeValueParseResult = BracketParser.getContentsWithinBrackets(attributeValue);
				PresentationField listColumnField = listField.getColumn(attributeName, locale);
				if (listColumnField != null) {
					WaarValue listAttributeValue;
					if (listColumnField.getType() instanceof List) {
						listAttributeValue = WaarListFactory.fromSiraPriseRelationValue(listColumnField, attributeValue, getUserData(), locale);
					} else {
						if (escapeProcessingForColumnNames != null && escapeProcessingForColumnNames.contains(attributeName)) {
							try {
								attributeValue = BracketParser.unMeta(attributeValue);
							} catch (InvalidEscapedCharacterException e) {
								throw new InvalidEscapedSyntaxReturnedException(attributeValue, locale);
							} catch (MissingEscapedCharacterException e) {
								throw new InvalidEscapedSyntaxReturnedException(attributeValue, locale);
							}
						}
						listAttributeValue = listColumnField.getType().encode(attributeValue, getUserData(), locale);
					}
					m.put(attributeName, listAttributeValue);
				}
			}
			listValue.append(m);
		}
		return listValue;
	}

	/**
	 * @return
	 */
	final boolean isDDLCapture ( ) {
		return getSiraPriseServer().isCaptureDDL();
	}

	/**
	 * Sets the server and monitorport values on the current presentation
	 * 
	 * @return A SiraPriseServer object identifying the server that is being managed by this session
	 * @throws WaarException
	 *             If the server and port values cannot be set
	 */
	final SiraPriseServer setSiraPriseServerMonitorPortPresentationValues ( ) throws WaarException {
		SiraPriseServer sirapriseServer = getSiraPriseServer();
		setPresentationValue(SERVER, sirapriseServer.getServer());
		setPresentationValue(DBConnectionProperties.MONITORPORT, sirapriseServer.getMonitorport());

		return sirapriseServer;
	}

	/**
	 * Sets the server and monitorport values on the current presentation from the given SiraPriseServer object
	 * 
	 * @param sirapriseServer
	 *            A SiraPriseServer object identifying the server that is being managed by this session
	 * @throws WaarException
	 *             If the server and port values cannot be set
	 */
	final void setSiraPriseServerMonitorPortPresentationValues (SiraPriseServer sirapriseServer) throws WaarException {
		setPresentationValue(SERVER, sirapriseServer.getServer());
		setPresentationValue(DBConnectionProperties.MONITORPORT, sirapriseServer.getMonitorport());
	}

	/**
	 * Sets the server and port values on the current presentation
	 * 
	 * @return A SiraPriseServer object identifying the server that is being managed by this session
	 * @throws WaarException
	 *             If the server and port values cannot be set
	 */
	final SiraPriseServer setSiraPriseServerPortPresentationValues ( ) throws WaarException {
		SiraPriseServer sirapriseServer = getSiraPriseServer();
		setPresentationValue(SERVER, sirapriseServer.getServer());
		setPresentationValue(DBConnectionProperties.PORT, sirapriseServer.getPort());

		return sirapriseServer;
	}

	/**
	 * Sets the server and port values on the current presentation from the given SiraPriseServer object
	 * 
	 * @param sirapriseServer
	 *            A SiraPriseServer object identifying the server that is being managed by this session
	 * @throws WaarException
	 *             If the server and port values cannot be set
	 */
	final void setSiraPriseServerPortPresentationValues (SiraPriseServer sirapriseServer) throws WaarException {
		setPresentationValue(SERVER, sirapriseServer.getServer());
		setPresentationValue(DBConnectionProperties.PORT, sirapriseServer.getPort());
	}
}