/*
 * Created on 7-aug-2008
 */
package be.SIRAPRISE.client;

import java.util.Iterator;
import java.util.Map;
import java.util.Map.Entry;

import be.SIRAPRISE.client.jsba.ClassDoesNotImplementDBObjectException;
import be.SIRAPRISE.client.jsba.DBObjectFactory;
import be.SIRAPRISE.util.IntersectableMap;

/**
 * AbstractTuple provides method implementations for the Tuple interface
 * 
 * @author Erwin Smout
 */
public abstract class AbstractTuple implements Tuple {

	/**
	 * the heading that the Tuple conforms to
	 */
	private Heading heading;

	/**
	 * Creates the Tuple object
	 * 
	 * @param heading
	 *            the heading that the Tuple conforms to
	 */
	public AbstractTuple (Heading heading) {
		this.heading = heading;
	}

	/**
	 * Gets The attribute value for the named attribute
	 * 
	 * @param attributeName
	 *            The name of the attribute whose value is to be obtained
	 * @return The attribute value including the SIRA_PRISE escape sequences
	 */
	public abstract String escapedValue (String attributeName);

	/**
	 * Gets the heading that the Tuple conforms to
	 * 
	 * @return the heading that the Tuple conforms to
	 */
	public final Heading getHeading ( ) {
		return heading;
	}

	/**
	 * Gets the tuple body value in textual format (i.e. attr1(value1)attr2(value2) with value1, value2, ... NOT naming the type (i.e. 1 instead of INT(1))
	 * 
	 * @return the tuple body value in textual format (i.e. attr1(value1)attr2(value2) with value1, value2, ... NOT naming the type (i.e. 1 instead of INT(1))
	 * @deprecated replaced by printBodyValueWithoutTypeNames()
	 */
	public final String printBodyValueWithoutPossrepNames ( ) {
		return printBodyValueWithoutTypeNames();
	}

	/**
	 * Gets the tuple body value in textual format (i.e. attr1(value1)attr2(value2) with value1, value2, ... NOT naming the type (i.e. 1 instead of INT(1))
	 * 
	 * @return the tuple body value in textual format (i.e. attr1(value1)attr2(value2) with value1, value2, ... NOT naming the type (i.e. 1 instead of INT(1))
	 */
	public final String printBodyValueWithoutTypeNames ( ) {
		StringBuilder out = new StringBuilder(256);
		Iterator<Entry<String, TypeDeclaration>> i_heading = getHeading().entrySet().iterator();
		while (i_heading.hasNext()) {
			String attributeName = i_heading.next().getKey();
			out.append(attributeName).append('(').append(value(attributeName)).append(')');
		}
		return out.toString();
	}

	/**
	 * Gets the tuple body value in textual format (i.e. attr1(value1)attr2(value2) with value1, value2, ... naming the type (i.e. INT(1) instead of 1)
	 * 
	 * @return the tuple body value in textual format (i.e. attr1(value1)attr2(value2) with value1, value2, ... naming the type (i.e. INT(1) instead of 1)
	 */
	public final String printBodyValueWithTypeNames ( ) {
		StringBuilder out = new StringBuilder(256);
		Iterator<Entry<String, TypeDeclaration>> i_heading = getHeading().entrySet().iterator();
		while (i_heading.hasNext()) {
			Entry<String, TypeDeclaration> me = i_heading.next();
			String attributeName = me.getKey();
			String typeName = me.getValue().getTypeName();
			out.append(attributeName).append('(').append(typeName).append('(').append(value(attributeName)).append(')').append(')');
		}
		return out.toString();
	}

	/**
	 * Gets the tuple body value in textual XML-style format (i.e. <attr1>value1</attr1> <attr2>value2</attr2>)
	 * 
	 * @return The tuple body value in textual XML-style format (i.e. <attr1>value1</attr1> <attr2>value2</attr2>)
	 */
	public final String printBodyValueXML ( ) {
		StringBuilder out = new StringBuilder(256);
		Iterator<Entry<String, TypeDeclaration>> i_heading = getHeading().entrySet().iterator();
		while (i_heading.hasNext()) {
			Entry<String, TypeDeclaration> me = i_heading.next();
			String attributeName = me.getKey();
			out.append('<').append(attributeName).append('>').append(EntityRefs.toXML(value(attributeName))).append("</").append(attributeName).append('>'); //$NON-NLS-1$
		}
		return out.toString();
	}

	/**
	 * Gets the tuple body value in textual format (i.e. attr1(value1)attr2(value2), INCLUDING escape processing for the ()\ characters of the metalanguage that appear in a printed value
	 * 
	 * @return the tuple body value in textual format (i.e. attr1(value1)attr2(value2), INCLUDING escape processing for the ()\ characters of the metalanguage that appear in a printed value
	 * @deprecated replaced by printEscapedBodyValueWithoutTypeNames();
	 */
	public final String printEscapedBodyValueWithoutPossrepNames ( ) {
		return printEscapedBodyValueWithoutTypeNames();
	}

	/**
	 * Gets the tuple body value in textual format (i.e. attr1(value1)attr2(value2), INCLUDING escape processing for the ()\ characters of the metalanguage that appear in a printed value
	 * 
	 * @return the tuple body value in textual format (i.e. attr1(value1)attr2(value2), INCLUDING escape processing for the ()\ characters of the metalanguage that appear in a printed value
	 */
	public final String printEscapedBodyValueWithoutTypeNames ( ) {
		StringBuilder out = new StringBuilder(256);
		Iterator<Entry<String, TypeDeclaration>> i_heading = getHeading().entrySet().iterator();
		while (i_heading.hasNext()) {
			Entry<String, TypeDeclaration> me = i_heading.next();
			String attributeName = me.getKey();
			out.append(attributeName).append('(').append(escapedValue(attributeName)).append(')');
		}
		return out.toString();
	}

	/**
	 * Gets the tuple body value in textual format (i.e. attr1(value1)attr2(value2), INCLUDING escape processing for the ()\ characters of the metalanguage that appear in a printed value
	 * 
	 * @return the tuple body value in textual format (i.e. attr1(value1)attr2(value2), INCLUDING escape processing for the ()\ characters of the metalanguage that appear in a printed value
	 */
	public final String printEscapedBodyValueWithTypeNames ( ) {
		StringBuilder out = new StringBuilder(256);
		Iterator<Entry<String, TypeDeclaration>> i_heading = getHeading().entrySet().iterator();
		while (i_heading.hasNext()) {
			Entry<String, TypeDeclaration> me = i_heading.next();
			String attributeName = me.getKey();
			String typeName = me.getValue().getTypeName();
			out.append(attributeName).append('(').append(typeName).append('(').append(escapedValue(attributeName)).append(')').append(')');
		}
		return out.toString();
	}

	/**
	 * Gets the tuple value in textual format (i.e. TUPLE(attr1(value1)attr2(value2)) INCLUDING escape processing for the ()\ characters of the metalanguage
	 * 
	 * @return the tuple value in textual format (i.e. TUPLE(attr1(value1)attr2(value2)) INCLUDING escape processing for the ()\ characters of the metalanguage
	 * @deprecated replaced by printValueEscapedWithoutTypeNames()
	 */
	public final String printEscapedValueWithoutPossrepNames ( ) {
		return printValueEscapedWithoutTypeNames();
	}

	/**
	 * Gets the tuple value in textual format (i.e. TUPLE(attr1(value1)attr2(value2)) INCLUDING escape processing for the ()\ characters of the metalanguage
	 * 
	 * @return the tuple value in textual format (i.e. TUPLE(attr1(value1)attr2(value2)) INCLUDING escape processing for the ()\ characters of the metalanguage
	 */
	public final String printValueEscapedWithoutTypeNames ( ) {
		return new StringBuilder(256).append("TUPLE(").append(printEscapedBodyValueWithoutTypeNames()).append(')').toString(); //$NON-NLS-1$
	}

	/**
	 * Gets the tuple value in textual format (i.e. TUPLE(attr1(value1)attr2(value2)) INCLUDING escape processing for the ()\ characters of the metalanguage
	 * 
	 * @return the tuple value in textual format (i.e. TUPLE(attr1(value1)attr2(value2)) INCLUDING escape processing for the ()\ characters of the metalanguage
	 */
	public final String printValueEscapedWithTypeNames ( ) {
		return new StringBuilder(256).append("TUPLE(").append(printEscapedBodyValueWithTypeNames()).append(')').toString(); //$NON-NLS-1$
	}

	/**
	 * Gets the tuple value in textual format (i.e. TUPLE(attr1(value1)attr2(value2)) with value1, value2, ... NOT naming the type (i.e. 1 instead of INT(1))
	 * 
	 * @return the tuple value in textual format (i.e. TUPLE(attr1(value1)attr2(value2)) with value1, value2, ... NOT naming the type (i.e. 1 instead of INT(1))
	 * @deprecated replaced by printValueWithoutTypeNames()
	 */
	public final String printValueWithoutPossrepNames ( ) {
		return printValueWithoutTypeNames();
	}

	/**
	 * Gets the tuple value in textual format (i.e. TUPLE(attr1(value1)attr2(value2)) with value1, value2, ... NOT naming the type (i.e. 1 instead of INT(1))
	 * 
	 * @return the tuple value in textual format (i.e. TUPLE(attr1(value1)attr2(value2)) with value1, value2, ... NOT naming the type (i.e. 1 instead of INT(1))
	 */
	public final String printValueWithoutTypeNames ( ) {
		return new StringBuilder(256).append("TUPLE(").append(printBodyValueWithoutTypeNames()).append(')').toString(); //$NON-NLS-1$
	}

	/**
	 * Gets the tuple value in textual format (i.e. TUPLE(attr1(value1)attr2(value2)) with value1, value2, ... NOT naming the type (i.e. 1 instead of INT(1))
	 * 
	 * @return the tuple value in textual format (i.e. TUPLE(attr1(value1)attr2(value2)) with value1, value2, ... NOT naming the type (i.e. 1 instead of INT(1))
	 */
	public final String printValueWithTypeNames ( ) {
		return new StringBuilder(256).append("TUPLE(").append(printBodyValueWithTypeNames()).append(')').toString(); //$NON-NLS-1$
	}

	/**
	 * Gets the tuple value in textual XML-style format (i.e. <Tuple><attr1>value1 </attr1> <attr2>value2 </attr2> </Tuple>
	 * 
	 * @return The tuple value in textual XML-style format (i.e. <Tuple><attr1>value1 </attr1> <attr2>value2 </attr2> </Tuple>
	 */
	public final String printValueXML ( ) {
		return new StringBuilder(256).append("<Tuple>").append(printBodyValueXML()).append("</Tuple>").toString(); //$NON-NLS-1$ //$NON-NLS-2$
	}

	/**
	 * Get an objects holding the same information as that which is held in this tuple.
	 * 
	 * @param <C>
	 * 
	 * @param objectClass
	 *            A Class object denoting the class of the objects to be returned. objectClass must denote a public, non-abstract class that has a public no-arg constructor, and that has a unique setter method for each attribute in the heading. The name of this unique method must be the concatenation of the word "set" (all lowercase), the attribute name (capitalized), and the suffix "FromDB". The argument list of this setter method must consist of excatly one java.lang.String argument.
	 * @return An array of objects of the given object Class.
	 * @throws SettersMissingException
	 *             if a needed setter is missing in the class denoted by objectClass
	 * @throws ConstructorMissingException
	 *             if objectClass does not denote a public class, or it denotes an abstract class, or the class does not have a public no-arg constructor
	 * @throws ClassDoesNotImplementDBObjectException
	 *             if objectClass does not implement the required DBObject interface
	 * @deprecated - replace by be.SIRAPRISE.client.jsba.DBObjectFactory.getObject(...) - e.g. by inlining the calls to this method
	 */
	public final <C> C toObject (Class<C> objectClass) throws SettersMissingException, ConstructorMissingException, ClassDoesNotImplementDBObjectException {
		return DBObjectFactory.getObject(this, objectClass);
	}

	/**
	 * Get a Map of objects holding the same information as that which is held in this Tuple. The entries in the map have one of the objectClass[] objects as their key, and an object of that class as their value. The object of that class holds the attribute values of the tuple for which a corresponding setter method was found in the class.
	 * 
	 * @param objectClass
	 *            An array of Class objects denoting the classes of the objects to be returned for each tuple. Each objectClass must denote a public, non-abstract class that has a public no-arg constructor, and that has a unique setter method for some attribute in the heading. The name of this unique method must be the concatenation of the word "set" (all lowercase), the attribute name (capitalized), and the suffix "FromDB". The argument list of this setter method must consist of excatly one java.lang.String argument.
	 * @return A map of objects holding the same information as this tuple. The key in each map entry is one of the objectClass objects given as an argument, the corresponding value in the map entry is the created object of that class.
	 * @throws SettersMissingException
	 *             if some attribute in the heading exists for which no corresponding setter could be found in any of the classes denoted by objectClass[]
	 * @throws ConstructorMissingException
	 *             if objectClass[] contains any class that does not denote a public class, or denotes an abstract class, or does not have a public no-arg constructor
	 * @throws ClassDoesNotImplementDBObjectException
	 *             if objectClass[] contains any class that does not implement the required DBObject interface
	 * @deprecated - replace by be.SIRAPRISE.client.jsba.DBObjectFactory.getObjects(...) - e.g. by inlining the calls to this method
	 */
	public final Map<Class<?>, Object> toObjects (Class<?>[] objectClass) throws SettersMissingException, ConstructorMissingException, ClassDoesNotImplementDBObjectException {
		return DBObjectFactory.getObjects(this, objectClass);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.lang.Object#toString()
	 */
	public String toString ( ) {
		return values().toString();
	}

	public abstract String value (String attributeName);

	public abstract IntersectableMap<?, ?> values ( );
}