/*
 * Created on 14-sep-2007
 */
package be.SIRAPRISE.client;

import java.util.*;

import be.SIRAPRISE.client.NAMES.POSSREPCOMPONENTNAMES;
import be.SIRAPRISE.client.NAMES.TYPENAMES;
import be.SIRAPRISE.client.jsba.ClassDoesNotImplementDBObjectException;
import be.SIRAPRISE.client.jsba.DBObjectFactory;
import be.SIRAPRISE.client.jsba.SetterMethod;
import be.erwinsmout.MyMessageFormat;

/**
 * Abstract implementation of the TupleContainer interface
 * 
 * @author Erwin Smout
 */
public abstract class AbstractTupleContainer implements TupleContainer {

	/**
	 * The heading for tuple container.
	 */
	private Heading heading;

	/**
	 * Creates the container, setting the intended heading for the tuples
	 * 
	 * @param heading
	 *            The heading for tuple container.
	 */
	public AbstractTupleContainer (Heading heading) {
		this.heading = heading;
	}

	/**
	 * Adds all tuples of the given TupleContainer to this one.
	 * 
	 * @param tupleContainer
	 *            The TupleContainer whose tuples are all to be added to this one
	 */
	public final void addTuplesWithHeadingCheck (TupleContainer tupleContainer) {
		for (Tuple tuple : tupleContainer) {
			addTupleWithHeadingCheck(tuple);
		}
	}

	/**
	 * Adds all tuples of the given TupleContainer to this one.
	 * 
	 * @param tupleContainer
	 *            The TupleContainer whose tuples are all to be added to this one
	 */
	public final void addTuplesWithoutHeadingCheck (TupleContainer tupleContainer) {
		for (Tuple tuple : tupleContainer) {
			addTupleWithoutHeadingCheck(tuple);
		}
	}

	/**
	 * Adds a Tuple to the container.
	 * 
	 * @param tuple
	 *            The tuple to be added to the container
	 */
	public final void addTupleWithHeadingCheck (Tuple tuple) {
		if (tuple.getHeading().keySet().equals(this.getHeading().keySet())) {
			addTupleWithoutHeadingCheck(tuple);
		} else {
			throw new IllegalArgumentException(MyMessageFormat.format(Messages.getString("AbstractRelation.HeadingMismatch"), new Object[] { tuple, getHeading().keySet() })); //$NON-NLS-1$
		}
	}

	/**
	 * Adds a Tuple to the container without checking the conformance of the tuple to the declared heading. The conformance should be guaranteed externally to this method.
	 * 
	 * @param tuple
	 *            The tuple to be added to the container
	 */
	public abstract void addTupleWithoutHeadingCheck (Tuple tuple);

	/**
	 * Gets The heading for tuple container.
	 * 
	 * @return The heading for tuple container.
	 */
	public final Heading getHeading ( ) {
		return heading;
	}

	/**
	 * Gets an iterator over the tuples in the container
	 * 
	 * @return an iterator over the tuples in the container
	 */
	public abstract Iterator<Tuple> iterator ( );

	/**
	 * Gets the relation body in text
	 * 
	 * @return The relation body in text
	 */
	public final String printBodyEscapedWithBodyHeaderWithoutTypeNames ( ) {
		StringBuilder sb = new StringBuilder(256 * size());
		sb.append(POSSREPCOMPONENTNAMES.BODY).append("("); //$NON-NLS-1$
		Iterator<Tuple> i_tuples = iterator();
		while (i_tuples.hasNext()) {
			Tuple tuple = i_tuples.next();
			sb.append(tuple.printValueEscapedWithoutTypeNames());
		}
		return sb.append(')').toString();
	}

	/**
	 * Gets the relation body in text
	 * 
	 * @return The relation body in text
	 */
	public final String printBodyEscapedWithBodyHeaderWithTypeNames ( ) {
		StringBuilder sb = new StringBuilder(256 * size());
		sb.append(POSSREPCOMPONENTNAMES.BODY).append("("); //$NON-NLS-1$
		Iterator<Tuple> i_tuples = iterator();
		while (i_tuples.hasNext()) {
			Tuple tuple = i_tuples.next();
			sb.append(tuple.printValueEscapedWithTypeNames());
		}
		return sb.append(')').toString();
	}

	/**
	 * Gets the relation body in text EXCLUDING the BODY( ) container
	 * 
	 * @return The relation body in text EXCLUDING the BODY( ) container
	 */
	public final String printBodyEscapedWithoutBodyHeaderWithoutTypeNames ( ) {
		StringBuilder sb = new StringBuilder(256 * size());
		Iterator<Tuple> i_tuples = iterator();
		while (i_tuples.hasNext()) {
			Tuple tuple = i_tuples.next();
			sb.append(tuple.printValueEscapedWithoutTypeNames());
		}
		return sb.toString();
	}

	/**
	 * Gets the relation body in text EXCLUDING the BODY( ) container
	 * 
	 * @return The relation body in text EXCLUDING the BODY( ) container
	 */
	public final String printBodyEscapedWithoutBodyHeaderWithTypeNames ( ) {
		StringBuilder sb = new StringBuilder(256 * size());
		Iterator<Tuple> i_tuples = iterator();
		while (i_tuples.hasNext()) {
			Tuple tuple = i_tuples.next();
			sb.append(tuple.printValueEscapedWithTypeNames());
		}
		return sb.toString();
	}

	/**
	 * Gets a relation body in text
	 * 
	 * @return The relation body in text
	 */
	public final String printBodyWithBodyHeaderWithoutTypeNames ( ) {
		StringBuilder sb = new StringBuilder(256 * size());
		sb.append(POSSREPCOMPONENTNAMES.BODY).append("("); //$NON-NLS-1$
		Iterator<Tuple> i_tuples = iterator();
		while (i_tuples.hasNext()) {
			Tuple tuple = i_tuples.next();
			sb.append(tuple.printValueWithoutTypeNames());
		}
		return sb.append(')').toString();
	}

	/**
	 * Gets a relation body in text
	 * 
	 * @return The relation body in text
	 */
	public final String printBodyWithBodyHeaderWithTypeNames ( ) {
		StringBuilder sb = new StringBuilder(256 * size());
		sb.append(POSSREPCOMPONENTNAMES.BODY).append("("); //$NON-NLS-1$
		Iterator<Tuple> i_tuples = iterator();
		while (i_tuples.hasNext()) {
			Tuple tuple = i_tuples.next();
			sb.append(tuple.printValueWithTypeNames());
		}
		return sb.append(')').toString();
	}

	/**
	 * Gets the relation body in text EXCLUDING the BODY( ) container
	 * 
	 * @return The relation body in text EXCLUDING the BODY( ) container
	 */
	public final String printBodyWithoutBodyHeaderWithoutTypeNames ( ) {
		StringBuilder sb = new StringBuilder(256 * size());
		Iterator<Tuple> i_tuples = iterator();
		while (i_tuples.hasNext()) {
			Tuple tuple = i_tuples.next();
			sb.append(tuple.printValueWithoutTypeNames());
		}
		return sb.toString();
	}

	/**
	 * Gets the relation body in text EXCLUDING the BODY( ) container
	 * 
	 * @return The relation body in text EXCLUDING the BODY( ) container
	 */
	public final String printBodyWithoutBodyHeaderWithTypeNames ( ) {
		StringBuilder sb = new StringBuilder(256 * size());
		Iterator<Tuple> i_tuples = iterator();
		while (i_tuples.hasNext()) {
			Tuple tuple = i_tuples.next();
			sb.append(tuple.printValueWithTypeNames());
		}
		return sb.toString();
	}

	/**
	 * Gets the relation body in XML-style text
	 * 
	 * @return the relation body in XML-style text
	 */
	public final String printBodyXMLWithBodyHeader ( ) {
		StringBuilder sb = new StringBuilder(256 * size());
		sb.append("<Body>"); //$NON-NLS-1$
		Iterator<Tuple> i_tuples = iterator();
		while (i_tuples.hasNext()) {
			Tuple tuple = i_tuples.next();
			sb.append(tuple.printValueXML());
		}
		return sb.append("</Body>").toString(); //$NON-NLS-1$
	}

	/**
	 * Gets the relation body in XML-style text EXCLUDING the <Body>container
	 * 
	 * @return the relation body in XML-style text EXCLUDING the <Body>container
	 */
	public final String printBodyXMLWithoutBodyHeader ( ) {
		StringBuilder sb = new StringBuilder(256 * size());
		Iterator<Tuple> i_tuples = iterator();
		while (i_tuples.hasNext()) {
			Tuple tuple = i_tuples.next();
			sb.append(tuple.printValueXML());
		}
		return sb.toString();
	}

	public final String printValueEscapedWithoutTypeNames ( ) {
		StringBuilder sb = new StringBuilder(256 * size());
		sb.append(TYPENAMES.RELATION).append("(").append(heading.print()).append(POSSREPCOMPONENTNAMES.BODY).append("("); //$NON-NLS-1$ //$NON-NLS-2$
		Iterator<Tuple> i_tuples = iterator();
		while (i_tuples.hasNext()) {
			Tuple tuple = i_tuples.next();
			sb.append(tuple.printValueEscapedWithoutTypeNames());
		}
		sb.append("))"); //$NON-NLS-1$

		return sb.toString();
	}

	public final String printValueEscapedWithTypeNames ( ) {
		StringBuilder sb = new StringBuilder(256 * size());
		sb.append(TYPENAMES.RELATION).append("(").append(heading.print()).append(POSSREPCOMPONENTNAMES.BODY).append("("); //$NON-NLS-1$ //$NON-NLS-2$
		Iterator<Tuple> i_tuples = iterator();
		while (i_tuples.hasNext()) {
			Tuple tuple = i_tuples.next();
			sb.append(tuple.printValueEscapedWithTypeNames());
		}
		sb.append("))"); //$NON-NLS-1$

		return sb.toString();
	}

	public final String printValueWithoutTypeNames ( ) {
		StringBuilder sb = new StringBuilder(256 * size());
		sb.append(TYPENAMES.RELATION + "(").append(heading.print()).append(POSSREPCOMPONENTNAMES.BODY + "("); //$NON-NLS-1$ //$NON-NLS-2$
		Iterator<Tuple> i_tuples = iterator();
		while (i_tuples.hasNext()) {
			Tuple tuple = i_tuples.next();
			sb.append(tuple.printValueWithoutTypeNames());
		}
		sb.append("))"); //$NON-NLS-1$

		return sb.toString();
	}

	public final String printValueWithTypeNames ( ) {
		StringBuilder sb = new StringBuilder(256 * size());
		sb.append(TYPENAMES.RELATION).append("(").append(heading.print()).append(POSSREPCOMPONENTNAMES.BODY).append("("); //$NON-NLS-1$ //$NON-NLS-2$
		Iterator<Tuple> i_tuples = iterator();
		while (i_tuples.hasNext()) {
			Tuple tuple = i_tuples.next();
			sb.append(tuple.printValueWithTypeNames());
		}
		sb.append("))"); //$NON-NLS-1$

		return sb.toString();
	}

	public final String printValueXML ( ) {
		StringBuilder sb = new StringBuilder(256 * size());
		sb.append("<Relation>").append(heading.printXML()).append("<Body>"); //$NON-NLS-1$ //$NON-NLS-2$
		Iterator<Tuple> i_tuples = iterator();
		while (i_tuples.hasNext()) {
			Tuple tuple = i_tuples.next();
			sb.append(tuple.printValueXML());
		}
		sb.append("</Body></Relation>"); //$NON-NLS-1$

		return sb.toString();
	}

	public abstract boolean removeTuple (Tuple t);

	public final int removeTuples (TupleContainer tupleContainer) {
		int i = 0;
		for (Tuple tuple : tupleContainer) {
			if (removeTuple(tuple)) {
				i++;
			}
		}
		return i;
	}

	public abstract int size ( );

	public final <C> C[] toObjectArray (Class<C> objectClass) throws SettersMissingException, ConstructorMissingException, ClassDoesNotImplementDBObjectException {
		return DBObjectFactory.getObjectArray(this, objectClass);
	}

	public final <C> C[] toObjectArrayIgnoringMissingSetters (Class<C> objectClass) throws ConstructorMissingException, ClassDoesNotImplementDBObjectException {
		return DBObjectFactory.getObjectArrayIgnoringMissingSetters(this, objectClass);
	}

	public final <C> Collection<C> toObjectCollection (Class<C> objectClass) throws SettersMissingException, ConstructorMissingException, ClassDoesNotImplementDBObjectException {
		return DBObjectFactory.getObjectCollection(this, objectClass);
	}

	public final <C> Collection<C> toObjectCollectionIgnoringMissingSetters (Class<C> objectClass) throws ConstructorMissingException, ClassDoesNotImplementDBObjectException {
		DBObjectFactory.getPublicNoArgConstructor(objectClass);

		HashMap<String, SetterMethod> setterMethods = DBObjectFactory.getSetterMethods(objectClass, heading);

		Collection<C> objects = new ArrayList<C>();
		for (Tuple tuple : this) {
			objects.add(DBObjectFactory.getObject(tuple, objectClass, setterMethods));
		}

		return objects;
	}

	@SuppressWarnings("unchecked")
	public final Map<Class<?>, Object>[] toObjectsArray (Class<?>[] objectClass) throws SettersMissingException, ConstructorMissingException, ClassDoesNotImplementDBObjectException {
		return (Map<Class<?>, Object>[]) toObjectsCollection(objectClass).toArray();
	}

	public final Collection<Map<Class<?>, Object>> toObjectsCollection (Class<?>[] objectClass) throws SettersMissingException, ConstructorMissingException, ClassDoesNotImplementDBObjectException {
		Set<String> attributeNames = this.getHeading().keySet();
		HashSet<String> unmatchedAttributeNames = new HashSet<String>();
		unmatchedAttributeNames.addAll(attributeNames);
		Map<Class<?>, HashMap<String, SetterMethod>> processingMap = new HashMap<Class<?>, HashMap<String, SetterMethod>>();
		// Each objectClass must have a public no-arg constructor
		for (int i = 0; i < objectClass.length; i++) {
			DBObjectFactory.getPublicNoArgConstructor(objectClass[i]);
			HashMap<String, SetterMethod> setterMethods = DBObjectFactory.getSetterMethods(objectClass[i], heading);
			processingMap.put(objectClass[i], setterMethods);
			unmatchedAttributeNames.removeAll(processingMap.keySet());
		}

		if (unmatchedAttributeNames.size() > 0) {
			throw new SettersMissingException(unmatchedAttributeNames.toString());
		}

		Collection<Map<Class<?>, Object>> maps = new ArrayList<Map<Class<?>, Object>>();
		for (Tuple tuple : this) {
			maps.add(DBObjectFactory.toObjects(tuple, processingMap));
		}

		return maps;
	}

	public final String toString ( ) {
		return printValueWithoutTypeNames();
	}

}