package be.SIRAPRISE.client;

import java.net.*;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.security.Signature;
import java.security.SignatureException;
import java.text.MessageFormat;
import java.util.*;
import java.io.*;

import javax.crypto.Cipher;
import javax.crypto.NoSuchPaddingException;

import be.SIRAPRISE.messages.AuthenticationOKMessage;
import be.SIRAPRISE.messages.AuthenticationOKMessageType;
import be.SIRAPRISE.messages.ClientAuthenticationMessage;
import be.SIRAPRISE.messages.ClientAuthenticationMessageType;
import be.SIRAPRISE.messages.ClientHelloMessage;
import be.SIRAPRISE.messages.ClientHelloMessageType;
import be.SIRAPRISE.messages.CommitRollbackMessage;
import be.SIRAPRISE.messages.CommitRollbackMessageType;
import be.SIRAPRISE.messages.CommittedMessage;
import be.SIRAPRISE.messages.CommittedMessageType;
import be.SIRAPRISE.messages.DmlExecutedMessage;
import be.SIRAPRISE.messages.DmlExecutedMessageType;
import be.SIRAPRISE.messages.EndTransactionMessageType;
import be.SIRAPRISE.messages.ExecuteDMLMessageType;
import be.SIRAPRISE.messages.ExecuteDMLMessageTypesV14;
import be.SIRAPRISE.messages.ServerHelloMessage;
import be.SIRAPRISE.messages.ServerHelloMessageType;
import be.SIRAPRISE.messages.ServerMessage;
import be.SIRAPRISE.messages.ServerMessageTypes;
import be.SIRAPRISE.messages.StartSubTransactionMessageType;
import be.SIRAPRISE.messages.StartTransactionMessageType;
import be.SIRAPRISE.messages.TransactionEndedMessage;
import be.SIRAPRISE.messages.TransactionEndedMessageType;
import be.SIRAPRISE.messages.TransactionStartedMessage;
import be.SIRAPRISE.messages.TransactionStartedMessageType;
import be.SIRAPRISE.security.ProprietaryOrJCECipher;
import be.SIRAPRISE.util.MyDataOutputStream;
import be.erwinsmout.MyMessageFormat;
import be.erwinsmout.NotFoundException;

/**
 * Class used to communicate between a program and the sira_prise dbms. When DBConnections are created, the following information must be provided :
 * <table border="1">
 * <tr>
 * <th>parameter</th>
 * <th>Description.</th>
 * </tr>
 * <tr>
 * <td>host</td>
 * <td>The DNS identification of the host to connect to. It may either be its DNS name or its IP address in dotted decimal.</td>
 * </tr>
 * <tr>
 * <td>port</td>
 * <td>The IP port to connect to.</td>
 * </tr>
 * <tr>
 * <td>clientID</td>
 * <td>The identification of the client trying to connect.</td>
 * </tr>
 * <tr>
 * <td>signatureAlgorithmNames</td>
 * <td>A set of Strings denoting Signature names that the client is willing and able to use in signing.</td>
 * </tr>
 * <tr>
 * <td>initialAutoCommitForTransactions</td>
 * <td>A boolean flag indicating the default autoCommit setting that will be given to each transaction started on this connection.</td>
 * </tr>
 * <tr>
 * <td>requestConnectionIdleTime</td>
 * <td>The idle time in milliseconds that the client requests for this connection.</td>
 * </tr>
 * <tr>
 * <td>encryptionAlgorithmNames</td>
 * <td>A set of Strings denoting names of Cipher algorithms that the client is willing and able to use.</td>
 * </tr>
 * <tr>
 * <td>signer</td>
 * <td>A callback object that will be called upon to compute signatures if and when message signing is used.</td>
 * </tr>
 * </table>
 * <p>
 * Constructors exist in various combinations, possibly leaving out some of the items mentioned. In such cases, default values are used. These default values are configurable for the client in the DBCONNECTION.PROPERTIES file (see the DBConnectionProperties class for more information on where the client package searches for this file). The following table mentions the property name holding this default value for each optional field, and the value the client package will use in the absence of such a property.
 * </P>
 * <table border="1">
 * <tr>
 * <th>Field</th>
 * <th>Property</th>
 * <th>Default if property absent</th>
 * </tr>
 * <tr>
 * <td>signatureAlgorithmNames</td>
 * <td>DEFAULTSIGNATUREALGORITHMNAMES (property value must be a comma-separated list of algorithm names)</td>
 * <td>SPS, MD5withRSA, MD2withRSA, SHA1withRSA, SHA1withDSA</td>
 * </tr>
 * <tr>
 * <td>initialAutoCommitForTransactions</td>
 * <td>DEFAULTAUTOCOMMITFORTRANSACTIONS</td>
 * <td>False</td>
 * </tr>
 * <tr>
 * <td>transationMode</td>
 * <td>TRANSACTIONSREADONLY (property value is a boolean ; required value for backward compatibility is 'False')</td>
 * <td>False</td>
 * </tr>
 * <tr>
 * <td>requestConnectionIdleTime</td>
 * <td>DEFAULTCONNECTIONIDLETIME</td>
 * <td>0</td>
 * </tr>
 * <tr>
 * <td>encryptionAlgorithmNames</td>
 * <td>DEFAULTENCRYPTIONALGORITHMNAMES (property value must be a comma-separated list of Cipher algorithm names</td>
 * <td>SPE</td>
 * </tr>
 * </table>
 * <BR>
 * </BR> Though this may look complicated, it only means that the constructors use the following sources of information, in the order given :
 * <OL>
 * <LI>The info explicitly provided by the constructor</LI>
 * <LI>The default configured for this client in DBCONNECTION.PROPERTIES</LI>
 * <LI>The default defined by the client package itself</LI>
 * </OL>
 */
public abstract class DBConnection {

	/**
	 * The idle time in milleseconds the client is to request from the server for new connections
	 */
	private static long defaultConnectionIdleTime;

	/**
	 * The configured default setting for the DDL capture flag when starting transactions, for DBConnection objects
	 */
	private static DDLCapture defaultDDLCapture;

	/**
	 * The default initial autocommit setting for transactions, as configured for the client
	 */
	private static boolean defaultInitialAutoCommitForTransactions = false;

	/**
	 * The set of supported encryption algorithm names
	 */
	private static Set<String> defaultSupportedEncryptionAlgorithmNames = new HashSet<String>();

	/**
	 * The set of supported signature algorithm names
	 */
	private static Set<String> defaultSupportedSignatureAlgorithmNames = new HashSet<String>();

	/**
	 * the default transaction mode for transactions created on this connection
	 */
	private static TransactionMode defaultTransactionMode;

	static {
		// Set the default connection idle time to be requested
		Properties dbConnectionProperties = DBConnectionProperties.getInstance();
		String defaultConnectionIdleTime_tx = dbConnectionProperties.getProperty(DBConnectionProperties.DEFAULTCONNECTIONIDLETIME);
		if (defaultConnectionIdleTime_tx != null) {
			// set the value
			defaultConnectionIdleTime = Long.parseLong(defaultConnectionIdleTime_tx);
		}

		// Set the default initial autocommit for transactions
		String defaultInitialAutoCommitForTransactions_tx = dbConnectionProperties.getProperty(DBConnectionProperties.DEFAULTAUTOCOMMITFORTRANSACTIONS, "FALSE"); //$NON-NLS-1$
		// if (defaultInitialAutoCommitForTransactions_tx != null) {
		// set the value
		defaultInitialAutoCommitForTransactions = Boolean.valueOf(defaultInitialAutoCommitForTransactions_tx).booleanValue();
		// }

		// Set the default read-only mode for transactions
		String defaultTransactionMode_tx = dbConnectionProperties.getProperty(DBConnectionProperties.TRANSACTIONSREADONLY, "FALSE"); //$NON-NLS-1$
		// if (defaultInitialAutoCommitForTransactions_tx != null) {
		// set the value
		defaultTransactionMode = TransactionMode.getTransactionMode(Boolean.valueOf(defaultTransactionMode_tx).booleanValue());
		// }

		// Set the default ddlcapture mode for transactions
		String defaultDDLCapture_tx = dbConnectionProperties.getProperty(DBConnectionProperties.DDLCAPTURE, "FALSE"); //$NON-NLS-1$
		// if (defaultInitialAutoCommitForTransactions_tx != null) {
		// set the value
		defaultDDLCapture = DDLCapture.getDDLCapture(Boolean.valueOf(defaultDDLCapture_tx).booleanValue());
		// }

		// See if the DBConnectionProperties hold a value for SIGNINGALGORITHMNAMES
		String defaultSigningAlgorithmNameList = dbConnectionProperties.getProperty(DBConnectionProperties.DEFAULTSIGNATUREALGORITHMNAMES, DBConnectionProperties.DEFAULTSIGNATUREALGORITHMNAMESDEFAULT);
		if (defaultSigningAlgorithmNameList != null) {
			// Value is supposed to be a comma-separated list of signing protocol names
			String[] defaultSigningProtocolNames = defaultSigningAlgorithmNameList.split(","); //$NON-NLS-1$
			int i = 0;
			while (i < defaultSigningProtocolNames.length) {
				String defaultSigningProtocolName = defaultSigningProtocolNames[i++].trim().toUpperCase();
				try {
					// Check if the signature algorithm is installed
					Signature.getInstance(defaultSigningProtocolName);
					defaultSupportedSignatureAlgorithmNames.add(defaultSigningProtocolName);
				} catch (NoSuchAlgorithmException e) {

				}
			}
		}

		// See if the DBConnectionProperties hold a value for ENCRYPTIONALGORITHMNAMES
		String defaultCipherNameList = dbConnectionProperties.getProperty(DBConnectionProperties.DEFAULTENCRYPTIONALGORITHMNAMES, DBConnectionProperties.DEFAULTENCRYPTIONALGORITHMNAMESDEFAULT);
		if (defaultCipherNameList != null) {
			// Value is supposed to be a comma-separated list of encryption protocol names
			String[] defaultCipherNames = defaultCipherNameList.split(","); //$NON-NLS-1$
			int i = 0;
			while (i < defaultCipherNames.length) {
				String defaultCipherName = defaultCipherNames[i++].trim().toUpperCase();
				if (defaultCipherName.equalsIgnoreCase("SPE")) { //$NON-NLS-1$
					// defaultSupportedEncryptionAlgorithms.add(new SPE());
					defaultSupportedEncryptionAlgorithmNames.add(defaultCipherName);
				} else {
					try {
						// Check if the Cipher is installed
						Cipher.getInstance(defaultCipherName);
						// defaultSupportedEncryptionAlgorithms.add(new JCECipher(cipher));
						defaultSupportedEncryptionAlgorithmNames.add(defaultCipherName);
					} catch (NoSuchAlgorithmException e) {

					} catch (NoSuchPaddingException e) {

					}
				}
			}
		}
	}

	/**
	 * Gets defaultInitialAutoCommitForTransactions
	 * 
	 * @return defaultInitialAutoCommitForTransactions.
	 */
	public static final boolean isDefaultInitialAutoCommitForTransactions ( ) {
		return defaultInitialAutoCommitForTransactions;
	}

	/**
	 * The set of alternative signing protocols that the server engine supports (e.g. for user identity signing)
	 */
	private Set<String> alternativeSigningProtocols;

	/**
	 * The encryption Cipher to use on this connection
	 */
	private ProprietaryOrJCECipher cryptoProtocol;

	/**
	 * The default autocommit setting for all transactions created on this connection
	 */
	private boolean initialAutoCommitForTransactions = false;

	/**
	 * Time in milliseconds that the server has decided the connection can remain idle
	 */
	// private long idleTime;
	/**
	 * Indicator whether the connection is open or not
	 */
	private boolean open = true;

	/**
	 * The client's private key to be used for message signing
	 */
	private Signer signer;

	/**
	 * The signing protocol to use on this connection
	 */
	private Signature signingProtocol;

	/**
	 * The socket over which the Connection will communicate with the server
	 */
	private Socket socket;

	/**
	 * The socket's inputstream
	 */
	private DataInputStream socketInputStream;

	/**
	 * The socket's outputstream
	 */
	private DataOutputStream socketOutputStream;

	/**
	 * The specification version to use in communications with the server
	 */
	private Version specificationVersionForServer;

	/**
	 * The transaction under which DML can be executed
	 */
	private DBTransaction transaction;

	/**
	 * Creates a connection to the server on the specified host and port with the given autoCommit setting, and opens it for communication.
	 * 
	 * @param host
	 *            The identification of the host. It may either be its DNS name or its IP address in dotted decimal.
	 * @param port
	 *            The port to which to connect.
	 * @param signatureAlgorithmNames
	 *            a specified set of usable signature algorithm names
	 * @param clientID
	 *            The ID identifying the client of this connection to sira_prise
	 * @param initialAutoCommitForTransactions
	 *            true if the transactions created on this connection should have their autocommit set to true
	 * @param requestConnectionIdleTime
	 *            The requested time in milliseconds that the connection is allowed to stay idle
	 * @param encryptionAlgorithmNames
	 *            A specified set of usable encryption protocol names
	 * @param signer
	 *            The Signer object that will be called upon to compute the needed signature
	 * @throws IOException
	 *             if an I/O error occurs when creating the TCP socket.
	 * @throws CommunicationProtocolException
	 *             If an unexpected message was received during the handshake
	 * @throws DBException
	 *             If an error was detected on the server during the handshake
	 */
	public DBConnection (String host, int port, Set<String> signatureAlgorithmNames, String clientID, boolean initialAutoCommitForTransactions, long requestConnectionIdleTime, Set<String> encryptionAlgorithmNames, Signer signer) throws DBException, IOException, CommunicationProtocolException {
		this.initialAutoCommitForTransactions = initialAutoCommitForTransactions;
		this.signer = signer;
		socket = new Socket(host, port);

		// Anxiously await the ServerHelloMessage (and hope it won't be an error message)
		try {
			socket.setKeepAlive(true);
			socketOutputStream = new DataOutputStream(new BufferedOutputStream(socket.getOutputStream()));
			socketInputStream = new DataInputStream(new BufferedInputStream(socket.getInputStream()));

			// Create and send the ClientHelloMessage
			ClientHelloMessageType clientHelloMessageType;
			try {
				clientHelloMessageType = (ClientHelloMessageType) ServerMessageTypes.getInstance().getServerMessageType(ClientHelloMessageType.MESSAGETYPEID);
			} catch (NotFoundException e2) {
				throw new CommunicationProtocolException(Messages.getString("DBConnection.ClientHelloMessagetypeNotFound"), e2); //$NON-NLS-1$
			}
			ClientHelloMessage clientHelloMessage = clientHelloMessageType.message(signatureAlgorithmNames, encryptionAlgorithmNames, requestConnectionIdleTime, ServerMessageTypes.getInstance().getThisPackagesSiraPriseVersion());
			clientHelloMessage.sendMessage(socketOutputStream, null, null, null);

			ServerMessage serverMessage = ServerMessage.readMessage(socketInputStream, null, null, null);
			if (!(serverMessage instanceof ServerHelloMessage)) {
				throw new CommunicationProtocolException(serverMessage.getClass().getName(), ServerHelloMessageType.class.getName());
			}
			ServerHelloMessage serverHelloMessage = (ServerHelloMessage) serverMessage;
			signingProtocol = serverHelloMessage.getSigningProtocol();
			cryptoProtocol = serverHelloMessage.getCryptoProtocol();
			specificationVersionForServer = serverHelloMessage.getVersion();
			alternativeSigningProtocols = serverHelloMessage.getAlternativeSigningProtocols();
			Iterator<String> i_alternativeSigningProtocols = alternativeSigningProtocols.iterator();
			while (i_alternativeSigningProtocols.hasNext()) {
				String alternativeSigningProtocolName = i_alternativeSigningProtocols.next();
				if (!defaultSupportedSignatureAlgorithmNames.contains(alternativeSigningProtocolName.toUpperCase())) {
					i_alternativeSigningProtocols.remove();
				}
			}

			// idleTime = serverHelloMessage.getIdleTime();

			// if (idleTime > 0) {
			//                
			// }

			if (cryptoProtocol != null) {
				// do some initting here ?
				// That is currently done inside the serverHelloMessage creation, which is not the best of ideas
			}

			if (signingProtocol != null) {
				try {
					// Get the message to be signed (this is the clientID as written by the smallUTF encoding
					ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream();
					MyDataOutputStream.writeSmallUTF(clientID, new DataOutputStream(byteArrayOutputStream));
					byte[] signMessage = byteArrayOutputStream.toByteArray();
					byte[] signature = signer.sign(signingProtocol, signMessage);

					ClientAuthenticationMessageType clientAuthenticationMessageType = (ClientAuthenticationMessageType) ServerMessageTypes.getInstance().getServerMessageTypeForSiraPriseVersion(ClientAuthenticationMessageType.MESSAGETYPEID, specificationVersionForServer);
					ClientAuthenticationMessage clientAuthenticationMessage = clientAuthenticationMessageType.message(clientID, signature);
					clientAuthenticationMessage.sendMessage(socketOutputStream, null, null, cryptoProtocol);
				} catch (Exception e1) {
					closeSockets();
					throw new CommunicationProtocolException(Messages.getString("DBConnection.SigningFailed"), e1); //$NON-NLS-1$
				}

				// And now anxiously await the authenticationOK message
				serverMessage = ServerMessage.readMessage(socketInputStream, cryptoProtocol, null, null);
				if (!(serverMessage instanceof AuthenticationOKMessage)) {
					throw new CommunicationProtocolException(serverMessage.getClass().getName(), AuthenticationOKMessageType.class.getName());
				}
			}
		} catch (ErrorMessageException e) {
			closeSockets();
			throw new DBException(e);
		} catch (RuntimeException e) {
			closeSockets();
			throw e;
		} catch (IOException e) {
			closeSockets();
			throw e;
		} catch (CommunicationProtocolException e) {
			closeSockets();
			throw e;
		}
	}

	/**
	 * Creates a connection to the server on the specified host and port with the given autoCommit setting, and opens it for communication.
	 * 
	 * @param host
	 *            The identification of the host. It may either be its DNS name or its IP address in dotted decimal.
	 * @param port
	 *            The port to which to connect.
	 * @param signatureAlgorithmNames
	 *            a specified set of usable signature algorithm names
	 * @param clientID
	 *            The ID identifying the client of this connection to sira_prise
	 * @param initialAutoCommitForTransactions
	 *            true if the transactions created on this connection should have their autocommit set to true
	 * @param requestConnectionIdleTime
	 *            The requested time in milliseconds that the connection is allowed to stay idle
	 * @param signer
	 *            The Signer object that will be called upon to compute the needed signature
	 * @throws IOException
	 *             if an I/O error occurs when creating the TCP socket.
	 * @throws CommunicationProtocolException
	 *             If an unexpected message was received during the handshake
	 * @throws DBException
	 *             If an error was detected on the server during the handshake
	 */
	public DBConnection (String host, int port, Set<String> signatureAlgorithmNames, String clientID, boolean initialAutoCommitForTransactions, long requestConnectionIdleTime, Signer signer) throws IOException, CommunicationProtocolException, DBException {
		this(host, port, signatureAlgorithmNames, clientID, initialAutoCommitForTransactions, requestConnectionIdleTime, defaultSupportedEncryptionAlgorithmNames, signer);
	}

	/**
	 * Creates a connection to the server on the specified host and port, and opens it for communication. AutoCommit will be OFF. The requested connection idle time is the default time configured in the client properties.
	 * 
	 * @param host
	 *            The identification of the host. It may either be its DNS name or its IP address in dotted decimal.
	 * @param port
	 *            The port to which to connect.
	 * @param signatureAlgorithmNames
	 *            a specified set of usable signature algorithm names
	 * @param clientID
	 *            The userID identifying the user of this connection to sira_prise
	 * @param initialAutoCommitForTransactions
	 *            true if the transactions created on this connection should have their autocommit set to true
	 * @param encryptionAlgorithmNames
	 *            a specified set of usable encryption protocol names
	 * @param signer
	 *            The Signer object that will be called upon to compute the needed signature
	 * @throws IOException
	 *             if an I/O error occurs when creating the TCP socket.
	 * @throws CommunicationProtocolException
	 *             If an unexpected message was received during the handshake
	 * @throws DBException
	 *             If an error was detected on the server during the handshake
	 */
	public DBConnection (String host, int port, Set<String> signatureAlgorithmNames, String clientID, boolean initialAutoCommitForTransactions, Set<String> encryptionAlgorithmNames, Signer signer) throws IOException, CommunicationProtocolException, DBException {
		this(host, port, signatureAlgorithmNames, clientID, initialAutoCommitForTransactions, defaultConnectionIdleTime, encryptionAlgorithmNames, signer);
	}

	/**
	 * Creates a connection to the server on the specified host and port, and opens it for communication. AutoCommit will be OFF. The requested connection idle time is the default time configured in the client properties.
	 * 
	 * @param host
	 *            The identification of the host. It may either be its DNS name or its IP address in dotted decimal.
	 * @param port
	 *            The port to which to connect.
	 * @param signatureAlgorithmNames
	 *            a specified set of usable signature algorithm names
	 * @param clientID
	 *            The userID identifying the user of this connection to sira_prise
	 * @param initialAutoCommitForTransactions
	 *            true if the transactions created on this connection should have their autocommit set to true
	 * @param privateKeyProvider
	 *            The Signer object that will be called upon to compute the needed signature
	 * @throws IOException
	 *             if an I/O error occurs when creating the TCP socket.
	 * @throws CommunicationProtocolException
	 *             If an unexpected message was received during the handshake
	 * @throws DBException
	 *             If an error was detected on the server during the handshake
	 */
	public DBConnection (String host, int port, Set<String> signatureAlgorithmNames, String clientID, boolean initialAutoCommitForTransactions, Signer privateKeyProvider) throws IOException, CommunicationProtocolException, DBException {
		this(host, port, signatureAlgorithmNames, clientID, initialAutoCommitForTransactions, defaultConnectionIdleTime, defaultSupportedEncryptionAlgorithmNames, privateKeyProvider);
	}

	/**
	 * Creates a connection to the server on the specified host and port, and opens it for communication. AutoCommit will be OFF. The requested connection idle time is the default time configured in the client properties.
	 * 
	 * @param host
	 *            The identification of the host. It may either be its DNS name or its IP address in dotted decimal.
	 * @param port
	 *            The port to which to connect.
	 * @param signatureAlgorithmNames
	 *            a specified set of usable signature algorithm names
	 * @param clientID
	 *            The userID identifying the user of this connection to sira_prise
	 * @param requestConnectionIdleTime
	 *            The requested time in milliseconds that the connection is allowed to stay idle
	 * @param encryptionAlgorithmNames
	 *            a specified set of usable encryption protocol names
	 * @param signer
	 *            The Signer object that will be called upon to compute the needed signature
	 * @throws IOException
	 *             if an I/O error occurs when creating the TCP socket.
	 * @throws CommunicationProtocolException
	 *             If an unexpected message was received during the handshake
	 * @throws DBException
	 *             If an error was detected on the server during the handshake
	 */
	public DBConnection (String host, int port, Set<String> signatureAlgorithmNames, String clientID, long requestConnectionIdleTime, Set<String> encryptionAlgorithmNames, Signer signer) throws IOException, CommunicationProtocolException, DBException {
		this(host, port, signatureAlgorithmNames, clientID, defaultInitialAutoCommitForTransactions, requestConnectionIdleTime, encryptionAlgorithmNames, signer);
	}

	/**
	 * Creates a connection to the server on the specified host and port, and opens it for communication. AutoCommit will be OFF. The requested connection idle time is the default time configured in the client properties.
	 * 
	 * @param host
	 *            The identification of the host. It may either be its DNS name or its IP address in dotted decimal.
	 * @param port
	 *            The port to which to connect.
	 * @param signatureAlgorithmNames
	 *            a specified set of usable signature algorithm names
	 * @param clientID
	 *            The userID identifying the user of this connection to sira_prise
	 * @param requestConnectionIdleTime
	 *            The requested time in milliseconds that the connection is allowed to stay idle
	 * @param signer
	 *            The Signer object that will be called upon to compute the needed signature
	 * @throws IOException
	 *             if an I/O error occurs when creating the TCP socket.
	 * @throws CommunicationProtocolException
	 *             If an unexpected message was received during the handshake
	 * @throws DBException
	 *             If an error was detected on the server during the handshake
	 */
	public DBConnection (String host, int port, Set<String> signatureAlgorithmNames, String clientID, long requestConnectionIdleTime, Signer signer) throws IOException, CommunicationProtocolException, DBException {
		this(host, port, signatureAlgorithmNames, clientID, defaultInitialAutoCommitForTransactions, requestConnectionIdleTime, defaultSupportedEncryptionAlgorithmNames, signer);
	}

	/**
	 * Creates a connection to the server on the specified host and port, and opens it for communication. AutoCommit will be OFF. The requested connection idle time is the default time configured in the client properties.
	 * 
	 * @param host
	 *            The identification of the host. It may either be its DNS name or its IP address in dotted decimal.
	 * @param port
	 *            The port to which to connect.
	 * @param signatureAlgorithmNames
	 *            a specified set of usable signature algorithm names
	 * @param clientID
	 *            The userID identifying the user of this connection to sira_prise
	 * @param encryptionAlgorithmNames
	 *            a specified set of usable encryption protocol names
	 * @param signer
	 *            The Signer object that will be called upon to compute the needed signature
	 * @throws IOException
	 *             if an I/O error occurs when creating the TCP socket.
	 * @throws CommunicationProtocolException
	 *             If an unexpected message was received during the handshake
	 * @throws DBException
	 *             If an error was detected on the server during the handshake
	 */
	public DBConnection (String host, int port, Set<String> signatureAlgorithmNames, String clientID, Set<String> encryptionAlgorithmNames, Signer signer) throws IOException, CommunicationProtocolException, DBException {
		this(host, port, signatureAlgorithmNames, clientID, defaultInitialAutoCommitForTransactions, defaultConnectionIdleTime, encryptionAlgorithmNames, signer);
	}

	/**
	 * Creates a connection to the server on the specified host and port, and opens it for communication. AutoCommit will be OFF. The requested connection idle time is the default time configured in the client properties.
	 * 
	 * @param host
	 *            The identification of the host. It may either be its DNS name or its IP address in dotted decimal.
	 * @param port
	 *            The port to which to connect.
	 * @param signatureAlgorithmNames
	 *            a specified set of usable signature algorithm names
	 * @param clientID
	 *            The userID identifying the user of this connection to sira_prise
	 * @param signer
	 *            The Signer object that will be called upon to compute the needed signature
	 * @throws IOException
	 *             if an I/O error occurs when creating the TCP socket.
	 * @throws CommunicationProtocolException
	 *             If an unexpected message was received during the handshake
	 * @throws DBException
	 *             If an error was detected on the server during the handshake
	 */
	public DBConnection (String host, int port, Set<String> signatureAlgorithmNames, String clientID, Signer signer) throws IOException, CommunicationProtocolException, DBException {
		this(host, port, signatureAlgorithmNames, clientID, defaultInitialAutoCommitForTransactions, defaultConnectionIdleTime, defaultSupportedEncryptionAlgorithmNames, signer);
	}

	/**
	 * Creates a connection to the server on the specified host and port with the given autoCommit setting, and opens it for communication.
	 * 
	 * @param host
	 *            The identification of the host. It may either be its DNS name or its IP address in dotted decimal.
	 * @param port
	 *            The port to which to connect.
	 * @param clientID
	 *            The ID identifying the client of this connection to sira_prise
	 * @param initialAutoCommitForTransactions
	 *            true if the transactions created on this connection should have their autocommit set to true
	 * @param requestConnectionIdleTime
	 *            The requested time in milliseconds that the connection is allowed to stay idle
	 * @param encryptionAlgorithmNames
	 *            A specified set of usable encryption protocol names
	 * @param signer
	 *            The Signer object that will be called upon to compute the needed signature
	 * @throws IOException
	 *             if an I/O error occurs when creating the TCP socket.
	 * @throws CommunicationProtocolException
	 *             If an unexpected message was received during the handshake
	 * @throws DBException
	 *             If an error was detected on the server during the handshake
	 */
	public DBConnection (String host, int port, String clientID, boolean initialAutoCommitForTransactions, long requestConnectionIdleTime, Set<String> encryptionAlgorithmNames, Signer signer) throws IOException, CommunicationProtocolException, DBException {
		this(host, port, defaultSupportedSignatureAlgorithmNames, clientID, initialAutoCommitForTransactions, requestConnectionIdleTime, encryptionAlgorithmNames, signer);
	}

	/**
	 * Creates a connection to the server on the specified host and port with the given autoCommit setting, and opens it for communication.
	 * 
	 * @param host
	 *            The identification of the host. It may either be its DNS name or its IP address in dotted decimal.
	 * @param port
	 *            The port to which to connect.
	 * @param clientID
	 *            The ID identifying the client of this connection to sira_prise
	 * @param initialAutoCommitForTransactions
	 *            true if the transactions created on this connection should have their autocommit set to true
	 * @param requestConnectionIdleTime
	 *            The requested time in milliseconds that the connection is allowed to stay idle
	 * @param signer
	 *            The Signer object that will be called upon to compute the needed signature
	 * @throws IOException
	 *             if an I/O error occurs when creating the TCP socket.
	 * @throws CommunicationProtocolException
	 *             If an unexpected message was received during the handshake
	 * @throws DBException
	 *             If an error was detected on the server during the handshake
	 */
	public DBConnection (String host, int port, String clientID, boolean initialAutoCommitForTransactions, long requestConnectionIdleTime, Signer signer) throws IOException, CommunicationProtocolException, DBException {
		this(host, port, defaultSupportedSignatureAlgorithmNames, clientID, initialAutoCommitForTransactions, requestConnectionIdleTime, defaultSupportedEncryptionAlgorithmNames, signer);
	}

	/**
	 * Creates a connection to the server on the specified host and port, and opens it for communication. AutoCommit will be OFF. The requested connection idle time is the default time configured in the client properties.
	 * 
	 * @param host
	 *            The identification of the host. It may either be its DNS name or its IP address in dotted decimal.
	 * @param port
	 *            The port to which to connect.
	 * @param clientID
	 *            The userID identifying the user of this connection to sira_prise
	 * @param initialAutoCommitForTransactions
	 *            true if the transactions created on this connection should have their autocommit set to true
	 * @param encryptionAlgorithmNames
	 *            a specified set of usable encryption protocol names
	 * @param signer
	 *            The Signer object that will be called upon to compute the needed signature
	 * @throws IOException
	 *             if an I/O error occurs when creating the TCP socket.
	 * @throws CommunicationProtocolException
	 *             If an unexpected message was received during the handshake
	 * @throws DBException
	 *             If an error was detected on the server during the handshake
	 */
	public DBConnection (String host, int port, String clientID, boolean initialAutoCommitForTransactions, Set<String> encryptionAlgorithmNames, Signer signer) throws IOException, CommunicationProtocolException, DBException {
		this(host, port, defaultSupportedSignatureAlgorithmNames, clientID, initialAutoCommitForTransactions, defaultConnectionIdleTime, encryptionAlgorithmNames, signer);
	}

	/**
	 * Creates a connection to the server on the specified host and port, and opens it for communication. AutoCommit will be OFF. The requested connection idle time is the default time configured in the client properties.
	 * 
	 * @param host
	 *            The identification of the host. It may either be its DNS name or its IP address in dotted decimal.
	 * @param port
	 *            The port to which to connect.
	 * @param clientID
	 *            The userID identifying the user of this connection to sira_prise
	 * @param initialAutoCommitForTransactions
	 *            true if the transactions created on this connection should have their autocommit set to true
	 * @param signer
	 *            The Signer object that will be called upon to compute the needed signature
	 * @throws IOException
	 *             if an I/O error occurs when creating the TCP socket.
	 * @throws CommunicationProtocolException
	 *             If an unexpected message was received during the handshake
	 * @throws DBException
	 *             If an error was detected on the server during the handshake
	 */
	public DBConnection (String host, int port, String clientID, boolean initialAutoCommitForTransactions, Signer signer) throws IOException, CommunicationProtocolException, DBException {
		this(host, port, defaultSupportedSignatureAlgorithmNames, clientID, initialAutoCommitForTransactions, defaultConnectionIdleTime, defaultSupportedEncryptionAlgorithmNames, signer);
	}

	/**
	 * Creates a connection to the server on the specified host and port, and opens it for communication. AutoCommit will be OFF. The requested connection idle time is the default time configured in the client properties.
	 * 
	 * @param host
	 *            The identification of the host. It may either be its DNS name or its IP address in dotted decimal.
	 * @param port
	 *            The port to which to connect.
	 * @param clientID
	 *            The userID identifying the user of this connection to sira_prise
	 * @param requestConnectionIdleTime
	 *            The requested time in milliseconds that the connection is allowed to stay idle
	 * @param encryptionAlgorithmNames
	 *            a specified set of usable encryption protocol names
	 * @param signer
	 *            The Signer object that will be called upon to compute the needed signature
	 * @throws IOException
	 *             if an I/O error occurs when creating the TCP socket.
	 * @throws CommunicationProtocolException
	 *             If an unexpected message was received during the handshake
	 * @throws DBException
	 *             If an error was detected on the server during the handshake
	 */
	public DBConnection (String host, int port, String clientID, long requestConnectionIdleTime, Set<String> encryptionAlgorithmNames, Signer signer) throws IOException, CommunicationProtocolException, DBException {
		this(host, port, defaultSupportedSignatureAlgorithmNames, clientID, defaultInitialAutoCommitForTransactions, requestConnectionIdleTime, encryptionAlgorithmNames, signer);
	}

	/**
	 * Creates a connection to the server on the specified host and port, and opens it for communication. AutoCommit will be OFF. The requested connection idle time is the default time configured in the client properties.
	 * 
	 * @param host
	 *            The identification of the host. It may either be its DNS name or its IP address in dotted decimal.
	 * @param port
	 *            The port to which to connect.
	 * @param clientID
	 *            The userID identifying the user of this connection to sira_prise
	 * @param requestConnectionIdleTime
	 *            The requested time in milliseconds that the connection is allowed to stay idle
	 * @param signer
	 *            The Signer object that will be called upon to compute the needed signature
	 * @throws IOException
	 *             if an I/O error occurs when creating the TCP socket.
	 * @throws CommunicationProtocolException
	 *             If an unexpected message was received during the handshake
	 * @throws DBException
	 *             If an error was detected on the server during the handshake
	 */
	public DBConnection (String host, int port, String clientID, long requestConnectionIdleTime, Signer signer) throws IOException, CommunicationProtocolException, DBException {
		this(host, port, defaultSupportedSignatureAlgorithmNames, clientID, defaultInitialAutoCommitForTransactions, requestConnectionIdleTime, defaultSupportedEncryptionAlgorithmNames, signer);
	}

	/**
	 * Creates a connection to the server on the specified host and port, and opens it for communication. AutoCommit will be OFF. The requested connection idle time is the default time configured in the client properties.
	 * 
	 * @param host
	 *            The identification of the host. It may either be its DNS name or its IP address in dotted decimal.
	 * @param port
	 *            The port to which to connect.
	 * @param clientID
	 *            The userID identifying the user of this connection to sira_prise
	 * @param encryptionAlgorithmNames
	 *            a specified set of usable encryption protocol names
	 * @param signer
	 *            The Signer object that will be called upon to compute the needed signature
	 * @throws IOException
	 *             if an I/O error occurs when creating the TCP socket.
	 * @throws CommunicationProtocolException
	 *             If an unexpected message was received during the handshake
	 * @throws DBException
	 *             If an error was detected on the server during the handshake
	 */
	public DBConnection (String host, int port, String clientID, Set<String> encryptionAlgorithmNames, Signer signer) throws IOException, CommunicationProtocolException, DBException {
		this(host, port, defaultSupportedSignatureAlgorithmNames, clientID, defaultInitialAutoCommitForTransactions, defaultConnectionIdleTime, encryptionAlgorithmNames, signer);
	}

	/**
	 * Creates a connection to the server on the specified host and port, and opens it for communication. AutoCommit will be OFF. The requested connection idle time is the default time configured in the client properties.
	 * 
	 * @param host
	 *            The identification of the host. It may either be its DNS name or its IP address in dotted decimal.
	 * @param port
	 *            The port to which to connect.
	 * @param clientID
	 *            The userID identifying the user of this connection to sira_prise
	 * @param signer
	 *            The Signer object that will be called upon to compute the needed signature
	 * @throws IOException
	 *             if an I/O error occurs when creating the TCP socket.
	 * @throws CommunicationProtocolException
	 *             If an unexpected message was received during the handshake
	 * @throws DBException
	 *             If an error was detected on the server during the handshake
	 */
	public DBConnection (String host, int port, String clientID, Signer signer) throws IOException, CommunicationProtocolException, DBException {
		this(host, port, defaultSupportedSignatureAlgorithmNames, clientID, defaultInitialAutoCommitForTransactions, defaultConnectionIdleTime, defaultSupportedEncryptionAlgorithmNames, signer);
	}

	/**
	 * Close the connection and release all allocated resources.
	 */
	void closeSockets ( ) {
		if (open) {
			try {
				socketInputStream.close();
				socketOutputStream.close();
				socket.close();
			} catch (IOException e) {

			} finally {
				open = false;
				socket = null;
				socketInputStream = null;
				socketOutputStream = null;
				cryptoProtocol = null;
				signingProtocol = null;
				signer = null;
				specificationVersionForServer = null;
			}
		}
	}

	/**
	 * 
	 */
	void endAnyTransaction ( ) {
		//		System.out.println("trying to end any current transaction"); //$NON-NLS-1$
		if (transaction != null) {
			//			System.out.println("there is a current transaction"); //$NON-NLS-1$
			try {
				long rootTransactionID = -1;
				// Create and send the EndTransactionMessage
				// make sure it's the root transaction that gets ended
				while (transaction != null) {
					rootTransactionID = transaction.getTransactionID();
					transaction = transaction.getParentTransaction();
				}
				//				System.out.println("sending endtrans command to " + rootTransactionID); //$NON-NLS-1$
				((EndTransactionMessageType) ServerMessageTypes.getInstance().getServerMessageTypeForSiraPriseVersion(EndTransactionMessageType.MESSAGETYPEID, specificationVersionForServer)).message(false, rootTransactionID).sendMessage(socketOutputStream, signingProtocol, signer, cryptoProtocol);
				// No need to await the response
				//					
			} catch (IOException e) {

			} catch (CommunicationProtocolException e) {

			} catch (NotFoundException e) {

			} finally {
				transaction = null;
			}
		}
	}

	/**
	 * Gets The encryption Cipher to use on this connection
	 * 
	 * @return The encryption Cipher to use on this connection
	 */
	final ProprietaryOrJCECipher getCryptoProtocol ( ) {
		return cryptoProtocol;
	}

	/**
	 * Gets The client's private key to be used for message signing
	 * 
	 * @return The client's private key to be used for message signing
	 */
	final Signer getSigner ( ) {
		return signer;
	}

	/**
	 * Gets The signing protocol to use on this connection
	 * 
	 * @return The signing protocol to use on this connection
	 */
	final Signature getSigningProtocol ( ) {
		return signingProtocol;
	}

	/**
	 * Gets The socket's outputstream
	 * 
	 * @return The socket's outputstream
	 */
	final DataOutputStream getSocketOutputStream ( ) {
		return socketOutputStream;
	}

	/**
	 * Performs a commit for the transaction that is active on the connection
	 * 
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to any problem that occurred during communication with the server
	 */
	void sendCommitMessage ( ) throws ConnectionClosedException {
		try {
			((CommitRollbackMessageType) ServerMessageTypes.getInstance().getServerMessageTypeForSiraPriseVersion(CommitRollbackMessageType.MESSAGETYPEID, specificationVersionForServer)).message(true, -1).sendMessage(socketOutputStream, signingProtocol, signer, cryptoProtocol);

			ServerMessage serverMessage = ServerMessage.readMessage(socketInputStream, cryptoProtocol, null, null);
			if (!(serverMessage instanceof CommittedMessage)) {
				throw new CommunicationProtocolException(serverMessage.getClass().getName(), CommittedMessageType.class.getName());
			}
		} catch (IOException e) {
			// IO error in the send ===> regard connection as "become unusable" ===>
			transaction = null;
			closeSockets();
			throw new ConnectionClosedException(e);
		} catch (CommunicationProtocolException e) {
			transaction = null;
			closeSockets();
			throw new ConnectionClosedException(e);
		} catch (NotFoundException e) {
			transaction = null;
			closeSockets();
			throw new ConnectionClosedException(new CommunicationProtocolException(CommitRollbackMessageType.MESSAGETYPEID, specificationVersionForServer.getFullVersion()));
		} catch (ErrorMessageException e) {
			transaction = null;
			closeSockets();
			throw new ConnectionClosedException(e);
		}
	}

	/**
	 * Ends the transaction, issuing either a commit or a rollback.
	 * 
	 * @param commit
	 *            true if the transaction is to be committed, false if it is to be rolled back.
	 * @param transactionID
	 *            The server-side ID of the transaction to be ended, or -1 for "the current transaction", meaning the lowest nested transaction in the current transaction structure
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server
	 */
	void sendEndTransactionMessage (boolean commit, long transactionID) throws ConnectionClosedException {
		try {
			try {
				((EndTransactionMessageType) ServerMessageTypes.getInstance().getServerMessageTypeForSiraPriseVersion(EndTransactionMessageType.MESSAGETYPEID, specificationVersionForServer)).message(commit, transactionID).sendMessage(socketOutputStream, signingProtocol, signer, cryptoProtocol);
			} catch (NotFoundException e1) {
				throw new CommunicationProtocolException(EndTransactionMessageType.MESSAGETYPEID, specificationVersionForServer.getFullVersion());
			}
			// And now anxiously await the TransactionEnded message
			ServerMessage serverMessage = ServerMessage.readMessage(socketInputStream, cryptoProtocol, null, null);

			if (!(serverMessage instanceof TransactionEndedMessage)) {
				throw new CommunicationProtocolException(serverMessage.getClass().getName(), TransactionEndedMessageType.class.getName());
			}
		} catch (IOException e) {
			// IO error in the send
			transaction = null;
			closeSockets();
			throw new ConnectionClosedException(e);
		} catch (CommunicationProtocolException e) {
			transaction = null;
			closeSockets();
			throw new ConnectionClosedException(e);
		} catch (ErrorMessageException e) {
			transaction = null;
			closeSockets();
			throw new ConnectionClosedException(e);
		} finally {
			while (transaction != null && transaction.getTransactionID() != transactionID) {
				transaction = transaction.getParentTransaction();
			}
			// transaction now points to the transaction that was ended ===> one more time
			if (transaction != null) {
				transaction = transaction.getParentTransaction();
			}
		}
	}

	/**
	 * Sends a rollback message on behalf of the transaction whose server-side ID is given.
	 * 
	 * @param transactionID
	 *            The server-side ID of the transaction that requests to roll back
	 * @throws ConnectionClosedException
	 *             If a severe error occurred that causes the connection to become unusable
	 */
	void sendRollbackMessage (long transactionID) throws ConnectionClosedException {
		try {
			((CommitRollbackMessageType) ServerMessageTypes.getInstance().getServerMessageTypeForSiraPriseVersion(CommitRollbackMessageType.MESSAGETYPEID, specificationVersionForServer)).message(false, transactionID).sendMessage(socketOutputStream, signingProtocol, signer, cryptoProtocol);

			ServerMessage serverMessage = ServerMessage.readMessage(socketInputStream, cryptoProtocol, null, null);
			if (!(serverMessage instanceof CommittedMessage)) {
				throw new CommunicationProtocolException(serverMessage.getClass().getName(), CommittedMessageType.class.getName());
			}
		} catch (IOException e) {
			// IO error in the send ===> regard connection as "become unusable" ===>
			transaction = null;
			closeSockets();
			throw new ConnectionClosedException(e);
		} catch (CommunicationProtocolException e) {
			transaction = null;
			closeSockets();
			throw new ConnectionClosedException(e);
		} catch (NotFoundException e) {
			transaction = null;
			closeSockets();
			throw new ConnectionClosedException(new CommunicationProtocolException(CommitRollbackMessageType.MESSAGETYPEID, specificationVersionForServer.getFullVersion()));
		} catch (ErrorMessageException e) {
			transaction = null;
			closeSockets();
			throw new ConnectionClosedException(e);
		}
	}

	/**
	 * Commits the non-autocommit transaction that is currently started on the connection.
	 * 
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to any error that occurred during communication with the server
	 */
	void verifyStateAndSendCommitMessage ( ) throws ConnectionClosedException {
		if (!open) {
			throw new IllegalConnectionStateException(Messages.getString("DBConnection.Closed")); //$NON-NLS-1$
		}
		if (transaction == null) {
			throw new IllegalConnectionStateException(Messages.getString("DBConnection.NoTransaction")); //$NON-NLS-1$
		}

		if (!transaction.isAutoCommit()) {
			sendCommitMessage();
		}
	}

	/**
	 * Ends the transaction, issuing either a commit or a rollback.
	 * 
	 * @param commit
	 *            true if the transaction is to be committed, false if it is to be rolled back.
	 * @param transactionID
	 *            The server-side ID of the transaction to be ended, or -1 for "the current transaction", meaning the lowest nested transaction in the current transaction structure
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server
	 */
	void verifyStateAndSendEndTransactionMessage (boolean commit, long transactionID) throws ConnectionClosedException {
		if (!open) {
			throw new IllegalConnectionStateException(Messages.getString("DBConnection.Closed")); //$NON-NLS-1$
		}
		if (transaction == null) {
			throw new IllegalConnectionStateException(Messages.getString("DBConnection.NoTransaction")); //$NON-NLS-1$
		}

		sendEndTransactionMessage(commit, transactionID);
	}

	/**
	 * Sends a command to the server and returns the result obtained. If that result indicates any kind of error, then the connection will be closed.
	 * 
	 * @param cmd
	 *            the command to be executed, in textual form.
	 * @return the result of the command. If the connection has already been closed, then a DBException is thrown.
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server
	 * @throws ErrorMessageException
	 *             If the response obtained from the server is not a commit confirmation, but an error response instead.
	 */
	AbstractRelation verifyStateAndSendExecDmlCommandMessage (String cmd) throws ConnectionClosedException, ErrorMessageException {
		if (!open) {
			throw new IllegalConnectionStateException(Messages.getString("DBConnection.Closed")); //$NON-NLS-1$
		}
		if (transaction == null) {
			throw new IllegalConnectionStateException(Messages.getString("DBConnection.NoTransaction")); //$NON-NLS-1$
		}

		try {
			((ExecuteDMLMessageType) ServerMessageTypes.getInstance().getServerMessageTypeForSiraPriseVersion(ExecuteDMLMessageType.MESSAGETYPEID, specificationVersionForServer)).message(cmd).sendMessage(socketOutputStream, signingProtocol, signer, cryptoProtocol);

			// And now anxiously await the DmlExecuted message
			ServerMessage serverMessage = ServerMessage.readMessage(socketInputStream, cryptoProtocol, null, null);
			if (!(serverMessage instanceof DmlExecutedMessage)) {
				throw new CommunicationProtocolException(serverMessage.getClass().getName(), DmlExecutedMessageType.class.getName());
			}

			return ((DmlExecutedMessage) serverMessage).getRelation();
		} catch (IOException e) {
			// IO error in the send
			transaction = null;
			closeSockets();
			throw new ConnectionClosedException(e);
		} catch (CommunicationProtocolException e) {
			transaction = null;
			closeSockets();
			throw new ConnectionClosedException(e);
		} catch (NotFoundException e) {
			transaction = null;
			closeSockets();
			throw new ConnectionClosedException(new CommunicationProtocolException(ExecuteDMLMessageType.MESSAGETYPEID, specificationVersionForServer.getFullVersion()));
		} catch (ErrorMessageException e) {
			if (specificationVersionForServer.isBefore(Version.ONE_TWO)) {
				transaction = null;
			}
			// closeSockets();
			throw e;
		}
	}

	/**
	 * Sends a command to the server and returns the result obtained. If that result indicates any kind of error, then the connection will be closed.
	 * 
	 * @param cmd
	 *            the command to be executed, in textual form.
	 * @return the result of the command. If the connection has already been closed, then a DBException is thrown.
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server
	 * @throws ErrorMessageException
	 *             If the response obtained from the server is not a commit confirmation, but an error response instead.
	 */
	AbstractRelation verifyStateAndSendExecDmlCommandMessage (ServerCommand cmd) throws ConnectionClosedException, ErrorMessageException {
		if (!open) {
			throw new IllegalConnectionStateException(Messages.getString("DBConnection.Closed")); //$NON-NLS-1$
		}
		if (transaction == null) {
			throw new IllegalConnectionStateException(Messages.getString("DBConnection.NoTransaction")); //$NON-NLS-1$
		}

		try {
			((ExecuteDMLMessageTypesV14) ServerMessageTypes.getInstance().getServerMessageTypeForSiraPriseVersion(cmd.getCorrespondingMessagetypeid(), specificationVersionForServer)).message(cmd).sendMessage(socketOutputStream, signingProtocol, signer, cryptoProtocol);

			// And now anxiously await the DmlExecuted message
			ServerMessage serverMessage = ServerMessage.readMessage(socketInputStream, cryptoProtocol, null, null);
			if (!(serverMessage instanceof DmlExecutedMessage)) {
				throw new CommunicationProtocolException(serverMessage.getClass().getName(), DmlExecutedMessageType.class.getName());
			}

			return ((DmlExecutedMessage) serverMessage).getRelation();
		} catch (IOException e) {
			// IO error in the send
			transaction = null;
			closeSockets();
			throw new ConnectionClosedException(e);
		} catch (CommunicationProtocolException e) {
			transaction = null;
			closeSockets();
			throw new ConnectionClosedException(e);
		} catch (NotFoundException e) {
			transaction = null;
			closeSockets();
			throw new ConnectionClosedException(new CommunicationProtocolException(ExecuteDMLMessageType.MESSAGETYPEID, specificationVersionForServer.getFullVersion()));
		} catch (ErrorMessageException e) {
			if (specificationVersionForServer.isBefore(Version.ONE_TWO)) {
				transaction = null;
			}
			// closeSockets();
			throw e;
		}
	}

	/**
	 * Starts a nested transaction on the connection with the given autoCommit setting and transaction mode
	 * 
	 * @param autoCommit
	 *            The autoCommit setting for the transaction
	 * @param transactionMode
	 *            the mode for the transaction
	 * @return The started transaction
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server
	 * @throws ErrorMessageException
	 *             If the response obtained from the server was not a confirmation, but an error message instead
	 */
	final DBTransaction verifyStateAndSendStartNestedTransactionMessage (boolean autoCommit, TransactionMode transactionMode) throws ConnectionClosedException, ErrorMessageException {
		if (!open) {
			throw new IllegalConnectionStateException(Messages.getString("DBConnection.Closed")); //$NON-NLS-1$
		}
		if (transaction == null) {
			throw new IllegalConnectionStateException(Messages.getString("DBConnection.NoMainTransaction")); //$NON-NLS-1$
		}

		try {
			((StartSubTransactionMessageType) ServerMessageTypes.getInstance().getServerMessageTypeForSiraPriseVersion(StartSubTransactionMessageType.MESSAGETYPEID, specificationVersionForServer)).message(transactionMode.isReadOnly(), autoCommit).sendMessage(socketOutputStream, signingProtocol, signer, cryptoProtocol);

			// And now anxiously await the TransactionStarted message
			ServerMessage serverMessage = ServerMessage.readMessage(socketInputStream, cryptoProtocol, null, null);

			if (!(serverMessage instanceof TransactionStartedMessage)) {
				throw new CommunicationProtocolException(serverMessage.getClass().getName(), TransactionStartedMessageType.class.getName());
			}

			return transaction = new DBTransaction(transaction, autoCommit, ((TransactionStartedMessage) serverMessage).getTransactionID(), transactionMode);
		} catch (IOException e1) {
			transaction = null;
			closeSockets();
			throw new ConnectionClosedException(e1);
		} catch (CommunicationProtocolException e1) {
			transaction = null;
			closeSockets();
			throw new ConnectionClosedException(e1);
		} catch (NotFoundException e) {
			throw new RuntimeException(MessageFormat.format(Messages.getString("DBConnection.IncompatibleServerVersion"), new Object[] { specificationVersionForServer.toString() })); //$NON-NLS-1$
		}
	}

	/**
	 * Closes the connection. If an uncommitted transaction is still pending on the connection, it will be rolled back and ended.
	 */
	public abstract void close ( );

	/**
	 * Commits the non-autocommit transaction that is currently started on the connection.
	 * 
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server
	 * @deprecated Use the commit() method of the DBTransaction object instead
	 */
	public void commit ( ) throws ConnectionClosedException {
		verifyStateAndSendCommitMessage();
	}

	/**
	 * Sends a query expression to the server for compilation-and-caching.
	 * 
	 * @param query
	 *            the query to be compiled-and-cached, in textual form.
	 * @return the result of the compilation. That result is a relation with the following attributes :
	 *         <ol>
	 *         <li>QUERYID(LONG(...)) The internal ID of the compiled query, by which the query can later be referenced</li>
	 *         <li>HEADING(RELATION(...)) The description of the heading of the relation as it will be output by the query</li>
	 *         <li>PARMS(RELATION(...)) The description of all parameters required by the query</li>
	 *         </ol>
	 *         The heading of the 'HEADING' relation is as follows :
	 *         <ol>
	 *         <li>ATTRIBUTENAME(NAME(...)) The name of an attribute in the output of the query</li>
	 *         <li>TYPENAME(NAME(...)) The name of the type of that attribute in the output of the query</li>
	 *         </ol>
	 *         The heading of the 'PARMS' relation is as follows :
	 *         <ol>
	 *         <li>PARAMETERNAME(NAME(...)) The name of a parameter required by the query</li>
	 *         <li>TYPENAME(NAME(...)) The name of the type of that parameter</li>
	 *         </ol>
	 * @throws DBException
	 *             If the connection has already been closed, no transaction was started, or an error was found in the compilation.
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server
	 * @deprecated use the compileAndCacheQuery() method from the DBTransaction instead
	 */
	public AbstractRelation compileAndCacheQuery (String query) throws DBException, ConnectionClosedException {
		return execDmlCommand("CACHE " + query); //$NON-NLS-1$
	}

	/**
	 * Sends a query expression to the server for verification only.
	 * 
	 * @param query
	 *            the query to be verified, in textual form.
	 * @return the result of the verification. That result is a relation with the following attributes :
	 *         <ol>
	 *         <li>HEADING(RELATION(...)) The description of the heading of the relation as it will be output by the query</li>
	 *         <li>PARMS(RELATION(...)) The description of all parameters required by the query</li>
	 *         </ol>
	 *         The heading of the 'HEADING' relation is as follows :
	 *         <ol>
	 *         <li>ATTRIBUTENAME(NAME(...)) The name of an attribute in the output of the query</li>
	 *         <li>TYPENAME(NAME(...)) The name of the type of that attribute in the output of the query</li>
	 *         </ol>
	 *         The heading of the 'PARMS' relation is as follows :
	 *         <ol>
	 *         <li>PARAMETERNAME(NAME(...)) The name of a parameter required by the query</li>
	 *         <li>TYPENAME(NAME(...)) The name of the type of that parameter</li>
	 *         </ol>
	 * @throws DBException
	 *             If the connection has already been closed, no transaction was started, or an error was found in the compilation.
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server
	 * @deprecated use compileQuery() on the DBTransaction object (obtained from startTransaction()) instead.
	 */
	public AbstractRelation compileQuery (String query) throws DBException, ConnectionClosedException {
		return execDmlCommand("COMPILE " + query); //$NON-NLS-1$
	}

	/**
	 * Ends the transaction, issuing either a commit or a rollback.
	 * 
	 * @param commit
	 *            true if the transaction is to be committed, false if it is to be rolled back.
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server
	 * @deprecated instead, use end() on the DBTransaction object obtained from startTransaction().
	 */
	public void endTransaction (boolean commit) throws ConnectionClosedException {
		verifyStateAndSendEndTransactionMessage(commit, -1);
	}

	/**
	 * Sends a command to the server and returns the result obtained. If that result indicates any kind of error, then the connection will be closed.
	 * 
	 * @param cmd
	 *            the command to be executed, in textual form.
	 * @return the result of the command. If the connection has already been closed, then a DBException is thrown.
	 * @throws DBException
	 *             If the connection has already been closed, no transaction was started, or an error was found.
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server
	 * @deprecated Use the execDmlCommand() method from the DBTransaction instead
	 */
	public AbstractRelation execDmlCommand (String cmd) throws DBException, ConnectionClosedException {
		try {
			AbstractRelation result = verifyStateAndSendExecDmlCommandMessage(cmd);

			if (transaction.isAutoCommit() && !((cmd.length() > 7 && cmd.substring(0, 8).equalsIgnoreCase("ROLLBACK")) || (cmd.length() > 5 && cmd.substring(0, 6).equalsIgnoreCase("COMMIT")))) { //$NON-NLS-1$ //$NON-NLS-2$
				sendCommitMessage();
			}

			return result;
		} catch (ErrorMessageException e) {
			if (specificationVersionForServer.isBefore(Version.ONE_TWO)) {
				transaction = null;
			}
			throw new DBException(e);
		}
	}

	/**
	 * Sends a command to the server and returns the result obtained. After execution of the command, the transaction is committed and ended.
	 * 
	 * @param cmd
	 *            the command to be executed, in textual form.
	 * @return The result of executing the command. Null if the command is an assignment command.
	 * @throws DBException
	 *             If the connection has already been closed, no transaction was started, or an error was found.
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server
	 * @deprecated instead, use execDmlCommandAndEndTransaction() on the DBTransaction object obtained from startTransaction()
	 */
	public AbstractRelation execDmlCommandAndEndTransaction (String cmd) throws DBException, ConnectionClosedException {
		AbstractRelation r = execDmlCommand(cmd);
		endTransaction(true);
		return r;
	}

	/**
	 * Executes a series of single-assignment commands as a multiple-assignment command. Optionally closes the connection immediately after succesfull execution of the series of assignments. If the result indicates any kind of error, then the connection will be closed.
	 * 
	 * @param commands
	 *            The commands to be executed. Execution is in the obvious order, and thus not randomly, or in reverse order, or whatever.
	 * @return the DBResponse obtained.
	 * @throws DBException
	 *             If the connection has already been closed, no transaction was started, or an error was found.
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server
	 * @deprecated replace this method by the execMultipleStatementAndEndTransaction() of the DBTransaction object obtained from startTransaction()
	 */
	public AbstractRelation execMultipleStatement (Collection<?> commands) throws DBException, ConnectionClosedException {
		Iterator<?> i_commands = commands.iterator();
		String multipleAssignmentCommand = ""; //$NON-NLS-1$
		while (i_commands.hasNext()) {
			multipleAssignmentCommand += ("CMD(" + i_commands.next() + ")"); //$NON-NLS-1$//$NON-NLS-2$
		}

		if (multipleAssignmentCommand.length() > 0) {
			return execDmlCommand(multipleAssignmentCommand);
		} else {
			return null;
		}
	}

	/**
	 * Executes a series of single-assignment commands as a multiple-assignment command. If the result indicates any kind of error, then the connection will be closed.
	 * 
	 * @param commands
	 *            The single-assignment commands to be executed. Execution is in the obvious order, and thus not randomly, or in reverse order, or whatever.
	 * @return the DBResponse obtained.
	 * @throws DBException
	 *             If the connection has already been closed, no transaction was started, or an error was found.
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server
	 * @deprecated replace this method by the execMultipleStatementAndEndTransaction() of the DBTransaction object obtained from startTransaction()
	 */
	public AbstractRelation execMultipleStatement (String[] commands) throws DBException, ConnectionClosedException {
		int c = commands.length, i = 0;
		if (c > 0) {
			String multipleAssignmentCommand = ""; //$NON-NLS-1$
			while (i < c) {
				multipleAssignmentCommand += ("CMD(" + commands[i++] + ")"); //$NON-NLS-1$//$NON-NLS-2$
			}
			return execDmlCommand(multipleAssignmentCommand);
		} else {
			return null;
		}
	}

	/**
	 * Executes a series of single-assignment commands as a multiple-assignment command.
	 * 
	 * @param commands
	 *            The commands to be executed. Execution is in the obvious order, and thus not randomly, or in reverse order, or whatever.
	 * @return the DBResponse obtained.
	 * @throws DBException
	 *             If the connection has already been closed, no transaction was started, or an error was found.
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server
	 * @deprecated replace this method by the execMultipleStatementAndEndTransaction() of the DBTransaction object obtained from startTransaction()
	 */
	public AbstractRelation execMultipleStatementAndEndTransaction (Collection<?> commands) throws DBException, ConnectionClosedException {
		AbstractRelation response = execMultipleStatement(commands);
		endTransaction(true);
		return response;
	}

	/**
	 * Executes a series of single-assignment commands as a multiple-assignment command.
	 * 
	 * @param commands
	 *            The single-assignment commands to be executed. Execution is in the obvious order, and thus not randomly, or in reverse order, or whatever.
	 * @return the DBResponse obtained.
	 * @throws DBException
	 *             If the connection has already been closed, no transaction was started, or an error was found.
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server
	 * @deprecated replace this method by the execMultipleStatementAndEndTransaction() of the DBTransaction object obtained from startTransaction()
	 */
	public AbstractRelation execMultipleStatementAndEndTransaction (String[] commands) throws DBException, ConnectionClosedException {
		AbstractRelation response = execMultipleStatement(commands);
		endTransaction(true);
		return response;
	}

	/**
	 * Executes a series of inquire commands on the server.
	 * 
	 * @param expression
	 *            The expressions of the relational algebra to be inquired.
	 * @return response A DBResponse array holding the values of the corresponding queried expressions.
	 * @throws DBException
	 *             If the connection has already been closed, no transaction was started, or an error was found.
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server
	 * @deprecated replace this method by the execQueries() method of the DBTransaction object obtained from startTransaction()
	 */
	public AbstractRelation[] execQueries (String[] expression) throws DBException, ConnectionClosedException {
		AbstractRelation[] rsp = new AbstractRelation[expression.length];
		int i = 0;
		while (i < expression.length) {
			rsp[i] = execQuery(expression[i]);
			i++;
		}

		return rsp;
	}

	/**
	 * Executes a series of inquire commands on the server.
	 * 
	 * @param expression
	 *            The expressions of the relational algebra to be inquired.
	 * @return response A DBResponse array holding the values of the corresponding queried expressions.
	 * @throws DBException
	 *             If the connection has already been closed, no transaction was started, or an error was found.
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server
	 * @deprecated replace this method by the execQueriesAndEndTransaction() method of the DBTransaction object obtained from startTransaction()
	 */
	public AbstractRelation[] execQueriesAndEndTransaction (String[] expression) throws DBException, ConnectionClosedException {
		AbstractRelation[] rsp = execQueries(expression);
		endTransaction(true);
		return rsp;
	}

	/**
	 * Executes an inquire command on the server. If the result indicates any kind of error, then the connection will be closed.
	 * 
	 * @param expression
	 *            The expression of the relational algebra to be inquired.
	 * @return response A DBResponse holding the value of the queried expression.
	 * @throws DBException
	 *             If the connection has already been closed, no transaction was started, or an error was found.
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server
	 * @deprecated replace this method by the execQuery() method of the DBTransaction object obtained from startTransaction()
	 */
	public AbstractRelation execQuery (String expression) throws DBException, ConnectionClosedException {
		return execDmlCommand("INQUIRE " + expression); //$NON-NLS-1$
	}

	/**
	 * Executes an inquire command on the server. Optionally closes the connection immediately after succesfull execution of the query. If the result indicates any kind of error, then the connection will be closed.
	 * 
	 * @param expression
	 *            The expression of the relational algebra to be inquired.
	 * @return response A DBResponse holding the value of the queried expression.
	 * @throws DBException
	 *             If the connection has already been closed, no transaction was started, or an error was found.
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server
	 * @deprecated replace this method by the execQueryAndEndTransaction() method of the DBTransaction object obtained from startTransaction()
	 */
	public AbstractRelation execQueryAndEndTransaction (String expression) throws DBException, ConnectionClosedException {
		AbstractRelation rsp = execQuery(expression);
		endTransaction(true);
		return rsp;
	}

	/**
	 * Gets The default autocommit setting for all transactions created on this connection
	 * 
	 * @return The default autocommit setting for all transactions created on this connection
	 */
	public boolean getDefaultTransactionAutoCommit ( ) {
		return initialAutoCommitForTransactions;
	}

	/**
	 * Gets The specification version to use in communications with the server
	 * 
	 * @return The specification version to use in communications with the server
	 */
	public final Version getSpecificationVersionForServer ( ) {
		return specificationVersionForServer;
	}

	/**
	 * Checks whether a connection is open
	 * 
	 * @return true if the connection is open
	 */
	public boolean isOpen ( ) {
		return open;
	}

	/**
	 * Rolls back the uncommitted updates on the connection. After completion, the connection will be closed.
	 * 
	 * @throws DBException
	 *             If the connection has already been closed, no transaction was started, or an error was found.
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server
	 * @deprecated Use the rollback() method of the DBTransaction object obtained from the connection instead
	 */
	public void rollback ( ) throws DBException, ConnectionClosedException {
		if (!open) {
			throw new IllegalConnectionStateException(Messages.getString("DBConnection.Closed")); //$NON-NLS-1$
		}
		if (transaction == null) {
			throw new IllegalConnectionStateException(Messages.getString("DBConnection.NoTransaction")); //$NON-NLS-1$
		}

		if (!transaction.isAutoCommit()) {
			try {
				CommitRollbackMessageType commitRollbackMessageType = (CommitRollbackMessageType) ServerMessageTypes.getInstance().getServerMessageTypeForSiraPriseVersion(CommitRollbackMessageType.MESSAGETYPEID, specificationVersionForServer);
				CommitRollbackMessage commitRollbackMessage = commitRollbackMessageType.message(false);
				commitRollbackMessage.sendMessage(socketOutputStream, signingProtocol, signer, cryptoProtocol);
			} catch (IOException e) {
				// IO error in the send ===> regard connection as "become unusable" ===>
				transaction = null;
				closeSockets();
				throw new ConnectionClosedException(e);
			} catch (CommunicationProtocolException e) {
				transaction = null;
				// throw new DBException(e);
				closeSockets();
				throw new ConnectionClosedException(e);
			} catch (NotFoundException e) {
				transaction = null;
				// throw new DBException(e);
				closeSockets();
				throw new ConnectionClosedException(e);
			}

			// And now anxiously await the Committed message
			ServerMessage serverMessage;
			try {
				serverMessage = ServerMessage.readMessage(socketInputStream, cryptoProtocol, null, null);
			} catch (IOException e) {
				transaction = null;
				closeSockets();
				throw new ConnectionClosedException(e);
			} catch (ErrorMessageException e) {
				transaction = null;
				// throw new DBException(e);
				closeSockets();
				throw new ConnectionClosedException(e);
			} catch (CommunicationProtocolException e) {
				transaction = null;
				// throw new DBException(e);
				closeSockets();
				throw new ConnectionClosedException(e);
			}

			if (!(serverMessage instanceof CommittedMessage)) {
				throw new DBException(MyMessageFormat.format(Messages.getString("DBConnection.UnexpectedMessageType"), new String[] { serverMessage.getClass().getName(), Long.toHexString(TransactionStartedMessageType.MESSAGETYPEID) })); //$NON-NLS-1$
			}
		}
	}

	/**
	 * Sets The default autocommit setting for all transactions created on this connection
	 * 
	 * @param autoCommit
	 *            The new default autocommit setting for all transactions created on this connection
	 */
	public void setDefaultTransactionAutoCommit (boolean autoCommit) {
		this.initialAutoCommitForTransactions = autoCommit;
	}

	/**
	 * Starts an anonymous transaction on the connection with the default autocommit, transactionmode and ddl capture settings for transactions created on this connection
	 * 
	 * @return The started transaction
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server
	 * @throws ErrorMessageException
	 *             If the response obtained from the server was not a confirmation, but an error message instead
	 */
	public final DBTransaction startTransaction ( ) throws ConnectionClosedException, ErrorMessageException {
		return startTransaction("", false, "", new byte[] {}, initialAutoCommitForTransactions, defaultTransactionMode, defaultDDLCapture); //$NON-NLS-1$//$NON-NLS-2$
	}

	/**
	 * Starts an anonymous transaction on the connection with the given autocommit setting
	 * 
	 * @param autoCommit
	 *            The autoCommit setting for the transaction
	 * @return The started transaction
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server
	 * @throws ErrorMessageException
	 *             If the response obtained from the server was not a confirmation, but an error message instead
	 */
	public final DBTransaction startTransaction (boolean autoCommit) throws ConnectionClosedException, ErrorMessageException {
		return startTransaction("", false, "", new byte[] {}, autoCommit, defaultTransactionMode, defaultDDLCapture); //$NON-NLS-1$//$NON-NLS-2$
	}

	/**
	 * Starts an anonymous transaction on the connection with the given autocommit and ddl capture settings
	 * 
	 * @param autoCommit
	 *            The autoCommit setting for the transaction
	 * @param ddlCapture
	 *            Flag indicating whether DDL capture is to be used in this transaction
	 * @return The started transaction
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server
	 * @throws ErrorMessageException
	 *             If the response obtained from the server was not a confirmation, but an error message instead
	 */
	public final DBTransaction startTransaction (boolean autoCommit, DDLCapture ddlCapture) throws ConnectionClosedException, ErrorMessageException {
		return startTransaction("", false, "", new byte[] {}, autoCommit, defaultTransactionMode, ddlCapture); //$NON-NLS-1$//$NON-NLS-2$
	}

	/**
	 * Starts an anonymous transaction in the given transaction mode on the connection with the given autocommit setting
	 * 
	 * @param autoCommit
	 *            The autoCommit setting for the transaction
	 * @param transactionMode
	 *            The transaction mode
	 * @return The started transaction
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server
	 * @throws ErrorMessageException
	 *             If the response obtained from the server was not a confirmation, but an error message instead
	 */
	public final DBTransaction startTransaction (boolean autoCommit, TransactionMode transactionMode) throws ConnectionClosedException, ErrorMessageException {
		return startTransaction("", false, "", new byte[] {}, autoCommit, transactionMode, defaultDDLCapture); //$NON-NLS-1$//$NON-NLS-2$
	}

	/**
	 * Starts an anonymous transaction in the given transaction mode on the connection with the given autocommit setting
	 * 
	 * @param autoCommit
	 *            The autoCommit setting for the transaction
	 * @param transactionMode
	 *            The transaction mode
	 * @param ddlCapture
	 *            Flag indicating whether DDL capture is to be used in this transaction
	 * @return The started transaction
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server
	 * @throws ErrorMessageException
	 *             If the response obtained from the server was not a confirmation, but an error message instead
	 */
	public final DBTransaction startTransaction (boolean autoCommit, TransactionMode transactionMode, DDLCapture ddlCapture) throws ConnectionClosedException, ErrorMessageException {
		return startTransaction("", false, "", new byte[] {}, autoCommit, transactionMode, ddlCapture); //$NON-NLS-1$//$NON-NLS-2$
	}

	/**
	 * Starts an anonymous transaction on the connection with the default autocommit and transactionmode settings for transactions created on this connection, and the given ddl capture setting
	 * 
	 * @param ddlCapture
	 *            Flag indicating whether DDL capture is to be used in this transaction
	 * 
	 * @return The started transaction
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server
	 * @throws ErrorMessageException
	 *             If the response obtained from the server was not a confirmation, but an error message instead
	 */
	public final DBTransaction startTransaction (DDLCapture ddlCapture) throws ConnectionClosedException, ErrorMessageException {
		return startTransaction("", false, "", new byte[] {}, initialAutoCommitForTransactions, defaultTransactionMode, ddlCapture); //$NON-NLS-1$//$NON-NLS-2$
	}

	/**
	 * Starts a transaction on the connection with the default autocommit setting for transactions created on this connection
	 * 
	 * @param userID
	 *            The identification of the user starting the transaction
	 * @param userAuthenticatedByClient
	 *            true if the user has been authenticated by the client
	 * @param signatureAlgorithm
	 *            The name of the signature algorithm that can be used by the server to verify the authenticity of the userID
	 * @param signature
	 *            the user's signature
	 * @return The started transaction
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server
	 * @throws ErrorMessageException
	 *             If the response obtained from the server was not a confirmation, but an error message instead
	 */
	public final DBTransaction startTransaction (String userID, boolean userAuthenticatedByClient, String signatureAlgorithm, byte[] signature) throws ConnectionClosedException, ErrorMessageException {
		return startTransaction(userID, userAuthenticatedByClient, signatureAlgorithm, signature, initialAutoCommitForTransactions, defaultTransactionMode, defaultDDLCapture);
	}

	/**
	 * Starts a transaction on the connection with the given autoCommit setting
	 * 
	 * @param userID
	 *            The identification of the user starting the transaction
	 * @param userAuthenticatedByClient
	 *            true if the user has been authenticated by the client
	 * @param signatureAlgorithm
	 *            The name of the signature algorithm that can be used by the server to verify the authenticity of the userID
	 * @param signature
	 *            the user's signature
	 * @param autoCommit
	 *            The autoCommit setting for the transaction
	 * @param transactionMode
	 *            the mode for the transaction
	 * @return The started transaction
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server
	 * @throws ErrorMessageException
	 *             If the response obtained from the server was not a confirmation, but an error message instead
	 */
	public final DBTransaction startTransaction (String userID, boolean userAuthenticatedByClient, String signatureAlgorithm, byte[] signature, boolean autoCommit, TransactionMode transactionMode) throws ConnectionClosedException, ErrorMessageException {
		return startTransaction(userID, userAuthenticatedByClient, signatureAlgorithm, signature, autoCommit, transactionMode, defaultDDLCapture);
	}

	/**
	 * Starts a transaction on the connection with the given autoCommit setting
	 * 
	 * @param userID
	 *            The identification of the user starting the transaction
	 * @param userAuthenticatedByClient
	 *            true if the user has been authenticated by the client
	 * @param signatureAlgorithm
	 *            The name of the signature algorithm that can be used by the server to verify the authenticity of the userID
	 * @param signature
	 *            the user's signature
	 * @param autoCommit
	 *            The autoCommit setting for the transaction
	 * @param transactionMode
	 *            the mode for the transaction
	 * @param captureDDL
	 *            Flag indicating whether DDL capture is to be used in this transaction
	 * @return The started transaction
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server
	 * @throws ErrorMessageException
	 *             If the response obtained from the server was not a confirmation, but an error message instead
	 */
	public final DBTransaction startTransaction (String userID, boolean userAuthenticatedByClient, String signatureAlgorithm, byte[] signature, boolean autoCommit, TransactionMode transactionMode, DDLCapture captureDDL) throws ConnectionClosedException, ErrorMessageException {
		if (!open) {
			throw new IllegalConnectionStateException(Messages.getString("DBConnection.Closed")); //$NON-NLS-1$
		}
		if (transaction != null) {
			throw new IllegalConnectionStateException(MyMessageFormat.format(Messages.getString("DBConnection.DuplicateTransaction"), new String[] { transaction.toString() })); //$NON-NLS-1$
		}

		try {
			((StartTransactionMessageType) ServerMessageTypes.getInstance().getServerMessageTypeForSiraPriseVersion(StartTransactionMessageType.MESSAGETYPEID, specificationVersionForServer)).message(userID, userAuthenticatedByClient, signatureAlgorithm, signature, transactionMode.isReadOnly(), autoCommit, captureDDL).sendMessage(socketOutputStream, signingProtocol, signer, cryptoProtocol);

			// And now anxiously await the TransactionStarted message
			ServerMessage serverMessage = ServerMessage.readMessage(socketInputStream, cryptoProtocol, null, null);

			if (!(serverMessage instanceof TransactionStartedMessage)) {
				throw new CommunicationProtocolException(serverMessage.getClass().getName(), TransactionStartedMessageType.class.getName());
			}

			return transaction = new DBTransaction(userID, autoCommit, this, ((TransactionStartedMessage) serverMessage).getTransactionID(), transactionMode);
		} catch (IOException e1) {
			transaction = null;
			closeSockets();
			throw new ConnectionClosedException(e1);
		} catch (CommunicationProtocolException e1) {
			transaction = null;
			closeSockets();
			throw new ConnectionClosedException(e1);
		} catch (NotFoundException e) {
			transaction = null;
			closeSockets();
			throw new ConnectionClosedException(new CommunicationProtocolException(StartTransactionMessageType.MESSAGETYPEID, specificationVersionForServer.getFullVersion()));
		}
	}

	/**
	 * Starts a transaction on the connection with the default autocommit setting for transactions created on this connection
	 * 
	 * @param userID
	 *            The identification of the user starting the transaction
	 * @param userAuthenticatedByClient
	 *            true if the user has been authenticated by the client
	 * @param signatureAlgorithm
	 *            The name of the signature algorithm that can be used by the server to verify the authenticity of the userID
	 * @param signature
	 *            the user's signature
	 * @param ddlCapture
	 *            Flag indicating whether DDL capture is to be used in this transaction
	 * @return The started transaction
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server
	 * @throws ErrorMessageException
	 *             If the response obtained from the server was not a confirmation, but an error message instead
	 */
	public final DBTransaction startTransaction (String userID, boolean userAuthenticatedByClient, String signatureAlgorithm, byte[] signature, DDLCapture ddlCapture) throws ConnectionClosedException, ErrorMessageException {
		return startTransaction(userID, userAuthenticatedByClient, signatureAlgorithm, signature, initialAutoCommitForTransactions, defaultTransactionMode, ddlCapture);
	}

	/**
	 * Starts a transaction on the connection with the default autocommit setting for transactions created on this connection
	 * 
	 * @param userID
	 *            The identification of the user starting the transaction
	 * @param userAuthenticatedByClient
	 *            true if the user has been authenticated by the client
	 * @param signatureAlgorithm
	 *            The name of the signature algorithm that can be used by the server to verify the authenticity of the userID
	 * @param signature
	 *            the user's signature
	 * @param transactionMode
	 *            The transaction mode
	 * @return The started transaction
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server
	 * @throws ErrorMessageException
	 *             If the response obtained from the server was not a confirmation, but an error message instead
	 */
	public final DBTransaction startTransaction (String userID, boolean userAuthenticatedByClient, String signatureAlgorithm, byte[] signature, TransactionMode transactionMode) throws ConnectionClosedException, ErrorMessageException {
		return startTransaction(userID, userAuthenticatedByClient, signatureAlgorithm, signature, initialAutoCommitForTransactions, transactionMode, defaultDDLCapture);
	}

	/**
	 * Starts a transaction on the connection with the default autocommit setting for transactions created on this connection
	 * 
	 * @param userID
	 *            The identification of the user starting the transaction
	 * @param userAuthenticatedByClient
	 *            true if the user has been authenticated by the client
	 * @param signatureAlgorithm
	 *            The name of the signature algorithm that can be used by the server to verify the authenticity of the userID
	 * @param signature
	 *            the user's signature
	 * @param transactionMode
	 *            The transaction mode
	 * @param ddlCapture
	 *            Flag indicating whether DDL capture is to be used in this transaction
	 * @return The started transaction
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server
	 * @throws ErrorMessageException
	 *             If the response obtained from the server was not a confirmation, but an error message instead
	 */
	public final DBTransaction startTransaction (String userID, boolean userAuthenticatedByClient, String signatureAlgorithm, byte[] signature, TransactionMode transactionMode, DDLCapture ddlCapture) throws ConnectionClosedException, ErrorMessageException {
		return startTransaction(userID, userAuthenticatedByClient, signatureAlgorithm, signature, initialAutoCommitForTransactions, transactionMode, ddlCapture);
	}

	/**
	 * Starts a transaction on the connection for the named user and with the default autocommit setting for transactions created on this connection. The user identity signature is selected from the set of signatures that are supported both by the server and the client JVM, and for which the given Signer is able to provide a valid private key.
	 * 
	 * @param userID
	 *            The identification of the user starting the transaction
	 * @param userIDSigner
	 *            The Signer object that will be called upon to compute the needed signature
	 * @return The started transaction
	 * @throws DBException
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server
	 * @throws ErrorMessageException
	 *             If the response obtained from the server was not a confirmation, but an error message instead
	 */
	public final DBTransaction startTransaction (String userID, Signer userIDSigner) throws DBException, ConnectionClosedException, ErrorMessageException {
		return startTransaction(userID, userIDSigner, initialAutoCommitForTransactions);
	}

	/**
	 * Starts a transaction on the connection for the named user and with the default autocommit setting for transactions created on this connection. The user identity signature is selected from the set of signatures that are supported both by the server and the client JVM, and for which the given Signer is able to provide a valid private key.
	 * 
	 * @param userID
	 *            The identification of the user starting the transaction
	 * @param userIDSigner
	 *            The Signer object that will be called upon to compute the needed signature
	 * @param autoCommit
	 *            The autoCommit setting for the transaction
	 * @return The started transaction
	 * @throws DBException
	 * @throws ConnectionClosedException
	 * @throws ErrorMessageException
	 */
	public final DBTransaction startTransaction (String userID, Signer userIDSigner, boolean autoCommit) throws DBException, ConnectionClosedException, ErrorMessageException {
		String signatureAlgorithmName = ""; //$NON-NLS-1$
		byte[] signature = null;

		Iterator<String> i_alternativeSigningProtocols = alternativeSigningProtocols.iterator();
		boolean signatureComputed = false;
		while (i_alternativeSigningProtocols.hasNext() & !signatureComputed) {
			signatureAlgorithmName = i_alternativeSigningProtocols.next();
			try {
				Signature signatureAlgorithm = Signature.getInstance(signatureAlgorithmName);
				// PrivateKey privateKey2 = signer.getPrivateKey(signatureAlgorithmName);
				// Prepare the bytes that are to be signed
				ByteArrayOutputStream w = new ByteArrayOutputStream();
				MyDataOutputStream.writeSmallUTF(userID, new DataOutputStream(w));
				byte[] signMessage = w.toByteArray();
				signature = userIDSigner.sign(signatureAlgorithm, signMessage);
				signatureComputed = true;
			} catch (NoSuchAlgorithmException shouldBeImpoosible) {
				// Try next algorithm if there is one.
			} catch (InvalidKeyException e) {
				// signer privdied a key that turned out to be invalid. Try next algorithm if there is one.
			} catch (NotFoundException e) {
				// signer says he cannot provide a key for this algorithm. Try next algorithm if there is one.
			} catch (IOException e) {
				throw new DBException(e);
			} catch (SignatureException e) {
				throw new DBException(e);
			}
		}

		if (!signatureComputed) {
			throw new DBException(MyMessageFormat.format(Messages.getString("DBConnection.UserIdentitySigningFailed"), new Object[] { alternativeSigningProtocols })); //$NON-NLS-1$
		}

		return startTransaction(userID, false, signatureAlgorithmName, signature, autoCommit, defaultTransactionMode);
	}

	/**
	 * Starts a transaction in the given transaction mode on the connection for the named user and with the default autocommit setting for transactions created on this connection. The user identity signature is selected from the set of signatures that are supported both by the server and the client JVM, and for which the given Signer is able to provide a valid private key.
	 * 
	 * @param userID
	 *            The identification of the user starting the transaction
	 * @param userIDSigner
	 *            The Signer object that will be called upon to compute the needed signature
	 * @param autoCommit
	 *            The autoCommit setting for the transaction
	 * @param transactionMode
	 *            The transaction mode
	 * @return The started transaction
	 * @throws DBException
	 * @throws ConnectionClosedException
	 * @throws ErrorMessageException
	 */
	public final DBTransaction startTransaction (String userID, Signer userIDSigner, boolean autoCommit, TransactionMode transactionMode) throws DBException, ConnectionClosedException, ErrorMessageException {
		String signatureAlgorithmName = ""; //$NON-NLS-1$
		byte[] signature = null;

		Iterator<String> i_alternativeSigningProtocols = alternativeSigningProtocols.iterator();
		boolean signatureComputed = false;
		while (i_alternativeSigningProtocols.hasNext() & !signatureComputed) {
			signatureAlgorithmName = i_alternativeSigningProtocols.next();
			try {
				Signature signatureAlgorithm = Signature.getInstance(signatureAlgorithmName);
				// PrivateKey privateKey2 = signer.getPrivateKey(signatureAlgorithmName);
				// Prepare the bytes that are to be signed
				ByteArrayOutputStream w = new ByteArrayOutputStream();
				MyDataOutputStream.writeSmallUTF(userID, new DataOutputStream(w));
				byte[] signMessage = w.toByteArray();
				signature = userIDSigner.sign(signatureAlgorithm, signMessage);
				signatureComputed = true;
			} catch (NoSuchAlgorithmException shouldBeImpoosible) {
				// Try next algorithm if there is one.
			} catch (InvalidKeyException e) {
				// signer privdied a key that turned out to be invalid. Try next algorithm if there is one.
			} catch (NotFoundException e) {
				// signer says he cannot provide a key for this algorithm. Try next algorithm if there is one.
			} catch (IOException e) {
				throw new DBException(e);
			} catch (SignatureException e) {
				throw new DBException(e);
			}
		}

		if (!signatureComputed) {
			throw new DBException(MyMessageFormat.format(Messages.getString("DBConnection.UserIdentitySigningFailed"), new Object[] { alternativeSigningProtocols })); //$NON-NLS-1$
		}

		return startTransaction(userID, false, signatureAlgorithmName, signature, autoCommit, transactionMode);
	}

	/**
	 * Starts a transaction in the given read-only mode on the connection for the named user and with the default autocommit setting for transactions created on this connection. The user identity signature is selected from the set of signatures that are supported both by the server and the client JVM, and for which the given Signer is able to provide a valid private key.
	 * 
	 * @param userID
	 *            The identification of the user starting the transaction
	 * @param userIDSigner
	 *            The Signer object that will be called upon to compute the needed signature
	 * @param transactionMode
	 *            The transaction mode
	 * @return The started transaction
	 * @throws DBException
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server
	 * @throws ErrorMessageException
	 *             If the response obtained from the server was not a confirmation, but an error message instead
	 */
	public final DBTransaction startTransaction (String userID, Signer userIDSigner, TransactionMode transactionMode) throws DBException, ConnectionClosedException, ErrorMessageException {
		return startTransaction(userID, userIDSigner, transactionMode);
	}

	/**
	 * Starts an anonymous transaction in the given transaction mode on the connection with the default autocommit setting for transactions created on this connection
	 * 
	 * @param transactionMode
	 *            The transaction mode
	 * 
	 * @return The started transaction
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server
	 * @throws ErrorMessageException
	 *             If the response obtained from the server was not a confirmation, but an error message instead
	 */
	public final DBTransaction startTransaction (TransactionMode transactionMode) throws ConnectionClosedException, ErrorMessageException {
		return startTransaction("", false, "", new byte[] {}, initialAutoCommitForTransactions, transactionMode, defaultDDLCapture); //$NON-NLS-1$//$NON-NLS-2$
	}

	/**
	 * Starts an anonymous transaction in the given transaction mode on the connection with the default autocommit setting for transactions created on this connection
	 * 
	 * @param transactionMode
	 *            The transaction mode
	 * @param ddlCapture
	 *            Flag indicating whether DDL capture is to be used in this transaction
	 * @return The started transaction
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server
	 * @throws ErrorMessageException
	 *             If the response obtained from the server was not a confirmation, but an error message instead
	 */
	public final DBTransaction startTransaction (TransactionMode transactionMode, DDLCapture ddlCapture) throws ConnectionClosedException, ErrorMessageException {
		return startTransaction("", false, "", new byte[] {}, initialAutoCommitForTransactions, transactionMode, ddlCapture); //$NON-NLS-1$//$NON-NLS-2$
	}
}