/*
 * Created on 30-jul-2008
 */
package be.SIRAPRISE.client;

import java.util.*;

import be.SIRAPRISE.client.jsba.DBObject;
import be.SIRAPRISE.client.jsba.UpdatableDBObject;

/**
 * A DBTransaction represents the transaction that is active on some DBConnection.
 * 
 * @author Erwin Smout
 */
public class DBTransaction {

	/**
	 * Flag indicating whether all DML statements executed in the transaction should be automatically committed
	 */
	private boolean autoCommit;

	/**
	 * The DBConnection on which the transaction is running
	 */
	private DBConnection dbConnection;

	/**
	 * The nested transaction that has been started inside this transaction
	 */
	private DBTransaction nestedTransaction;

	/**
	 * The "parent" transaction of this nested transaction, or null if this is not a nested transaction
	 */
	private DBTransaction parentTransaction;

	/**
	 * The numerical server-side ID of the started transaction, or -1 if the server is pre 1.2
	 */
	private long transactionID;

	/**
	 * Flag indicating whether the transaction is read-only
	 */
	private TransactionMode transactionMode;

	/**
	 * The identification of the user owning the transaction
	 */
	private String userID;

	/**
	 * Creates the DBTransaction as a nested transaction in some "parent" transaction
	 * 
	 * @param parentTransaction
	 *            The "parent" transaction
	 * @param autoCommit
	 *            Flag indicating whether all DML statements executed in the transaction should be automatically committed
	 * @param transactionID
	 *            The numerical server-side ID of the started transaction, or -1 if the server is pre 1.2
	 * @param transactionMode
	 *            Flag indicating whether the transaction is read-only
	 */
	DBTransaction (DBTransaction parentTransaction, boolean autoCommit, long transactionID, TransactionMode transactionMode) {
		this.parentTransaction = parentTransaction;
		this.userID = parentTransaction.userID;
		this.dbConnection = parentTransaction.dbConnection;
		this.autoCommit = autoCommit;
		this.transactionID = transactionID;
		this.transactionMode = transactionMode;
	}

	/**
	 * Creates the transaction, setting the userID, the autocommit flag and the DBConnection to which DML requests should be directed.
	 * 
	 * @param userID
	 *            The identification of the user owning the transaction
	 * @param autoCommit
	 *            Flag indicating whether all DML statements executed in the transaction should be automatically committed
	 * @param dbConnection
	 *            The DBConnection on which the transaction is running
	 * @param transactionID
	 *            The numerical server-side ID of the started transaction, or -1 if the server is pre 1.2
	 * @param transactionMode
	 *            Flag indicating whether the transaction is read-only
	 */
	DBTransaction (String userID, boolean autoCommit, DBConnection dbConnection, long transactionID, TransactionMode transactionMode) {
		this.userID = userID;
		this.autoCommit = autoCommit;
		this.dbConnection = dbConnection;
		this.transactionID = transactionID;
		this.transactionMode = transactionMode;
	}

	/**
	 * 
	 */
	private void checkConnectionPresent ( ) {
		if (dbConnection == null) {
			throw new IllegalTransactionStateException();
		}
	}

	/**
	 * @param serverCommand
	 *            the ServerCommand object representing the server command to be executed
	 * @return the result of the command.
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server
	 * @throws ErrorMessageException
	 *             If the response obtained from the server is not a commit confirmation, but an error response instead.
	 */
	private AbstractRelation execDmlCommandWithoutConnectionCheck (ServerCommand serverCommand) throws ConnectionClosedException, ErrorMessageException {
		try {
			return dbConnection.verifyStateAndSendExecDmlCommandMessage(serverCommand);
		} catch (ConnectionClosedException e) {
			dbConnection = null;
			throw e;
			// } catch (ErrorMessageException e) {
			// ServerCommand message can only be served by servers as of 1.4
			// if (dbConnection.getSpecificationVersionForServer().isBefore(Version.ONE_TWO)) {
			// dbConnection = null;
			// }
			// throw e;
		}
	}

	/**
	 * Sends a command to the server and returns the result obtained.
	 * 
	 * @param cmd
	 *            the command to be executed, in textual form.
	 * @return the result of the command.
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server
	 * @throws ErrorMessageException
	 *             If the response obtained from the server is not a commit confirmation, but an error response instead.
	 */
	private AbstractRelation execDmlCommandWithoutConnectionCheck (String cmd) throws ConnectionClosedException, ErrorMessageException {
		try {
			AbstractRelation result = dbConnection.verifyStateAndSendExecDmlCommandMessage(cmd);

			if (autoCommit && dbConnection.getSpecificationVersionForServer().isBefore(Version.ONE_TWO) && !((cmd.length() > 7 && cmd.substring(0, 8).equalsIgnoreCase("ROLLBACK")) || (cmd.length() > 5 && cmd.substring(0, 6).equalsIgnoreCase("COMMIT")))) { //$NON-NLS-1$//$NON-NLS-2$
				dbConnection.sendCommitMessage();
			}

			return result;
		} catch (ConnectionClosedException e) {
			dbConnection = null;
			throw e;
		} catch (ErrorMessageException e) {
			if (dbConnection.getSpecificationVersionForServer().isBefore(Version.ONE_TWO)) {
				dbConnection = null;
			}
			throw e;
		}
	}

	/**
	 * Commits this non-autocommit transaction. If the transaction is auto-commit, this method is a NOP.
	 * 
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server, or if the commit failed.
	 */
	public final void commit ( ) throws ConnectionClosedException {
		checkConnectionPresent();

		if (!autoCommit || dbConnection.getSpecificationVersionForServer().isBefore(Version.ONE_TWO)) {
			try {
				dbConnection.verifyStateAndSendCommitMessage();
			} catch (ConnectionClosedException e) {
				dbConnection = null;
				throw e;
			}
		}
	}

	/**
	 * Sends a query expression to the server for compilation-and-caching.
	 * 
	 * @param query
	 *            the query to be compiled-and-cached, in textual form.
	 * @return the result of the compilation. That result is a relation with the following attributes :
	 *         <ul>
	 *         <li>QUERYID(LONG(...)) The internal ID of the compiled query, by which the query can later be referenced</li>
	 *         <li>HEADING(RELATION(...)) The description of the heading of the relation as it will be output by the query</li>
	 *         <li>PARMS(RELATION(...)) The description of all parameters required by the query</li>
	 *         </ul>
	 *         The heading of the 'HEADING' relation is as follows :
	 *         <ul>
	 *         <li>ATTRIBUTENAME(NAME(...)) The name of an attribute in the output of the query</li>
	 *         <li>TYPENAME(NAME(...)) The name of the type of that attribute in the output of the query</li>
	 *         </ul>
	 *         The heading of the 'PARMS' relation is as follows :
	 *         <ul>
	 *         <li>PARAMETERNAME(NAME(...)) The name of a parameter required by the query</li>
	 *         <li>TYPENAME(NAME(...)) The name of the type of that parameter</li>
	 *         </ul>
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server
	 * @throws ErrorMessageException
	 *             If the response obtained from the server is not a commit confirmation, but an error response instead.
	 */
	public final AbstractRelation compileAndCacheQuery (String query) throws ConnectionClosedException, ErrorMessageException {
		return execDmlCommand("CACHE " + query); //$NON-NLS-1$
	}

	/**
	 * Sends a query expression to the server for verification only.
	 * 
	 * @param query
	 *            the query to be verified, in textual form.
	 * @return the result of the verification. That result is a relation with the following attributes :
	 *         <ol>
	 *         <li>HEADING(RELATION(...)) The description of the heading of the relation as it will be output by the query</li>
	 *         <li>PARMS(RELATION(...)) The description of all parameters required by the query</li>
	 *         </ol>
	 *         The heading of the 'HEADING' relation is as follows :
	 *         <ol>
	 *         <li>ATTRIBUTENAME(NAME(...)) The name of an attribute in the output of the query</li>
	 *         <li>TYPENAME(NAME(...)) The name of the type of that attribute in the output of the query</li>
	 *         </ol>
	 *         The heading of the 'PARMS' relation is as follows :
	 *         <ol>
	 *         <li>PARAMETERNAME(NAME(...)) The name of a parameter required by the query</li>
	 *         <li>TYPENAME(NAME(...)) The name of the type of that parameter</li>
	 *         </ol>
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server
	 * @throws ErrorMessageException
	 *             If the response obtained from the server is not a commit confirmation, but an error response instead.
	 */
	public final AbstractRelation compileQuery (String query) throws ConnectionClosedException, ErrorMessageException {
		return execDmlCommand("COMPILE " + query); //$NON-NLS-1$
	}

	/**
	 * Adds an entry in the named relvar (in the database the transaction is connected to), for the given object. The tuple that is registered in the database is constructed by invoking all the object's methods that satisfy the following properties :
	 * <ul>
	 * <li>The name of the method is the word "get" (all lowercase), followed by the capitalized relvar name (i.e. first character of the relvar name uppercase, all remaining characters of the relvar name lowercase), followed by a capitalized attribute name</LI>
	 * <li>The method is a public method that takes no arguments</LI>
	 * <li>The return type of the method is java.lang.String</LI>
	 * </UL>
	 * 
	 * @param relvarName
	 *            The name of the relvar into which a tuple is to be added for the given object
	 * @param o
	 *            The object from which the tuple to be registered in the named relvar is to be drawn.
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server
	 * @throws ErrorMessageException
	 *             If the response obtained from the server is not a commit confirmation, but an error response instead.
	 */
	public final void dbAdd (String relvarName, DBObject o) throws ConnectionClosedException, ErrorMessageException {
		execDmlCommand(dbAddCommand(relvarName, o));
	}

	/**
	 * Adds an entry in the named relvar (in the database the transaction is connected to), for the given object. The tuple that is registered in the database is constructed by invoking all the object's methods that satisfy the following properties :
	 * <ul>
	 * <li>The name of the method is the word "get" (all lowercase), followed by the capitalized relvar name (i.e. first character of the relvar name uppercase, all remaining characters of the relvar name lowercase), followed by a capitalized attribute name</LI>
	 * <li>The method is a public method that takes no arguments</LI>
	 * <li>The return type of the method is java.lang.String</LI>
	 * </UL>
	 * 
	 * @param relvarName
	 *            The name of the relvar into which a tuple is to be added for the given object
	 * @param o
	 *            The object from which the tuple to be registered in the named relvar is to be drawn.
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server
	 * @throws ErrorMessageException
	 *             If the response obtained from the server is not a commit confirmation, but an error response instead.
	 */
	public final void dbAddAndEndTransaction (String relvarName, DBObject o) throws ConnectionClosedException, ErrorMessageException {
		try {
			dbAdd(relvarName, o);
			end(true);
		} catch (ErrorMessageException e) {
			end(false);
			throw e;
		} catch (RuntimeException e) {
			end(false);
			throw e;
		} catch (Error e) {
			end(false);
			throw e;
		}
	}

	/**
	 * Gets the command for adding an entry in the named relvar (in the database the transaction is connected to), for the given object. The tuple that is registered in the database is constructed by invoking all the object's methods that satisfy the following properties :
	 * <ul>
	 * <li>The name of the method is the word "get" (all lowercase), followed by the capitalized relvar name (i.e. first character of the relvar name uppercase, all remaining characters of the relvar name lowercase), followed by a capitalized attribute name</LI>
	 * <li>The method is a public method that takes no arguments</LI>
	 * <li>The return type of the method is java.lang.String</LI>
	 * </UL>
	 * 
	 * @param relvarName
	 *            The name of the relvar into which a tuple is to be added for the given object
	 * @param o
	 *            The object from which the tuple to be registered in the named relvar is to be drawn.
	 * @return The add command to be issued for adding an entry in the named relvar with attribute values drawn from the given object
	 */
	public final DmlAddCommand dbAddCommand (String relvarName, DBObject o) {
		return new DmlAddCommand(relvarName, o);
	}

	/**
	 * Asserts an entry in the named relvar (in the database the transaction is connected to), for the given object. The tuple that is registered in the database is constructed by invoking all the object's methods that satisfy the following properties :
	 * <ul>
	 * <li>The name of the method is the word "get" (all lowercase), followed by the capitalized relvar name (i.e. first character of the relvar name uppercase, all remaining characters of the relvar name lowercase), followed by a capitalized attribute name</LI>
	 * <li>The method is a public method that takes no arguments</LI>
	 * <li>The return type of the method is java.lang.String</LI>
	 * </UL>
	 * 
	 * @param relvarName
	 *            The name of the relvar into which a tuple is to be added for the given object
	 * @param o
	 *            The object from which the tuple to be registered in the named relvar is to be drawn.
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server
	 * @throws ErrorMessageException
	 *             If the response obtained from the server is not a commit confirmation, but an error response instead.
	 */
	public final void dbAssert (String relvarName, DBObject o) throws ConnectionClosedException, ErrorMessageException {
		execDmlCommand(dbAssertCommand(relvarName, o));
	}

	/**
	 * Asserts an entry in the named relvar (in the database the transaction is connected to), for the given object. The tuple that is registered in the database is constructed by invoking all the object's methods that satisfy the following properties :
	 * <ul>
	 * <li>The name of the method is the word "get" (all lowercase), followed by the capitalized relvar name (i.e. first character of the relvar name uppercase, all remaining characters of the relvar name lowercase), followed by a capitalized attribute name</LI>
	 * <li>The method is a public method that takes no arguments</LI>
	 * <li>The return type of the method is java.lang.String</LI>
	 * </UL>
	 * 
	 * @param relvarName
	 *            The name of the relvar into which a tuple is to be added for the given object
	 * @param o
	 *            The object from which the tuple to be registered in the named relvar is to be drawn.
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server
	 * @throws ErrorMessageException
	 *             If the response obtained from the server is not a commit confirmation, but an error response instead.
	 */
	public final void dbAssertAndEndTransaction (String relvarName, DBObject o) throws ConnectionClosedException, ErrorMessageException {
		try {
			dbAssert(relvarName, o);
			end(true);
		} catch (ErrorMessageException e) {
			end(false);
			throw e;
		} catch (RuntimeException e) {
			end(false);
			throw e;
		} catch (Error e) {
			end(false);
			throw e;
		}
	}

	/**
	 * Gets the Assert command for asserting an entry in the named relvar (in the database the transaction is connected to), for the given object. The tuple that is registered in the database is constructed by invoking all the object's methods that satisfy the following properties :
	 * <ul>
	 * <li>The name of the method is the word "get" (all lowercase), followed by the capitalized relvar name (i.e. first character of the relvar name uppercase, all remaining characters of the relvar name lowercase), followed by a capitalized attribute name</LI>
	 * <li>The method is a public method that takes no arguments</LI>
	 * <li>The return type of the method is java.lang.String</LI>
	 * </UL>
	 * 
	 * @param relvarName
	 *            The name of the relvar into which a tuple is to be added for the given object
	 * @param o
	 *            The object from which the tuple to be registered in the named relvar is to be drawn.
	 * @return The assert command to be issued for adding an entry in the named relvar with attribute values drawn from the given object
	 */
	public final DmlAssertCommand dbAssertCommand (String relvarName, DBObject o) {
		return new DmlAssertCommand(relvarName, o);
	}

	/**
	 * Deletes an entry in the named relvar (in the database the transaction is connected to), for the given object. The tuple that is registered in the database is constructed by invoking all the object's methods that satisfy the following properties :
	 * <ul>
	 * <li>The name of the method is the word "get" (all lowercase), followed by the capitalized relvar name (i.e. first character of the relvar name uppercase, all remaining characters of the relvar name lowercase), followed by a capitalized attribute name</LI>
	 * <li>The method is a public method that takes no arguments</LI>
	 * <li>The return type of the method is java.lang.String</LI>
	 * </UL>
	 * 
	 * @param relvarName
	 *            The name of the relvar into which a tuple is to be added for the given object
	 * @param o
	 *            The object from which the tuple to be registered in the named relvar is to be drawn.
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server
	 * @throws ErrorMessageException
	 *             If the response obtained from the server is not a commit confirmation, but an error response instead.
	 */
	public final void dbDelete (String relvarName, DBObject o) throws ConnectionClosedException, ErrorMessageException {
		execDmlCommand(dbDeleteCommand(relvarName, o));
	}

	/**
	 * Deletes an entry in the named relvar (in the database the transaction is connected to), for the given object. The tuple that is registered in the database is constructed by invoking all the object's methods that satisfy the following properties :
	 * <ul>
	 * <li>The name of the method is the word "get" (all lowercase), followed by the capitalized relvar name (i.e. first character of the relvar name uppercase, all remaining characters of the relvar name lowercase), followed by a capitalized attribute name</LI>
	 * <li>The method is a public method that takes no arguments</LI>
	 * <li>The return type of the method is java.lang.String</LI>
	 * </UL>
	 * After the deletion has been carried out, the transaction is ended
	 * 
	 * @param relvarName
	 *            The name of the relvar into which a tuple is to be added for the given object
	 * @param o
	 *            The object from which the tuple to be registered in the named relvar is to be drawn.
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server
	 * @throws ErrorMessageException
	 *             If the response obtained from the server is not a commit confirmation, but an error response instead.
	 */
	public final void dbDeleteAndEndTransaction (String relvarName, DBObject o) throws ConnectionClosedException, ErrorMessageException {
		try {
			dbDelete(relvarName, o);
			end(true);
		} catch (ErrorMessageException e) {
			end(false);
			throw e;
		} catch (RuntimeException e) {
			end(false);
			throw e;
		} catch (Error e) {
			end(false);
			throw e;
		}
	}

	/**
	 * Gets the command for deleting the tuple in the named relvar (in the database the transaction is connected to), for the given object. The tuple that is registered in the database is constructed by invoking all the object's methods that satisfy the following properties :
	 * <ul>
	 * <li>The name of the method is the word "get" (all lowercase), followed by the capitalized relvar name (i.e. first character of the relvar name uppercase, all remaining characters of the relvar name lowercase), followed by a capitalized attribute name</LI>
	 * <li>The method is a public method that takes no arguments</LI>
	 * <li>The return type of the method is java.lang.String</LI>
	 * </UL>
	 * 
	 * @param relvarName
	 *            The name of the relvar into which a tuple is to be added for the given object
	 * @param o
	 *            The object from which the tuple to be registered in the named relvar is to be drawn.
	 * @return the command for deleting the tuple in the named relvar (in the database the transaction is connected to), for the given object.
	 */
	public final DmlDeleteCommand dbDeleteCommand (String relvarName, DBObject o) {
		return new DmlDeleteCommand(relvarName, o);
	}

	/**
	 * Unasserts an entry in the named relvar (in the database the transaction is connected to), for the given object. The tuple that is registered in the database is constructed by invoking all the object's methods that satisfy the following properties :
	 * <ul>
	 * <li>The name of the method is the word "get" (all lowercase), followed by the capitalized relvar name (i.e. first character of the relvar name uppercase, all remaining characters of the relvar name lowercase), followed by a capitalized attribute name</LI>
	 * <li>The method is a public method that takes no arguments</LI>
	 * <li>The return type of the method is java.lang.String</LI>
	 * </UL>
	 * 
	 * @param relvarName
	 *            The name of the relvar into which a tuple is to be added for the given object
	 * @param o
	 *            The object from which the tuple to be registered in the named relvar is to be drawn.
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server
	 * @throws ErrorMessageException
	 *             If the response obtained from the server is not a commit confirmation, but an error response instead.
	 */
	public final void dbUnAssert (String relvarName, DBObject o) throws ConnectionClosedException, ErrorMessageException {
		execDmlCommand(dbUnAssertCommand(relvarName, o));
	}

	/**
	 * Unasserts an entry in the named relvar (in the database the transaction is connected to), for the given object. The tuple that is registered in the database is constructed by invoking all the object's methods that satisfy the following properties :
	 * <ul>
	 * <li>The name of the method is the word "get" (all lowercase), followed by the capitalized relvar name (i.e. first character of the relvar name uppercase, all remaining characters of the relvar name lowercase), followed by a capitalized attribute name</LI>
	 * <li>The method is a public method that takes no arguments</LI>
	 * <li>The return type of the method is java.lang.String</LI>
	 * </UL>
	 * 
	 * @param relvarName
	 *            The name of the relvar into which a tuple is to be added for the given object
	 * @param o
	 *            The object from which the tuple to be registered in the named relvar is to be drawn.
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server
	 * @throws ErrorMessageException
	 *             If the response obtained from the server is not a commit confirmation, but an error response instead.
	 */
	public final void dbUnAssertAndEndTransaction (String relvarName, DBObject o) throws ConnectionClosedException, ErrorMessageException {
		try {
			dbUnAssert(relvarName, o);
			end(true);
		} catch (ErrorMessageException e) {
			end(false);
			throw e;
		} catch (RuntimeException e) {
			end(false);
			throw e;
		} catch (Error e) {
			end(false);
			throw e;
		}
	}

	/**
	 * Gets the command for unasserting the tuple in the named relvar (in the database the transaction is connected to), for the given object. The tuple that is registered in the database is constructed by invoking all the object's methods that satisfy the following properties :
	 * <ul>
	 * <li>The name of the method is the word "get" (all lowercase), followed by the capitalized relvar name (i.e. first character of the relvar name uppercase, all remaining characters of the relvar name lowercase), followed by a capitalized attribute name</LI>
	 * <li>The method is a public method that takes no arguments</LI>
	 * <li>The return type of the method is java.lang.String</LI>
	 * </UL>
	 * 
	 * @param relvarName
	 *            The name of the relvar into which a tuple is to be added for the given object
	 * @param o
	 *            The object from which the tuple to be registered in the named relvar is to be drawn.
	 * @return the command for unasserting the tuple in the named relvar (in the database the transaction is connected to), for the given object.
	 */
	public final DmlUnassertCommand dbUnAssertCommand (String relvarName, DBObject o) {
		return new DmlUnassertCommand(relvarName, o);
	}

	/**
	 * Updates an entry in the named relvar (in the database the transaction is connected to), for the given object. The tuple that is updated in the database is constructed by invoking all the object's methods that satisfy the following properties :
	 * <ul>
	 * <li>The name of the method is the word "get" (all lowercase), followed by the capitalized relvar name (i.e. first character of the relvar name uppercase, all remaining characters of the relvar name lowercase), followed by a capitalized attribute name</LI>
	 * <li>The method is a public method that takes no arguments</LI>
	 * <li>The return type of the method is java.lang.String</LI>
	 * </UL>
	 * <P>
	 * The updates carried out are determined as follows : For each attribute in the tuple, the value for that attribute (obtained through invoking its getter method on the object) is compared to the pre-update value of that attribute (this value is obtained by invoking the same getter method on the result of the getPreUpdateState() method invocation on the object). If those are equal, nothing is done. If those values are unequal, then an entry is inserted in the update command to make sure the update for the attribute value will happen.
	 * </P>
	 * <P>
	 * After a successful update, the setPreUpdateState() method in o is invoked to make that preUpdateState accurately reflect the current database state.
	 * </P>
	 * 
	 * @param relvarName
	 *            The name of the relvar into which a tuple is to be added for the given object
	 * @param o
	 *            The object from which the tuple to be registered in the named relvar is to be drawn. // *
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server
	 * @throws ErrorMessageException
	 *             If the response obtained from the server is not a commit confirmation, but an error response instead.
	 */
	public final void dbUpdate (String relvarName, UpdatableDBObject o) throws ConnectionClosedException, ErrorMessageException {
		try {
			execDmlCommand(dbUpdateCommand(relvarName, o));
			// o.setPreUpdateState(); now done inside execDmlCommand
		} catch (NoUpdatesException e) {

		}
	}

	/**
	 * Updates an entry in the named relvar (in the database the transaction is connected to), for the given object. The tuple that is updated in the database is constructed by invoking all the object's methods that satisfy the following properties :
	 * <ul>
	 * <li>The name of the method is the word "get" (all lowercase), followed by the capitalized relvar name (i.e. first character of the relvar name uppercase, all remaining characters of the relvar name lowercase), followed by a capitalized attribute name</LI>
	 * <li>The method is a public method that takes no arguments</LI>
	 * <li>The return type of the method is java.lang.String</LI>
	 * </UL>
	 * <P>
	 * The updates carried out are determined as follows : For each attribute in the tuple, the value for that attribute (obtained through invoking its getter method on the object) is compared to the pre-update value of that attribute (this value is obtained by invoking the same getter method on the result of the getPreUpdateState() method invocation on the object). If those are equal, nothing is done. If those values are unequal, then an entry is inserted in the update command to make sure the update for the attribute value will happen.
	 * </P>
	 * <P>
	 * After a successful update, the setPreUpdateState() method in o is invoked to make that preUpdateState accurately reflect the current database state.
	 * </P>
	 * 
	 * @param relvarName
	 *            The name of the relvar into which a tuple is to be added for the given object
	 * @param o
	 *            The object from which the tuple to be registered in the named relvar is to be drawn. // *
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server
	 * @throws ErrorMessageException
	 *             If the response obtained from the server is not a commit confirmation, but an error response instead.
	 */
	public final void dbUpdateAndEndTransaction (String relvarName, UpdatableDBObject o) throws ConnectionClosedException, ErrorMessageException {
		try {
			dbUpdate(relvarName, o);
			end(true);
		} catch (ErrorMessageException e) {
			end(false);
			throw e;
		} catch (RuntimeException e) {
			end(false);
			throw e;
		} catch (Error e) {
			end(false);
			throw e;
		}
	}

	/**
	 * Gets the command for Updating an entry in the named relvar (in the database the transaction is connected to), for the given object. The tuple that is updated in the database is constructed by invoking all the object's methods that satisfy the following properties :
	 * <ul>
	 * <li>The name of the method is the word "get" (all lowercase), followed by the capitalized relvar name (i.e. first character of the relvar name uppercase, all remaining characters of the relvar name lowercase), followed by a capitalized attribute name</LI>
	 * <li>The method is a public method that takes no arguments</LI>
	 * <li>The return type of the method is java.lang.String</LI>
	 * </UL>
	 * The updates carried out are determined as follows : For each attribute in the tuple, the value for that attribute (obtained through invoking its getter method on the object) is compared to the pre-update value of that attribute (this value is obtained by invoking the same getter method on the result of the getPreUpdateState() method invocation on the object). If those are equal, nothing is done. If those values are unequal, then an entry is inserted in the update command to make sure the update for the attribute value will happen.
	 * 
	 * @param relvarName
	 *            The name of the relvar into which a tuple is to be added for the given object
	 * @param o
	 *            The object from which the tuple to be registered in the named relvar is to be drawn. // *
	 * @return the command for Updating an entry in the named relvar (in the database the transaction is connected to), for the given object.
	 * @throws NoUpdatesException
	 *             if no attribute value changes at all are detected
	 */
	public final DmlUpdateCommand dbUpdateCommand (String relvarName, UpdatableDBObject o) throws NoUpdatesException {
		return new DmlUpdateCommand(relvarName, o);
	}

	/**
	 * Ends the transaction, issuing either a commit or a rollback.
	 * 
	 * @param commit
	 *            true if the transaction is to be committed, false if it is to be rolled back.
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server
	 */
	public final void end (boolean commit) throws ConnectionClosedException {
		checkConnectionPresent();

		try {
			dbConnection.verifyStateAndSendEndTransactionMessage(commit, transactionID);
		} finally {
			dbConnection = null; // Why was this not present in pre-1.2 code ? Maybe because subsequent use of the transaction would fail in the DBConnection or on the server anyhow.
		}
	}

	/**
	 * Sends a command to the server and returns the result obtained.
	 * 
	 * @param command
	 *            the command to be executed, in textual form.
	 * @return the result of the command.
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server
	 * @throws ErrorMessageException
	 *             If the response obtained from the server is not a commit confirmation, but an error response instead.
	 */
	public final AbstractRelation execDmlCommand (DmlCommand command) throws ConnectionClosedException, ErrorMessageException {
		checkConnectionPresent();
		AbstractRelation result;
		if (dbConnection.getSpecificationVersionForServer().isBefore(Version.ONE_FOUR)) {
			result = execDmlCommandWithoutConnectionCheck(command.getCommand());
		} else {
			result = execDmlCommand(command.getServerCommand());
		}
		command.success();
		return result;
	}

	/**
	 * Sends a command to the server and returns the result obtained.
	 * 
	 * @param serverCommand
	 *            the command to be executed.
	 * @return the result of the command.
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server
	 * @throws ErrorMessageException
	 *             If the response obtained from the server is not a commit confirmation, but an error response instead.
	 */
	public final AbstractRelation execDmlCommand (ServerCommand serverCommand) throws ConnectionClosedException, ErrorMessageException {
		checkConnectionPresent();
		return execDmlCommandWithoutConnectionCheck(serverCommand);
	}

	/**
	 * Sends a command to the server and returns the result obtained.
	 * 
	 * @param cmd
	 *            the command to be executed, in textual form.
	 * @return the result of the command.
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server
	 * @throws ErrorMessageException
	 *             If the response obtained from the server is not a commit confirmation, but an error response instead.
	 */
	public final AbstractRelation execDmlCommand (String cmd) throws ConnectionClosedException, ErrorMessageException {
		checkConnectionPresent();
		return execDmlCommandWithoutConnectionCheck(cmd);
	}

	/**
	 * Sends a command to the server and returns the result obtained.
	 * 
	 * @param command
	 *            the command to be executed, in textual form.
	 * @return the result of the command.
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server
	 * @throws ErrorMessageException
	 *             If the response obtained from the server is not a commit confirmation, but an error response instead.
	 */
	public final AbstractRelation execDmlCommandAndEndTransaction (DmlCommand command) throws ConnectionClosedException, ErrorMessageException {
		try {
			AbstractRelation result = execDmlCommand(command.getCommand());
			command.success();
			end(true);
			return result;
		} catch (ErrorMessageException e) {
			end(false);
			throw e;
		} catch (RuntimeException e) {
			end(false);
			throw e;
		} catch (Error e) {
			end(false);
			throw e;
		}
	}

	/**
	 * Sends a command to the server and returns the result obtained.
	 * 
	 * @param serverCommand
	 *            the command to be executed.
	 * @return the result of the command.
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server
	 * @throws ErrorMessageException
	 *             If the response obtained from the server is not a commit confirmation, but an error response instead.
	 */
	public final AbstractRelation execDmlCommandAndEndTransaction (ServerCommand serverCommand) throws ConnectionClosedException, ErrorMessageException {
		try {
			AbstractRelation result = execDmlCommand(serverCommand);
			end(true);
			return result;
		} catch (ErrorMessageException e) {
			end(false);
			throw e;
		} catch (RuntimeException e) {
			end(false);
			throw e;
		} catch (Error e) {
			end(false);
			throw e;
		}
	}

	/**
	 * Sends a command to the server and returns the result obtained. After execution of the command, the transaction is committed and ended.
	 * 
	 * @param cmd
	 *            the command to be executed, in textual form.
	 * @return The result of executing the command. Null if the command is an assignment command.
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server
	 * @throws ErrorMessageException
	 *             If the response obtained from the server is not a commit confirmation, but an error response instead.
	 */
	public final AbstractRelation execDmlCommandAndEndTransaction (String cmd) throws ConnectionClosedException, ErrorMessageException {
		try {
			AbstractRelation result = execDmlCommand(cmd);
			end(true);
			return result;
		} catch (ErrorMessageException e) {
			end(false);
			throw e;
		} catch (RuntimeException e) {
			end(false);
			throw e;
		} catch (Error e) {
			end(false);
			throw e;
		}
	}

	/**
	 * Executes a series of single-assignment commands as a multiple-assignment command. Optionally closes the connection immediately after succesfull execution of the series of assignments. If the result indicates any kind of error, then the connection will be closed.
	 * 
	 * @param commands
	 *            The commands to be executed. Execution is in the obvious order, and thus not randomly, or in reverse order, or whatever.
	 * @return the DBResponse obtained.
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server
	 * @throws ErrorMessageException
	 *             If the response obtained from the server is not a commit confirmation, but an error response instead.
	 */
	public final AbstractRelation execMultipleStatement (Collection<? extends Object> commands) throws ConnectionClosedException, ErrorMessageException {
		boolean feedback = true;
		String multipleAssignmentCommand = ""; //$NON-NLS-1$
		Iterator<?> i_commands = commands.iterator();
		while (i_commands.hasNext()) {
			Object command = i_commands.next();
			if (!(command instanceof DmlCommand)) {
				feedback = false;
			}
			multipleAssignmentCommand += ("CMD(" + command + ")"); //$NON-NLS-1$//$NON-NLS-2$
		}

		if (multipleAssignmentCommand.length() > 0) {
			AbstractRelation result = execDmlCommand(multipleAssignmentCommand);
			if (feedback) {
				i_commands = commands.iterator();
				while (i_commands.hasNext()) {
					DmlCommand command = (DmlCommand) i_commands.next();
					command.success();
				}
			}
			return result;
		} else {
			return null;
		}
	}

	/**
	 * Executes a series of single-assignment commands as a multiple-assignment command. If the result indicates any kind of error, then the connection will be closed.
	 * 
	 * @param commands
	 *            The single-assignment commands to be executed. Execution is in the obvious order, and thus not randomly, or in reverse order, or whatever.
	 * @return the DBResponse obtained.
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server
	 * @throws ErrorMessageException
	 *             If the response obtained from the server is not a commit confirmation, but an error response instead.
	 */
	public final AbstractRelation execMultipleStatement (DmlAssignmentCommand[] commands) throws ConnectionClosedException, ErrorMessageException {
		int c = commands.length, i = 0;
		if (c > 0) {
			String multipleAssignmentCommand = ""; //$NON-NLS-1$
			while (i < c) {
				multipleAssignmentCommand += ("CMD(" + commands[i++].getCommand() + ")"); //$NON-NLS-1$//$NON-NLS-2$
			}
			AbstractRelation result = execDmlCommand(multipleAssignmentCommand);

			for (DmlAssignmentCommand command : commands) {
				command.success();
			}

			return result;
		} else {
			return null;
		}
	}

	/**
	 * Executes a series of single-assignment commands as a multiple-assignment command. If the result indicates any kind of error, then the connection will be closed.
	 * 
	 * @param commands
	 *            The single-assignment commands to be executed. Execution is in the obvious order, and thus not randomly, or in reverse order, or whatever.
	 * @return the DBResponse obtained.
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server
	 * @throws ErrorMessageException
	 *             If the response obtained from the server is not a commit confirmation, but an error response instead.
	 */
	public final AbstractRelation execMultipleStatement (String[] commands) throws ConnectionClosedException, ErrorMessageException {
		int c = commands.length, i = 0;
		if (c > 0) {
			String multipleAssignmentCommand = ""; //$NON-NLS-1$
			while (i < c) {
				multipleAssignmentCommand += ("CMD(" + commands[i++] + ")"); //$NON-NLS-1$//$NON-NLS-2$
			}
			return execDmlCommand(multipleAssignmentCommand);
		} else {
			return null;
		}
	}

	/**
	 * Executes a series of single-assignment commands as a multiple-assignment command.
	 * 
	 * @param commands
	 *            The commands to be executed. Execution is in the obvious order, and thus not randomly, or in reverse order, or whatever.
	 * @return the DBResponse obtained.
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server
	 * @throws ErrorMessageException
	 *             If the response obtained from the server is not a commit confirmation, but an error response instead.
	 */
	public final AbstractRelation execMultipleStatementAndEndTransaction (Collection<? extends Object> commands) throws ConnectionClosedException, ErrorMessageException {
		try {
			AbstractRelation result = execMultipleStatement(commands);
			end(true);
			return result;
		} catch (ErrorMessageException e) {
			end(false);
			throw e;
		} catch (RuntimeException e) {
			end(false);
			throw e;
		} catch (Error e) {
			end(false);
			throw e;
		}
	}

	/**
	 * Executes a series of single-assignment commands as a multiple-assignment command.
	 * 
	 * @param commands
	 *            The single-assignment commands to be executed. Execution is in the obvious order, and thus not randomly, or in reverse order, or whatever.
	 * @return the DBResponse obtained.
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server
	 * @throws ErrorMessageException
	 *             If the response obtained from the server is not a commit confirmation, but an error response instead.
	 */
	public final AbstractRelation execMultipleStatementAndEndTransaction (DmlAssignmentCommand[] commands) throws ConnectionClosedException, ErrorMessageException {
		try {
			AbstractRelation result = execMultipleStatement(commands);
			end(true);
			return result;
		} catch (ErrorMessageException e) {
			end(false);
			throw e;
		} catch (RuntimeException e) {
			end(false);
			throw e;
		} catch (Error e) {
			end(false);
			throw e;
		}
	}

	/**
	 * Executes a series of single-assignment commands as a multiple-assignment command.
	 * 
	 * @param commands
	 *            The single-assignment commands to be executed. Execution is in the obvious order, and thus not randomly, or in reverse order, or whatever.
	 * @return the DBResponse obtained.
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server
	 * @throws ErrorMessageException
	 *             If the response obtained from the server is not a commit confirmation, but an error response instead.
	 */
	public final AbstractRelation execMultipleStatementAndEndTransaction (String[] commands) throws ConnectionClosedException, ErrorMessageException {
		try {
			AbstractRelation result = execMultipleStatement(commands);
			end(true);
			return result;
		} catch (ErrorMessageException e) {
			end(false);
			throw e;
		} catch (RuntimeException e) {
			end(false);
			throw e;
		} catch (Error e) {
			end(false);
			throw e;
		}
	}

	/**
	 * Executes a series of inquire commands on the server.
	 * 
	 * @param expression
	 *            The expressions of the relational algebra to be inquired.
	 * @return response A DBResponse array holding the values of the corresponding queried expressions.
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server
	 * @throws ErrorMessageException
	 *             If the response obtained from the server is not a commit confirmation, but an error response instead.
	 */
	public final AbstractRelation[] execQueries (String[] expression) throws ConnectionClosedException, ErrorMessageException {
		AbstractRelation[] rsp = new AbstractRelation[expression.length];
		int i = 0;
//		while (i < expression.length) {
//			rsp[i] = execQuery(expression[i]);
//			i++;
//		}

		StringBuilder cmd = new StringBuilder("EXTEND(TABLE_DEE,("); //$NON-NLS-1$
		i = 0;
		while (i < expression.length) {
			cmd.append(Integer.toString(i)).append('(').append(expression[i]).append(')');
			i++;
		}
		cmd.append(')').append(')');

		AbstractRelation r = execQuery(cmd.toString());
		i = 0;
		while (i < expression.length) {
			rsp[i] = r.iterator().next().rvaValue(Integer.toString(i));
			i++;
		}
		return rsp;
	}

	/**
	 * Executes a series of inquire commands on the server.
	 * 
	 * @param expression
	 *            The expressions of the relational algebra to be inquired.
	 * @return response A DBResponse array holding the values of the corresponding queried expressions.
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server
	 * @throws ErrorMessageException
	 *             If the response obtained from the server is not a commit confirmation, but an error response instead.
	 */
	public final AbstractRelation[] execQueriesAndEndTransaction (String[] expression) throws ConnectionClosedException, ErrorMessageException {
		try {
			AbstractRelation[] result = execQueries(expression);
			end(true);
			return result;
		} catch (ErrorMessageException e) {
			end(false);
			throw e;
		} catch (RuntimeException e) {
			end(false);
			throw e;
		} catch (Error e) {
			end(false);
			throw e;
		}
	}

	/**
	 * Executes an inquire command on the server. If the result indicates any kind of error, then the connection will be closed.
	 * 
	 * @param expression
	 *            The expression of the relational algebra to be inquired.
	 * @return response A DBResponse holding the value of the queried expression.
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server
	 * @throws ErrorMessageException
	 *             If the response obtained from the server is not a commit confirmation, but an error response instead.
	 */
	public final AbstractRelation execQuery (String expression) throws ConnectionClosedException, ErrorMessageException {
		return execDmlCommand("INQUIRE " + expression); //$NON-NLS-1$
	}

	/**
	 * Executes an inquire command on the server. Optionally closes the connection immediately after succesfull execution of the query. If the result indicates any kind of error, then the connection will be closed.
	 * 
	 * @param expression
	 *            The expression of the relational algebra to be inquired.
	 * @return response A DBResponse holding the value of the queried expression.
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server
	 * @throws ErrorMessageException
	 *             If the response obtained from the server is not a commit confirmation, but an error response instead.
	 */
	public final AbstractRelation execQueryAndEndTransaction (String expression) throws ConnectionClosedException, ErrorMessageException {
		try {
			AbstractRelation result = execQuery(expression);
			end(true);
			return result;
		} catch (ErrorMessageException e) {
			end(false);
			throw e;
		} catch (RuntimeException e) {
			end(false);
			throw e;
		} catch (Error e) {
			end(false);
			throw e;
		}
	}

	/**
	 * Gets The "parent" transaction of this nested transaction, or null if this is not a nested transaction
	 * 
	 * @return The "parent" transaction of this nested transaction, or null if this is not a nested transaction
	 */
	public final DBTransaction getParentTransaction ( ) {
		return parentTransaction;
	}

	/**
	 * Gets The numerical server-side ID of the started transaction, or -1 if the server is pre 1.2
	 * 
	 * @return The numerical server-side ID of the started transaction, or -1 if the server is pre 1.2
	 */
	public final long getTransactionID ( ) {
		return transactionID;
	}

	/**
	 * Gets the flag indicating whether all DML statements executed in the transaction should be automatically committed
	 * 
	 * @return true if all DML statements executed in the transaction should be automatically committed
	 */
	public final boolean isAutoCommit ( ) {
		return autoCommit;
	}

	/**
	 * Gets the Flag indicating whether the transaction is read-only
	 * 
	 * @return Flag indicating whether the transaction is read-only
	 */
	public final boolean isReadOnly ( ) {
		return transactionMode.isReadOnly();
	}

	/**
	 * Commits this non-autocommit transaction. If the transaction is auto-commit, this method is a NOP.
	 * 
	 * @throws ConnectionClosedException
	 *             If the connection was closed due to an I/O error that occurred during communication with the server
	 */
	public final void rollback ( ) throws ConnectionClosedException {
		checkConnectionPresent();

		if (!autoCommit) {
			try {
				dbConnection.sendRollbackMessage(transactionID);
			} catch (ConnectionClosedException e) {
				dbConnection = null;
				throw e;
			}
		}
	}

	/**
	 * Takes a savepoint in the current transaction. Rolling back to this savepoint is done by invoking the rollback() method on the DBTransaction object returned by this method invocation. Rolling back to this savepoint AND discarding it, is done by invoking the endTransaction(false) method on the DBTransaction object returned from this method.
	 * 
	 * @return A DBTransaction object representing the new "nested" transaction that is started on the server side and that will span all activities done on the server from this point on.
	 * @throws ConnectionClosedException
	 * @throws ErrorMessageException
	 */
	public final DBTransaction savePoint ( ) throws ErrorMessageException, ConnectionClosedException {
		return startNestedTransaction(autoCommit, isReadOnly());
	}

	/**
	 * Starts a nested transaction within this transaction.
	 * 
	 * @param newAutoCommit
	 *            The autocommit setting for the new transaction nested inside the current one
	 * @param readOnly
	 *            The readOnly setting for the new transaction nested inside the current one. May only be false if this transaction is not read-only too.
	 * 
	 * @return A DBTransaction object representing the new "nested" transaction that is started on the server side and that will span all activities done on the server from this point on.
	 * @throws ConnectionClosedException
	 * @throws ErrorMessageException
	 */
	public final DBTransaction startNestedTransaction (boolean newAutoCommit, boolean readOnly) throws ErrorMessageException, ConnectionClosedException {
		if (nestedTransaction == null) {
			try {
				return (nestedTransaction = dbConnection.verifyStateAndSendStartNestedTransactionMessage(newAutoCommit, TransactionMode.getTransactionMode(readOnly)));
			} catch (ConnectionClosedException e) {
				dbConnection = null;
				throw e;
			}
		} else {
			return nestedTransaction.startNestedTransaction(newAutoCommit, readOnly);
		}
	}

	public String toString ( ) {
		return "Transaction " + super.toString() + "with server-side ID " + transactionID + " owned by " + userID + "on " + dbConnection; //$NON-NLS-1$//$NON-NLS-2$//$NON-NLS-3$ //$NON-NLS-4$
	}

}