/**
 * 
 */
package be.SIRAPRISE.client;

import java.lang.reflect.InvocationTargetException;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;

import be.SIRAPRISE.client.jsba.GetterMethodCache;
import be.SIRAPRISE.client.jsba.JSBAMethodInvoker;
import be.SIRAPRISE.client.jsba.UpdatableDBObject;

/**
 * @author Erwin
 * 
 */
public abstract class DmlSingleAssignmentCommand extends DmlAssignmentCommand {

	/**
	 * The map of getter methods that exist in the object's class to get attribute values for the named relvar
	 */
	private Map<String, JSBAMethodInvoker> getterMethods = new HashMap<String, JSBAMethodInvoker>();

	/**
	 * The textual singleton relation value drawn from the database object and relvar name used to construct the command
	 */
	private String relationText;

	/**
	 * The name of the relvar to which the update is targeted
	 */
	private String relvarName;

	/**
	 * The object from which the AssignmentCommand was created, in the case this object implements UpdatableDBObject. Any successful DML can then be fed back to the object by invoking the setPreUpdateState() method.
	 */
	private UpdatableDBObject updatableObject;

	/**
	 * Creates the command object. It is required that for each attribute in the heading of the named database relvar, there must be a corresponding getter method with the following characteristics :
	 * <ul>
	 * <li>the method name is composed of the lowercase string 'get', the capitalized name of the relvar (e.g. 'Relvar' or 'Storagespace'), and the capitalized attribute name (e.g. Extentscount)</li>
	 * <li>the method takes no arguments</li>
	 * <li>the method returns a java.lang.String</li>
	 * <li>the method is public</li>
	 * </ul>
	 * Other methods may exist besides these required getter methods. The getter methods in question need not be declared in the object's class itself, they may also be inherited from some superclass of the object's class.
	 * 
	 * @param relvarName
	 *            The name of the relvar in which a tuple for the object is to be added
	 * @param o
	 *            The object for which an update operation (add/assert/delete/unassert/update) is to be performed in the named relvar
	 * @throws IllegalArgumentException
	 *             If relvarName or o are null, or if relvarName is the zero-length string.
	 */
	public DmlSingleAssignmentCommand (String relvarName, Object o) {
		if (relvarName == null || relvarName.length() < 1) {
			throw new IllegalArgumentException();
		}

		this.relvarName = Character.toUpperCase(relvarName.charAt(0)) + relvarName.substring(1).toLowerCase();

		if (o == null) {
			throw new IllegalArgumentException();
		}

		if (o instanceof UpdatableDBObject) {
			this.updatableObject = (UpdatableDBObject) o;
		}

		getterMethods = GetterMethodCache.getInstance().getGetterMethods(o.getClass(), this.relvarName);

		StringBuilder relation = new StringBuilder(relvarName.toUpperCase() + "(TUPLE("); //$NON-NLS-1$

		for (Entry<String, JSBAMethodInvoker> me : getterMethods.entrySet()) {
			try {
				relation.append(me.getKey()).append('(').append(me.getValue().invoke(o, (Object[]) null)).append(')');
			} catch (IllegalAccessException e) {
				throw new RuntimeException(e);
			} catch (InvocationTargetException e) {
				throw new RuntimeException(e.getCause());
			}
		}
		relation.append(')').append(')');

		relationText = relation.toString();
	}

	/**
	 * Gets the getter methods from a class applicable for the given relvar name
	 * 
	 * @return A Map of Method objects representing each applicable getter method found in the class, mapped to their attribute name.
	 */
	final Map<String, JSBAMethodInvoker> getGetterMethods ( ) {
		return getterMethods;
	}

	/**
	 * If the object from which this command was created, implements UpdatableDBObject, invokes that object's setPreUpdateState() method to reflect the successfull database insert or update.
	 */
	final void success ( ) {
		if (updatableObject != null) {
			updatableObject.setPreUpdateState();
		}
	}

	/**
	 * Gets the relation text denoting the singleton relation holding the tuple holding the values drawn from object o for the named relvar
	 * 
	 * @return the relation text denoting the singleton relation holding the tuple holding the values drawn from object o for the named relvar
	 */
	public final String getRelationText ( ) {
		return relationText;
	}

	/**
	 * Gets The name of the relvar to which the update is targeted
	 * 
	 * @return The name of the relvar to which the update is targeted
	 */
	public final String getRelvarName ( ) {
		return relvarName;
	}
}
