/*
 * Created on 7-aug-2008
 */
package be.SIRAPRISE.client;

import java.io.DataOutputStream;
import java.io.IOException;
import java.util.Iterator;
import java.util.Set;

import be.SIRAPRISE.client.NAMES.TYPENAMES;
import be.SIRAPRISE.util.IntersectableLinkedHashMap;
import be.SIRAPRISE.util.IntersectableMap;
import be.SIRAPRISE.util.MyDataOutputStream;
import be.erwinsmout.NotFoundException;

/**
 * A Heading maps attribute names onto typenames.
 * 
 * @author Erwin Smout
 */
public class Heading extends IntersectableLinkedHashMap<String, TypeDeclaration> {

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.SIRAPRISE.util.IntersectableLinkedHashMap#getClone()
	 */
	@Override
	protected IntersectableMap<String, TypeDeclaration> getClone ( ) {
		Heading h = new Heading();
		h.putAll(this);
		return h;
	}

	/**
	 * Adds a relation-typed attribute to the heading. The relation type of the attribute is defined by the given heading.
	 * 
	 * @param attributeName
	 *            The name of a relation-typed attribute to be added to the heading.
	 * @param heading
	 *            The relation heading defining the relation type of the attribute
	 * @throws DuplicateException
	 *             If the attributeName is already present in the heading
	 */
	public final void add (String attributeName, Heading heading) throws DuplicateException {
		// if (containsKey(attributeName)) {
		// throw new DuplicateException(attributeName);
		// }
		add(attributeName, new NonScalarTypeDeclaration(heading));
		// relationTypes.put(attributeName, heading);
	}

	/**
	 * Adds a scalar attribute to the heading.
	 * 
	 * @param attributeName
	 *            The name of the attribute to be added to the heading.
	 * @param typeName
	 *            The name of the attribute's type.
	 * @throws DuplicateException
	 *             If the attributeName is already present in the heading
	 */
	public final void add (String attributeName, String typeName) throws DuplicateException {
		add(attributeName, new ScalarTypeDeclaration(typeName));
	}

	/**
	 * Adds an attribute declaration to the heading.
	 * 
	 * @param attributeName
	 *            The name of the attribute to be added to the heading.
	 * @param typeDeclaration
	 *            The type declaration of the attribute.
	 * @throws DuplicateException
	 *             If the attributeName is already present in the heading
	 */
	public final void add (String attributeName, TypeDeclaration typeDeclaration) throws DuplicateException {
		if (containsKey(attributeName)) {
			throw new DuplicateException(attributeName);
		}
		put(attributeName, typeDeclaration);
	}

	/**
	 * Gets The set of attribute names
	 * 
	 * @return The set of attribute names
	 */
	public final Set<String> getAttributeNames ( ) {
		return keySet();
	}

	/**
	 * Gets the heading object defining the relation type of the named attribute, which must be relation-typed
	 * 
	 * @param attributeName
	 *            The name of a relation-typed attribute whose heading is to be obtained
	 * @return the heading object defining the relation type of the named attribute, which must be relation-typed
	 * @throws NotFoundException
	 * @throws TypeIsScalarException
	 */
	public final Heading getHeading (String attributeName) throws NotFoundException, TypeIsScalarException {
		if (containsKey(attributeName)) {
			// if (relationTypes.containsKey(attributeName)) {
			// return (HeadingType)relationTypes.get(attributeName);
			// }
			return get(attributeName).getHeading();
		}

		throw new NotFoundException(attributeName);
	}

	/**
	 * Gets the typename of the named attribute. If the attribute is relation-typed, the string "RELATION" is returned. Further specification of the precise relation heading of the attribute can be obtained through a getHeadingType() invocation.
	 * 
	 * @param attributeName
	 *            The name of the attribute whose type name is to be obtained
	 * @return The typename of the named attribute
	 * @throws NotFoundException
	 */
	public final String getTypeName (String attributeName) throws NotFoundException {
		if (containsKey(attributeName)) {
			return get(attributeName).getTypeName();
		}
		throw new NotFoundException(attributeName);
	}

	/**
	 * The map containing the Heading objects defining the relation types of each relation-typed attribute
	 */
	// private IntersectableMap relationTypes = new IntersectableHashMap();
	/**
	 * Checks if the heading of another Relvar is exactly the same as this one :
	 * <P>
	 * <UL>
	 * <LI>The set of attribute names in the other heading is equal to the set of attribute names in this heading</LI>
	 * <LI>The Type associated with each attribute name is the same in both headings</LI>
	 * </UL>
	 * </P>
	 * 
	 * @param other
	 *            the Heading of another Relvar
	 * @return true if the other RelvarHeading contains exactly the same set of atrributename/typedeclaration pairs
	 */
	public final boolean isStrictUnionCompatibleWith (Heading other) {
		if (keySet().equals(other.keySet())) {
			Iterator<String> i_attributeNames = keySet().iterator();
			while (i_attributeNames.hasNext()) {
				String attributeName = i_attributeNames.next();
				try {
					if (!getTypeName(attributeName).equals(other.getTypeName(attributeName))) {
						return false;
					}
				} catch (NotFoundException impossible) {
					throw new RuntimeException(impossible);
				}
			}
			return true;
		}

		return false;
	}

	/**
	 * Gets the heading specification in textual format HEADING(ATTRIBUTENAME(TYPENAME)RELATIONTYPEDATTRIBUTENAME(RELATION(HEADING(...))))
	 * 
	 * @return the heading specification in textual format HEADING(ATTRIBUTENAME(TYPENAME)RELATIONTYPEDATTRIBUTENAME(RELATION(HEADING(...))))
	 */
	public final String print ( ) {
		StringBuilder sb = new StringBuilder(60 * size());
		sb.append("HEADING("); //$NON-NLS-1$
		Iterator<Entry<String, TypeDeclaration>> i_attributes = entrySet().iterator();
		while (i_attributes.hasNext()) {
			java.util.Map.Entry<String, TypeDeclaration> me = i_attributes.next();
			String attributeName = me.getKey();
			String typeName = me.getValue().getTypeName();
			sb.append(attributeName).append('(').append(typeName);
			if (typeName.equalsIgnoreCase(TYPENAMES.RELATION)) {
				try {
					sb.append('(').append(getHeading(attributeName).print()).append(')');
				} catch (NotFoundException impossible) {
					// assert false
					throw new RuntimeException(impossible);
				} catch (TypeIsScalarException impossible) {
					// assert false
					throw new RuntimeException(impossible);
				}
			}
			// if (!attribute.isScalar()) {
			// sb.append('(').append(((RelationTypedAttribute)attribute).getRvaHeadingType().print()).append(')');
			// }
			sb.append(')');
		}
		sb.append(')');
		return sb.toString();
	}

	/**
	 * Gets the heading specification in XML style
	 * 
	 * @return the formatted heading specification in XML xtyle
	 */
	public final String printXML ( ) {
		StringBuilder sb = new StringBuilder(60 * size());
		sb.append("<Heading>"); //$NON-NLS-1$
		Iterator<java.util.Map.Entry<String, TypeDeclaration>> i_attributes = entrySet().iterator();
		while (i_attributes.hasNext()) {
			java.util.Map.Entry<String, TypeDeclaration> me = i_attributes.next();
			String attributeName = me.getKey();
			String typeName = me.getValue().getTypeName();
			sb.append('<').append(attributeName).append('>');
			if (typeName.equalsIgnoreCase(TYPENAMES.RELATION)) {
				try {
					sb.append("<Relation>").append(getHeading(attributeName).printXML()).append("</Relation>"); //$NON-NLS-1$ //$NON-NLS-2$
				} catch (NotFoundException impossible) {
					// assert false
					throw new RuntimeException(impossible);
				} catch (TypeIsScalarException impossible) {
					// assert false
					throw new RuntimeException(impossible);
				}
			} else {
				sb.append(typeName);
			}
			sb.append("</").append(attributeName).append('>'); //$NON-NLS-1$
		}
		sb.append("</Heading>"); //$NON-NLS-1$
		return sb.toString();
	}

	/**
	 * Gets a Heading that includes only the attribute definitions from this Heading whose attribute names do not appear in the given set
	 * 
	 * @param attributeNames
	 *            The names of the attributes that are not to appear in the subHeading
	 * @return A Heading that includes only the attribute definitions from this Heading whose attribute names do not appear in the given set
	 */
	public final Heading subHeadingExcluding (Set<String> attributeNames) {
		Heading newHeadingType;
		try {
			newHeadingType = (Heading) minus(attributeNames, Heading.class);
		} catch (InstantiationException impossible) {
			// assert false
			throw new RuntimeException(impossible);
		} catch (IllegalAccessException impossible) {
			// assert false
			throw new RuntimeException(impossible);
		}
		// newHeadingType.relationTypes = (IntersectableMap)relationTypes.minus(attributeNames);
		return newHeadingType;
	}

	/**
	 * Gets a Heading that includes only the attribute definitions from this Heading whose attribute names appear in the given set
	 * 
	 * @param attributeNames
	 *            The names of the attributes that are to appear in the subHeading
	 * @return A Heading that includes only the attribute definitions from this Heading whose attribute names appear in the given set
	 */
	public final Heading subHeadingIncluding (Set<String> attributeNames) {
		Heading newHeadingType;
		try {
			newHeadingType = (Heading) intersect(attributeNames, Heading.class);
		} catch (InstantiationException impossible) {
			// assert false
			throw new RuntimeException(impossible);
		} catch (IllegalAccessException impossible) {
			// assert false
			throw new RuntimeException(impossible);
		}
		// newHeadingType.relationTypes = (IntersectableMap)relationTypes.intersect(attributeNames);
		return newHeadingType;
	}

	/**
	 * Writes the heading to an outputstream in the V1.0 format
	 * 
	 * @param outputStream
	 *            the outputstream to write the heading to
	 * @throws IOException
	 */
	public final void toStream1_0 (DataOutputStream outputStream) throws IOException {
		outputStream.writeInt(size());
		Iterator<java.util.Map.Entry<String, TypeDeclaration>> i_heading = entrySet().iterator();
		while (i_heading.hasNext()) {
			java.util.Map.Entry<String, TypeDeclaration> me = i_heading.next();
			String attributeName = me.getKey();
			String typeName = me.getValue().getTypeName();
			MyDataOutputStream.writeSmallUTF(attributeName, outputStream);
			MyDataOutputStream.writeSmallUTF(typeName, outputStream);
			if (typeName.equalsIgnoreCase(TYPENAMES.RELATION)) {
				try {
					getHeading(attributeName).toStream1_0(outputStream);
				} catch (NotFoundException impossible) {
					// assert false
					throw new RuntimeException(impossible);
				} catch (TypeIsScalarException impossible) {
					// assert false
					throw new RuntimeException(impossible);
				}
			}
		}
	}
}