/*
 * Created on 2-feb-2005
 */
package be.SIRAPRISE.client;

import java.io.BufferedOutputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.net.InetAddress;
import java.net.Socket;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.security.Signature;
import java.security.SignatureException;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

import be.SIRAPRISE.messages.AuthenticationOKMessage;
import be.SIRAPRISE.messages.AuthenticationOKMessageType;
import be.SIRAPRISE.messages.ClientAuthenticationMessage;
import be.SIRAPRISE.messages.ClientAuthenticationMessageType;
import be.SIRAPRISE.messages.ClientHelloMessage;
import be.SIRAPRISE.messages.ClientHelloMessageType;
import be.SIRAPRISE.messages.DmlExecutedMessage;
import be.SIRAPRISE.messages.EndConnectionMessageType;
import be.SIRAPRISE.messages.ExecuteMonitorCommandMessage;
import be.SIRAPRISE.messages.ExecuteMonitorCommandMessageType;
import be.SIRAPRISE.messages.ServerHelloMessage;
import be.SIRAPRISE.messages.ServerHelloMessageType;
import be.SIRAPRISE.messages.ServerMessage;
import be.SIRAPRISE.messages.ServerMessageTypes;
import be.SIRAPRISE.messages.TransactionStartedMessageType;
import be.SIRAPRISE.util.MyDataOutputStream;
import be.erwinsmout.MyMessageFormat;
import be.erwinsmout.NotFoundException;

/**
 * Class used to communicate between a program and the SIRA_PRISE monitor.
 * 
 * @author Erwin Smout
 */
public final class MonitorConnection {

	/**
	 * The complete set of signing protocols supported by the server to authenticate (user) identity
	 */
	private Set<String> alternativeSigningProtocols;

	/**
	 * The private Key to be used for signing each message sent
	 */
	private Signer signer;

	/**
	 * The signing protocol to be used for signing each message sent
	 */
	private Signature signingProtocol;

	/**
	 * The socket connected to the server monitor
	 */
	private Socket socket;

	/**
	 * The input stream from which responses can be read
	 */
	private DataInputStream socketInputStream;

	/**
	 * The output stream to which to write commands
	 */
	private DataOutputStream socketOutputStream;

	/**
	 * The SIRA_PRISE specification version to use when selecting message types to send to the server
	 */
	private Version specificationVersionForServer;

	/**
	 * @param netAddress
	 * @param monitorPort
	 * @throws IOException
	 * @throws DBException
	 * @throws CommunicationProtocolException
	 */
	public MonitorConnection (InetAddress netAddress, int monitorPort) throws CommunicationProtocolException, IOException, DBException {
		this(netAddress, monitorPort, new HashSet<String>(), "", null); //$NON-NLS-1$
	}

	/**
	 * @param netAddress
	 * @param monitorPort
	 * @param signatureAlgorithmNames
	 * @param clientID
	 * @param signer
	 * @throws IOException
	 * @throws DBException
	 * @throws CommunicationProtocolException
	 */
	public MonitorConnection (InetAddress netAddress, int monitorPort, Set<String> signatureAlgorithmNames, String clientID, Signer signer) throws CommunicationProtocolException, IOException, DBException {
		this.signer = signer;
		socket = new Socket(netAddress, monitorPort);
		socketOutputStream = new DataOutputStream(new BufferedOutputStream(socket.getOutputStream()));
		socketInputStream = new DataInputStream(socket.getInputStream());

		handShake(signatureAlgorithmNames, clientID);
	}

	/**
	 * Creates a monitor connection to the specified host and port for an anonymous client.
	 * 
	 * @param host
	 *            The identification of the host. It may either be its DNS name or its ip address in dotted decimal.
	 * @param monitorPort
	 *            The port to which to connect.
	 * @throws IOException
	 * @throws CommunicationProtocolException
	 * @throws DBException
	 */
	public MonitorConnection (String host, int monitorPort) throws IOException, CommunicationProtocolException, DBException {
		this(host, monitorPort, new HashSet<String>(), "", null); //$NON-NLS-1$
	}

	/**
	 * Creates the monitor connection to the specified host and port.
	 * 
	 * @param host
	 *            The identification of the host. It may either be its DNS name or its ip address in dotted decimal.
	 * @param monitorPort
	 *            The port to which to connect.
	 * @param signatureAlgorithmNames
	 *            The set of Signature algorithm names that can be used to authenticate the client
	 * @param clientID
	 *            The client ID
	 * @param signer
	 *            The object that will be called upon to compute the authentication signature
	 * @throws IOException
	 * @throws CommunicationProtocolException
	 * @throws DBException
	 */
	public MonitorConnection (String host, int monitorPort, Set<String> signatureAlgorithmNames, String clientID, Signer signer) throws IOException, CommunicationProtocolException, DBException {
		this.signer = signer;
		socket = new Socket(host, monitorPort);
		socketOutputStream = new DataOutputStream(new BufferedOutputStream(socket.getOutputStream()));
		socketInputStream = new DataInputStream(socket.getInputStream());

		handShake(signatureAlgorithmNames, clientID);
	}

	/**
	 * 
	 */
	private void closeSockets ( ) {
		try {
			socketInputStream.close();
		} catch (IOException e) {

		}
		try {
			socketOutputStream.close();
		} catch (IOException e1) {

		}
		try {
			socket.close();
		} catch (IOException e) {

		}
	}

	/**
	 * @param signatureAlgorithmNames
	 * @param clientID
	 * @throws CommunicationProtocolException
	 * @throws IOException
	 * @throws DBException
	 */
	private void handShake (Set<String> signatureAlgorithmNames, String clientID) throws CommunicationProtocolException, IOException, DBException {
		// Create and send the ClientHelloMessage
		ClientHelloMessageType clientHelloMessageType;
		try {
			clientHelloMessageType = (ClientHelloMessageType) ServerMessageTypes.getInstance().getServerMessageType(ClientHelloMessageType.MESSAGETYPEID);
		} catch (NotFoundException e2) {
			throw new CommunicationProtocolException(Messages.getString("DBConnection.ClientHelloMessagetypeNotFound"), e2); //$NON-NLS-1$
		}
		ClientHelloMessage clientHelloMessage = clientHelloMessageType.message(signatureAlgorithmNames, new HashSet<String>(), 0, ServerMessageTypes.getInstance().getThisPackagesSiraPriseVersion());
		clientHelloMessage.sendMessage(socketOutputStream, null, null, null);

		// Anxiously await the ServerHelloMessage (and hope it won't be an error message)
		try {
			ServerMessage serverMessage = ServerMessage.readMessage(socketInputStream, null, null, null);
			if (!(serverMessage instanceof ServerHelloMessage)) {
				throw new CommunicationProtocolException(serverMessage.getClass().getName(), ServerHelloMessageType.class.getName());
			}
			ServerHelloMessage serverHelloMessage = (ServerHelloMessage) serverMessage;
			this.signingProtocol = serverHelloMessage.getSigningProtocol();
			// cryptoProtocol = serverHelloMessage.getCryptoProtocol();
			this.specificationVersionForServer = serverHelloMessage.getVersion();
			alternativeSigningProtocols = serverHelloMessage.getAlternativeSigningProtocols();
			// Iterator i_alternativeSigningProtocols = alternativeSigningProtocols.iterator();
			// while (i_alternativeSigningProtocols.hasNext()) {
			// String alternativeSigningProtocolName = (String) i_alternativeSigningProtocols.next();
			// if (!defaultSupportedSignatureAlgorithmNames.contains(alternativeSigningProtocolName)) {
			// i_alternativeSigningProtocols.remove();
			// }
			// }
			// idleTime = serverHelloMessage.getIdleTime();

			if (signingProtocol != null) {
				try {
					// privateKey = privateKeyProvider.getPrivateKey(signingProtocol.getAlgorithm());
					// Get the message to be signed (this is the clientID as written by the smallUTF encoding
					ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream();
					MyDataOutputStream.writeSmallUTF(clientID, new DataOutputStream(byteArrayOutputStream));
					byte[] signMessage = byteArrayOutputStream.toByteArray();
					byte[] signature = signer.sign(signingProtocol, signMessage);

					ClientAuthenticationMessageType clientAuthenticationMessageType = (ClientAuthenticationMessageType) ServerMessageTypes.getInstance().getServerMessageTypeForSiraPriseVersion(ClientAuthenticationMessageType.MESSAGETYPEID, specificationVersionForServer);
					ClientAuthenticationMessage clientAuthenticationMessage = clientAuthenticationMessageType.message(clientID, signature);
					clientAuthenticationMessage.sendMessage(socketOutputStream, null, null, null);
				} catch (Exception e1) {
					closeSockets();
					throw new CommunicationProtocolException(Messages.getString("DBConnection.SigningFailed"), e1); //$NON-NLS-1$
				}

				// And now anxiously await the authenticationOK message
				serverMessage = ServerMessage.readMessage(socketInputStream, null, null, null);
				if (!(serverMessage instanceof AuthenticationOKMessage)) {
					throw new CommunicationProtocolException(serverMessage.getClass().getName(), AuthenticationOKMessageType.class.getName());
				}
			}
		} catch (ErrorMessageException e) {
			closeSockets();
			throw new DBException(e);
		}
	}

	/**
	 * Simply closes the connection.
	 * 
	 * @throws DBException
	 *             If the sira_prise monitor reports an error it encountered
	 */
	public void end ( ) throws DBException {
		try {
			((EndConnectionMessageType) ServerMessageTypes.getInstance().getServerMessageType(EndConnectionMessageType.MESSAGETYPEID)).message().sendMessage(socketOutputStream, signingProtocol, signer, null);
//			ServerMessage.readMessage(socketInputStream, null, null, null);  No response is sent from an END command
//			System.out.println("  EndMonitorConnection has been sent.");
		} catch (IOException e) {
			throw new DBException(e);
		} catch (NotFoundException e) {
			throw new DBException(e);
		} catch (CommunicationProtocolException e) {
			throw new DBException(e);
//		} catch (ErrorMessageException e) {
//			throw new DBException(e);
		} finally {
			closeSockets();
		}
	}

	/**
	 * Sends a monitor command to the server and returns the result obtained. The monitor connection is immediately closed again.
	 * 
	 * @param command
	 *            The monitor command to be executed
	 * @param userID
	 *            The Identification of the user issuing the request
	 * @param userIDSigner
	 *            The Object that can be called upon to provide the private key needed to sign/authenticate user identity
	 * @param userIDAuthenticatedByClient
	 *            true if the identity of the user has been authenticated by the client
	 * @return The result of the command
	 * @throws DBException
	 *             If the sira_prise monitor reports an error it encountered
	 */
	public AbstractRelation execCommand (String command, String userID, Signer userIDSigner, boolean userIDAuthenticatedByClient) throws DBException {
		String signatureAlgorithmName = ""; //$NON-NLS-1$
		byte[] signature = new byte[] {};

		// No need to prove authenticity of the anonymous user
		if (userID.length() > 0 && userIDSigner != null) {
			Iterator<String> i_alternativeSigningProtocols = alternativeSigningProtocols.iterator();
			boolean signatureComputed = false;
			while (i_alternativeSigningProtocols.hasNext() & !signatureComputed) {
				signatureAlgorithmName = i_alternativeSigningProtocols.next();
				try {
					Signature signatureAlgorithm = Signature.getInstance(signatureAlgorithmName);
					// Prepare the bytes that are to be signed
					ByteArrayOutputStream w = new ByteArrayOutputStream();
					MyDataOutputStream.writeSmallUTF(userID, new DataOutputStream(w));
					// PrivateKey privateKey2 = privateKeyProvider.getPrivateKey(signatureAlgorithmName);
					byte[] signMessage = w.toByteArray();
					signature = userIDSigner.sign(signatureAlgorithm, signMessage);
					signatureComputed = true;
				} catch (NoSuchAlgorithmException e) {
					// Try next algorithm if there is one.
				} catch (InvalidKeyException e) {
					// privateKeyProvider privdied a key that turned out to be invalid. Try next algorithm if there is one.
				} catch (NotFoundException e) {
					// privateKeyProvider says he cannot provide a key for this algorithm. Try next algorithm if there is one.
				} catch (IOException e) {
					throw new DBException(e);
				} catch (SignatureException e) {
					throw new DBException(e);
				}
			}

			// try and see if it passes unauthenticated
			if (!signatureComputed) {
				// throw new DBException(MessageFormat.format(Messages.getString("DBConnection.UserIdentitySigningFailed"), new Object[]{alternativeSigningProtocols})); //$NON-NLS-1$
				signatureAlgorithmName = ""; //$NON-NLS-1$
			}
		}

		try {
			ExecuteMonitorCommandMessageType executeMonitorCommandMessageType = (ExecuteMonitorCommandMessageType) ServerMessageTypes.getInstance().getServerMessageType(ExecuteMonitorCommandMessageType.MESSAGETYPEID);
			ExecuteMonitorCommandMessage executeMonitorCommandMessage = executeMonitorCommandMessageType.message(command, signature, signatureAlgorithmName, userIDAuthenticatedByClient, userID);
			executeMonitorCommandMessage.sendMessage(socketOutputStream, signingProtocol, signer, null);

			// Await the response
			ServerMessage serverMessage = ServerMessage.readMessage(socketInputStream, null, null, null);
			if (!(serverMessage instanceof DmlExecutedMessage)) {
				throw new DBException(MyMessageFormat.format(Messages.getString("DBConnection.UnexpectedMessageType"), new String[] { serverMessage.getClass().getName(), Long.toHexString(TransactionStartedMessageType.MESSAGETYPEID) })); //$NON-NLS-1$
			}

			return ((DmlExecutedMessage) serverMessage).getRelation();
		} catch (IOException e) {
			throw new DBException(e);
		} catch (NotFoundException e) {
			throw new DBException(e);
		} catch (CommunicationProtocolException e) {
			throw new DBException(e);
		} catch (ErrorMessageException e) {
			throw new DBException(e);
		} finally {
			closeSockets();
		}
	}
}
