/*
 * Created on 12-sep-2007
 */
package be.SIRAPRISE.client;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.EOFException;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.nio.charset.Charset;
import java.security.InvalidKeyException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.Signature;
import java.security.SignatureException;
import java.security.UnrecoverableKeyException;
import java.security.cert.CertificateException;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.Set;
import be.SIRAPRISE.util.BracketParser;
import be.SIRAPRISE.util.CRLFInputStream;
import be.SIRAPRISE.util.CommandLineParameters;
import be.SIRAPRISE.util.MyKeyStore;
import be.SIRAPRISE.util.NameValuePair;
import be.SIRAPRISE.util.NameValueResult;
import be.SIRAPRISE.util.NoClosingBracketException;
import be.SIRAPRISE.util.UserPrivateKeyProvider;
import be.erwinsmout.MyMessageFormat;
import be.erwinsmout.NotFoundException;

/**
 * @author Erwin Smout
 */
public class ProcessScript implements Runnable, Signer {
	/**
	 * @author Erwin Smout
	 */
	static class SetContent {

		/**
		 * The host name
		 */
		private String host;

		/**
		 * The port number
		 */
		private String port;

		/**
		 * The user
		 */
		private String user;

		/**
		 * @param host
		 * @param port
		 * @param user
		 */
		public SetContent (String host, String port, String user) {
			this.host = host;
			this.port = port;
			this.user = user;
		}

		/**
		 * Gets The host name
		 * 
		 * @return The host name
		 */
		String getHost ( ) {
			return host;
		}

		/**
		 * Gets The port number
		 * 
		 * @return The port number
		 */
		String getPort ( ) {
			return port;
		}

		/**
		 * Gets the user
		 * 
		 * @return the user
		 */
		String getUser ( ) {
			return user;
		}
	}

	/**
	 * 
	 */
	public static final String CLIENTID = "SIRAPRISESCRIPTPROCESSOR"; //$NON-NLS-1$

	/**
	 * The name of the property used to indicate the name of the script file to be processed
	 */
	public static final String FILE = "FILE"; //$NON-NLS-1$

	/**
	 * The property name for specifying the output file.
	 */
	public static final String OUTFILE = "OUTFILE"; //$NON-NLS-1$

	/**
	 * The property name for specifying the directory containing the SIRA_PRISE script to be run
	 */
	public static final String SCRIPTSDIRECTORY = "SCRIPTSDIRECTORY"; //$NON-NLS-1$

	/**
	 * The suffix by which script files can be recognized
	 */
	public static final String SPS = ".SPS"; //$NON-NLS-1$

	/**
	 * The property name that specifies the directory name where the output file will be placed. Only applicable when the OUTFILE options does not itself specify an "absolute" path, i.e. a name starting with File.SeparatorChar.
	 */
	static final String OUTPUTDIRECTORY = "OUTPUTDIRECTORY"; //$NON-NLS-1$

	/**
	 * Gets a command. A command is the concatenation of all consecutive lines ending in the concatenation sequence ( |> ), plus the first line encountered not ending in a concatenation sequence. Obviously, the concatenation sequence is removed from each line before the line is concatenated into the final command.
	 * 
	 * @param fin
	 *            The bufferedreader to read
	 * @param charSet
	 *            The Charset to use for decoding the bytes into Strings
	 * @return The complete command
	 * @throws IOException
	 */
	private static String getCommand (CRLFInputStream fin, Charset charSet) throws IOException {
		String cmd = "|>"; //$NON-NLS-1$
		while (cmd.endsWith("|>")) { //$NON-NLS-1$
			String t = fin.readLine(charSet);
			if (t == null) {
				if (cmd.length() < 3) {
					throw new EOFException();
				} else {
					throw new IOException(Messages.getString("ProcessScript.ExpectedContinuationNotFound")); //$NON-NLS-1$
				}
			} else {
				cmd = cmd.substring(0, cmd.length() - 2).trim() + t;
			}
		}
		return cmd;
	}

	/**
	 * Gets a SetContent object holding hostname, portnumber and user values (or any possible subset of those)
	 * 
	 * @param string
	 *            A string of the form HOST()PORT()
	 * @return A SetContent object holding hostname, portnumber and user values (or any possible subset of those)
	 */
	private static SetContent getSetContent (String string) {
		String w = string;
		String host = null;
		String port = null;
		String user = null;
		try {
			while (w.length() > 0) {
				NameValueResult result = BracketParser.getNameValueFromNonEscaped(w, 0);
				NameValuePair nameValuePair = result.getNameValuePair();
				String name = nameValuePair.getName();
				w = w.substring(result.getNextParsePos());
				if (name.equalsIgnoreCase(DBConnectionProperties.HOST)) {
					host = nameValuePair.getValue();
				} else {
					if (name.equalsIgnoreCase(DBConnectionProperties.PORT)) {
						port = nameValuePair.getValue();
					} else {
						if (name.equalsIgnoreCase(DBConnectionProperties.USER)) {
							user = nameValuePair.getValue();
						} else {
							throw new RuntimeException("Invalid keyword " + name); //$NON-NLS-1$
						}
					}
				}
			}
		} catch (NoClosingBracketException e) {
			throw new RuntimeException("Invalid syntax " + w); //$NON-NLS-1$
		}
		return new SetContent(host, port, user);
	}

	/**
	 * Runs a series of commands through the server
	 * 
	 * @param args
	 *            No arguments are used
	 * @throws IOException
	 */
	public static void main (String args[]) throws IOException {
		new ProcessScript(args).run();
	}

	/**
	 * Comment for <code>cryptoAlgorithmNames</code>
	 */
	private Set<String> cryptoAlgorithmNames;

	/**
	 * Comment for <code>fin</code>
	 */
	private CRLFInputStream fin;

	/**
	 * Comment for <code>host</code>
	 */
	private String host;

	/**
	 * Comment for <code>out</code>
	 */
	private PrintWriter out;

	/**
	 * Comment for <code>port</code>
	 */
	private int port;

	/**
	 * Comment for <code>signingAlgorithmNames</code>
	 */
	private Set<String> signingAlgorithmNames;

	/**
	 * 
	 */
	private LinkedList<DBTransaction> transactionChain = new LinkedList<DBTransaction>();

	/**
	 * 
	 */
	private HashMap<String, DBTransaction> transactionsByName = new HashMap<String, DBTransaction>();

	/**
	 * Comment for <code>user</code>
	 */
	private String user;

	/**
	 * Creates a script processor
	 * 
	 * @param args
	 *            The command-line arguments used to invoke the Script processor
	 * @throws IOException
	 */
	ProcessScript (String[] args) throws IOException {

		CommandLineParameters commandLineParameters = CommandLineParameters.getCommandLineParameters(args, FILE, ProcessScriptProperties.getInstance());
		String scriptFileName = commandLineParameters.getProperty(FILE);
		if (scriptFileName == null) {
			System.out.println(Messages.getString("ProcessScript.FilenameRequired")); //$NON-NLS-1$
			BufferedReader conin = new BufferedReader(new InputStreamReader(System.in));
			scriptFileName = conin.readLine();
			// conin.close();
		}

		user = commandLineParameters.getProperty(DBConnectionProperties.USER, DBConnectionProperties.USERDEFAULT);
		host = commandLineParameters.getProperty(DBConnectionProperties.HOST, DBConnectionProperties.HOSTDEFAULT);
		port = Integer.parseInt(commandLineParameters.getProperty(DBConnectionProperties.PORT, DBConnectionProperties.PORTDEFAULT));

		String fullScriptFileName, baseScriptFileName;
		if (!scriptFileName.toUpperCase().endsWith(SPS)) {
			baseScriptFileName = scriptFileName + SPS;
		} else {
			baseScriptFileName = scriptFileName;
		}

		fullScriptFileName = baseScriptFileName;
		File scriptFile = null;
		while (fin == null) {
			scriptFile = new File(fullScriptFileName);
			try {
				fin = new CRLFInputStream(new BufferedInputStream(new FileInputStream(scriptFile)));
			} catch (FileNotFoundException e) {
				if (scriptFile.isAbsolute() || fullScriptFileName.charAt(0) == File.separatorChar) {
					throw e;
				} else {
					String scriptsDirectory = commandLineParameters.getProperty(SCRIPTSDIRECTORY);
					if (scriptsDirectory == null) {
						throw e;
					} else {
						if (!scriptsDirectory.endsWith(File.separator)) {
							scriptsDirectory += File.separatorChar;
						}
						fullScriptFileName = scriptsDirectory + fullScriptFileName;
					}
				}
			}
		}

		String outputFileName = commandLineParameters.getProperty(OUTFILE);
		if (outputFileName == null) {
			outputFileName = baseScriptFileName + ".out.txt"; //$NON-NLS-1$
		}
		
		String outputDirectoryName = commandLineParameters.getProperty(OUTPUTDIRECTORY);
		if (outputDirectoryName != null) {
			File outputDirectory = new File(outputDirectoryName);
			if (!(outputDirectory.isAbsolute() || outputDirectoryName.charAt(0) == File.separatorChar)) {
				outputDirectoryName = File.separatorChar + outputDirectoryName;
			}
			if (outputDirectory.exists()) {
				if (!outputDirectory.isDirectory()) {
					throw new FileNotFoundException(outputDirectoryName + Messages.getString("RunAllScripts.NotADirectory")); //$NON-NLS-1$
				}
			} else {
				outputDirectory.mkdirs();
			}
			if (new File(outputFileName).isAbsolute() || outputFileName.charAt(0) == File.separatorChar) {
				outputFileName = outputDirectoryName + File.separatorChar + new File(outputFileName).getName();
			} else {
				outputFileName = outputDirectoryName + File.separatorChar + outputFileName;
			}
			new File(outputFileName).getParentFile().mkdirs();
		} else {
			if (!(new File(outputFileName).isAbsolute() || outputFileName.charAt(0) == File.separatorChar)) {
				outputFileName = scriptFile.getParentFile().getAbsolutePath() + File.separatorChar + outputFileName;
			}
		}
		out = new PrintWriter(new BufferedOutputStream(new FileOutputStream(outputFileName)));

		String signingAlgorithmNameList = commandLineParameters.getProperty(DBConnectionProperties.SIGNINGALGORITHMS);
		if (signingAlgorithmNameList == null) {
			signingAlgorithmNameList = commandLineParameters.getProperty(DBConnectionProperties.SIGNINGALGORITHMS, ""); //$NON-NLS-1$
		}
		signingAlgorithmNames = DBConnectionProperties.getSigningAlgorithmNameSet(signingAlgorithmNameList);

		String cryptoAlgorithmNameList = commandLineParameters.getProperty(DBConnectionProperties.CRYPTOALGORITHMS);
		if (cryptoAlgorithmNameList == null) {
			cryptoAlgorithmNameList = commandLineParameters.getProperty(DBConnectionProperties.CRYPTOALGORITHMS, ""); //$NON-NLS-1$
		}
		cryptoAlgorithmNames = DBConnectionProperties.getCryptoAlgorithmNameSet(cryptoAlgorithmNameList);
	}

	/**
	 * Gets The private Key
	 * 
	 * @param algorithm
	 * @return The private Key
	 * @throws NotFoundException
	 */
	private PrivateKey getPrivateKey (String algorithm) throws NotFoundException {
		try {
			// Get the keystore
			KeyStore jks = MyKeyStore.getKeyStore(this.getClass());

			// Get the private Key from the keystore. This is the key that is stored in the keystore for the alias that is equal to the clientID, suffixed with the algorithm name
			return (PrivateKey) jks.getKey(CLIENTID + algorithm, this.getClass().getSimpleName().toCharArray());
		} catch (KeyStoreException e) {
			throw new NotFoundException(e.getClass().getName() + e.getMessage() != null ? e.getMessage() : ""); //$NON-NLS-1$
		} catch (NoSuchAlgorithmException e) {
			throw new NotFoundException(e.getClass().getName() + e.getMessage() != null ? e.getMessage() : ""); //$NON-NLS-1$
		} catch (CertificateException e) {
			throw new NotFoundException(e.getClass().getName() + e.getMessage() != null ? e.getMessage() : ""); //$NON-NLS-1$
		} catch (IOException e) {
			throw new NotFoundException(e.getClass().getName() + e.getMessage() != null ? e.getMessage() : ""); //$NON-NLS-1$
		} catch (UnrecoverableKeyException e) {
			throw new NotFoundException(e.getClass().getName() + e.getMessage() != null ? e.getMessage() : ""); //$NON-NLS-1$
		}
	}

	/**
	 * @param transaction
	 * @param autoCommit
	 * @param readOnlyTransaction
	 * @param transactionSymbolicName
	 * @return
	 * @throws ErrorMessageException
	 * @throws ConnectionClosedException
	 */
	private DBTransaction startNestedTransaction (DBTransaction transaction, boolean autoCommit, boolean readOnlyTransaction, String transactionSymbolicName) throws ConnectionClosedException, ErrorMessageException {
		if (transactionsByName.containsKey(transactionSymbolicName)) {
			throw new IllegalArgumentException("Duplicate transaction name " + transactionSymbolicName); //$NON-NLS-1$
		}

		DBTransaction startedTransaction = transaction.startNestedTransaction(autoCommit, readOnlyTransaction);

		transactionChain.addLast(startedTransaction);
		transactionsByName.put(transactionSymbolicName, startedTransaction);

		return startedTransaction;
	}

	/**
	 * Gets The Transaction object to which the script commands are to be communicated
	 * 
	 * @param dbc
	 *            The DBConnection on which to start a transaction
	 * @param autoCommit
	 *            The autoCommit status for the new transaction
	 * @param readOnlyTransaction
	 *            The read-only mode for the new transaction
	 * @param transactionSymbolicName
	 * @param explicitTransactionUser
	 * @return The Transaction object to which the script commands are to be communicated
	 * @throws DBException
	 * @throws ConnectionClosedException
	 * @throws ErrorMessageException
	 */
	private DBTransaction startTransaction (DBConnection dbc, boolean autoCommit, boolean readOnlyTransaction, String transactionSymbolicName, String explicitTransactionUser) throws DBException, ConnectionClosedException, ErrorMessageException {
		DBTransaction startedTransaction;
		String transactionUser = explicitTransactionUser == null ? user : explicitTransactionUser;
		if (transactionUser.length() > 0) {
			try {
				UserPrivateKeyProvider userPrivateKeyProvider = new UserPrivateKeyProvider(transactionUser);
				startedTransaction = dbc.startTransaction(transactionUser, userPrivateKeyProvider, autoCommit, TransactionMode.getTransactionMode(readOnlyTransaction));
			} catch (NotFoundException e3) {
				out.println(Messages.getString("ProcessScript.WarningTag") + MyMessageFormat.format(Messages.getString("ProcessScript.UserNotAuthenticable"), new String[] { transactionUser })); //$NON-NLS-1$ //$NON-NLS-2$
				startedTransaction = dbc.startTransaction(autoCommit, TransactionMode.getTransactionMode(readOnlyTransaction));
			}
		} else {
			startedTransaction = dbc.startTransaction(autoCommit, TransactionMode.getTransactionMode(readOnlyTransaction));
		}

		transactionChain.addLast(startedTransaction);
		transactionsByName.put(transactionSymbolicName, startedTransaction);

		return startedTransaction;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.lang.Runnable#run()
	 */
	public void run ( ) {
		Charset charSet = Charset.defaultCharset();
		String replyprompt = Messages.getString("ProcessScript.ReplyPrompt"); //$NON-NLS-1$
		try {
			DBConnection dbc = new OneShotDBConnection(host, port, signingAlgorithmNames, CLIENTID, cryptoAlgorithmNames, this);
			try {
				// DBTransaction scriptTransaction = null;
				boolean serverVersionBeforeONE_TWO = dbc.getSpecificationVersionForServer().isBefore(Version.ONE_TWO);
				if (serverVersionBeforeONE_TWO) {
					// scriptTransaction = startTransaction(dbc, false, false, ""); //$NON-NLS-1$
					startTransaction(dbc, false, false, "", null); //$NON-NLS-1$
				}
				boolean proceed = true;
				do {
					try {
						String cmd = getCommand(fin, charSet);
						if (cmd.startsWith("* ")) { //$NON-NLS-1$
							out.println(Messages.getString("ProcessScript.CommentTag") + cmd.substring(2)); //$NON-NLS-1$
							String w = cmd.substring(2).trim();
							if (w.toUpperCase().startsWith("CHARSET=")) { //$NON-NLS-1$
								charSet = Charset.forName(w.substring(8).trim());
							}
						} else {
							if (cmd.length() > 0) {
								out.println(Messages.getString("ProcessScript.RequestTag") + cmd); //$NON-NLS-1$
								cmd = cmd.trim();
								if (cmd.toUpperCase().startsWith("SET ")) { //$NON-NLS-1$
									SetContent setContent = getSetContent(cmd.substring(4).trim());
									host = setContent.getHost() == null ? host : setContent.getHost();
									port = setContent.getPort() == null ? port : Integer.parseInt(setContent.getPort());
									user = setContent.getUser() == null ? user : setContent.getUser();
								}
								boolean readOnlyTransaction = false;
								boolean autoCommit = false;
								if (cmd.toUpperCase().startsWith("STARTTRANSACTION ")) { //$NON-NLS-1$
									if (!serverVersionBeforeONE_TWO) {
										// Parse the statement STARTTRANSACTION SYMBOLICNAME[,readonly][,autocommit][,USER(username)]
										String cmdContent = cmd.substring(17).trim().toUpperCase(); // possible because user names are case-insensitive on the server anyway
										int i = cmdContent.indexOf(',');
										String transactionSymbolicName = null;
										if (i >= 0) {
											transactionSymbolicName = cmdContent.substring(0, i).trim();
											cmdContent = cmdContent.substring(i + 1).trim(); // READONLY[,AUTOCOMMIT] [USER()]
										} else {
											transactionSymbolicName = cmdContent.trim();
											cmdContent = ""; //$NON-NLS-1$
										}

										String explicitTransactionUser = null;
										while (cmdContent.length() > 0) {
											i = cmdContent.indexOf(',');
											String optionValue;
											if (i >= 0) {
												optionValue = cmdContent.substring(0, i).trim();
												cmdContent = cmdContent.substring(i + 1).trim();
											} else {
												optionValue = cmdContent;
												cmdContent = ""; //$NON-NLS-1$
											}
											if (optionValue.equals("READONLY")) { //$NON-NLS-1$
												readOnlyTransaction = true;
											} else {
												if (optionValue.equals("AUTOCOMMIT")) { //$NON-NLS-1$
													autoCommit = true;
												} else {
													if (optionValue.startsWith(DBConnectionProperties.USER)) {
														NameValueResult result = BracketParser.getNameValueFromNonEscaped(optionValue, 0);
														explicitTransactionUser = result.getNameValuePair().getValue();
													} else {
														throw new IllegalArgumentException(optionValue);
													}
												}
											}
										}

										// scriptTransaction = startTransaction(dbc, autoCommit, readOnlyTransaction, transactionSymbolicName);
										startTransaction(dbc, autoCommit, readOnlyTransaction, transactionSymbolicName, explicitTransactionUser);

										out.println(replyprompt);
									}
								} else {
									if (cmd.toUpperCase().startsWith("STARTNESTEDTRANSACTION ")) { //$NON-NLS-1$
										if (serverVersionBeforeONE_TWO) {
											throw new UnsupportedOperationException(Messages.getString("ProcessScript.NestedTransactionsWithPre12Server")); //$NON-NLS-1$
										}

										// Parse the statement STARTNESTEDTRANSACTION SYMBOLICNAME[,readonly][,autocommit]
										String cmdContent = cmd.substring(22).trim().toUpperCase(); // possible because user names are case-insensitive on the server anyway
										int i = cmdContent.indexOf(',');
										String transactionSymbolicName = null;
										if (i >= 0) {
											transactionSymbolicName = cmdContent.substring(0, i).trim();
											cmdContent = cmdContent.substring(i + 1).trim(); // [,READONLY][,AUTOCOMMIT]
										} else {
											transactionSymbolicName = cmdContent.trim();
											cmdContent = ""; //$NON-NLS-1$
										}

										while (cmdContent.length() > 0) {
											i = cmdContent.indexOf(',');
											String optionValue;
											if (i >= 0) {
												optionValue = cmdContent.substring(0, i).trim();
												cmdContent = cmdContent.substring(i + 1).trim();
											} else {
												optionValue = cmdContent;
												cmdContent = ""; //$NON-NLS-1$
											}
											if (optionValue.equals("READONLY")) { //$NON-NLS-1$
												readOnlyTransaction = true;
											} else {
												if (optionValue.equals("AUTOCOMMIT")) { //$NON-NLS-1$
													autoCommit = true;
												} else {
													throw new IllegalArgumentException(optionValue);
												}
											}
										}

										// scriptTransaction = startTransaction(dbc, autoCommit, readOnlyTransaction, transactionSymbolicName);
										startNestedTransaction(transactionChain.getLast(), autoCommit, readOnlyTransaction, transactionSymbolicName);

										out.println(replyprompt);
									} else {
										if (cmd.toUpperCase().startsWith("ENDTRANSACTION ") || cmd.equalsIgnoreCase("ENDTRANSACTION")) { //$NON-NLS-1$ //$NON-NLS-2$
											if (!serverVersionBeforeONE_TWO) {
												// Parse the statement ENDTRANSACTION [SYMBOLICNAME]
												if (cmd.trim().length() > 15) {
													// symbolic name given, fetch it from the transactionsByName map
													String transactionSymbolicName = cmd.substring(15).trim().toUpperCase(); // possible because user names are case-insensitive on the server anyway
													if (!transactionsByName.containsKey(transactionSymbolicName)) {
														throw new IllegalArgumentException(transactionSymbolicName);
													}
													DBTransaction endTransaction = transactionsByName.get(transactionSymbolicName);
													DBTransaction transaction;
													do {
														transaction = transactionChain.removeLast();
														transaction.end(true);
													} while (transaction != endTransaction);

													// if (transactionChain.size() > 0) {
													// scriptTransaction = transactionChain.getLast();
													// } else {
													// scriptTransaction = startTransaction(dbc, autoCommit, readOnlyTransaction, transactionSymbolicName);
													// }
												} else {
													DBTransaction transaction = transactionChain.removeLast();
													transaction.end(true);
												}
												out.println(replyprompt);
											}
										} else {
											if (cmd.toUpperCase().startsWith("COMMIT ") || cmd.equalsIgnoreCase("COMMIT")) { //$NON-NLS-1$ //$NON-NLS-2$
												if (!serverVersionBeforeONE_TWO) {
													if (cmd.trim().length() > 6) {
														String transactionSymbolicName = cmd.substring(6).trim().toUpperCase(); // possible because user names are case-insensitive on the server anyway
														if (!transactionsByName.containsKey(transactionSymbolicName)) {
															throw new IllegalArgumentException(transactionSymbolicName);
														}
														DBTransaction commitTransaction = transactionsByName.get(transactionSymbolicName);
														DBTransaction transaction;
														do {
															transaction = transactionChain.getLast();
															if (transaction != commitTransaction) {
																transaction.end(true);
																transactionChain.removeLast();
															} else {
																transaction.commit();
															}
														} while (transaction != commitTransaction);
													} else {
														transactionChain.getLast().commit();
													}

													out.println(replyprompt);
												} else {
													// Send the commit as a regular DML command, but only if no transaction name was specified
													if (cmd.trim().length() > 6) {
														throw new IllegalArgumentException(cmd);
													} else {
														String reply;
														try {
															AbstractRelation rsp = transactionChain.getLast().execDmlCommand(cmd);
															reply = rsp == null ? "" : rsp.printValueEscapedWithoutTypeNames(); //$NON-NLS-1$
														} catch (ErrorMessageException e) {
															reply = e.getMessage();
															// For 1.2 clients to deal with the behaviour of 1.1 servers (which automatically end the transaction in the event of errors)
															// if (serverVersionBeforeONE_TWO) {
															transactionChain.clear();
															// scriptTransaction = startTransaction(dbc, autoCommit, readOnlyTransaction, ""); //$NON-NLS-1$
															startTransaction(dbc, autoCommit, readOnlyTransaction, "", null); //$NON-NLS-1$
															// }
														}
														out.println(replyprompt + reply);
													}
												}

											} else {
												if (cmd.toUpperCase().startsWith("ROLLBACK ") || cmd.equalsIgnoreCase("ROLLBACK")) { //$NON-NLS-1$ //$NON-NLS-2$
													if (!serverVersionBeforeONE_TWO) {
														if (cmd.trim().length() > 8) {
															String transactionSymbolicName = cmd.substring(8).trim().toUpperCase(); // possible because user names are case-insensitive on the server anyway
															if (!transactionsByName.containsKey(transactionSymbolicName)) {
																throw new IllegalArgumentException(transactionSymbolicName);
															}
															DBTransaction commitTransaction = transactionsByName.get(transactionSymbolicName);
															DBTransaction transaction;
															do {
																transaction = transactionChain.getLast();
																if (transaction != commitTransaction) {
																	transaction.end(false);
																	transactionChain.removeLast();
																} else {
																	transaction.rollback();
																}
															} while (transaction != commitTransaction);
														} else {
															transactionChain.getLast().rollback();
														}

														out.println(replyprompt);
													} else {
														// Send the commit as a regular DML command, but only if no transaction name was specified
														if (cmd.trim().length() > 8) {
															throw new IllegalArgumentException(cmd);
														} else {
															String reply;
															try {
																AbstractRelation rsp = transactionChain.getLast().execDmlCommand(cmd);
																reply = rsp == null ? "" : rsp.printValueEscapedWithoutTypeNames(); //$NON-NLS-1$
															} catch (ErrorMessageException e) {
																reply = e.getMessage();
																// For 1.2 clients to deal with the behaviour of 1.1 servers (which automatically end the transaction in the event of errors)
																// if (serverVersionBeforeONE_TWO) {
																transactionChain.clear();
																// scriptTransaction = startTransaction(dbc, autoCommit, readOnlyTransaction, ""); //$NON-NLS-1$
																startTransaction(dbc, autoCommit, readOnlyTransaction, "", null); //$NON-NLS-1$
																// }
															}
															out.println(replyprompt + reply);
														}
													}
												} else {
													String reply;
													try {
														if (transactionChain.size() == 0) {
															// scriptTransaction = startTransaction(dbc, autoCommit, readOnlyTransaction, ""); //$NON-NLS-1$
															startTransaction(dbc, autoCommit, readOnlyTransaction, "", null); //$NON-NLS-1$
														}
														AbstractRelation rsp = transactionChain.getLast().execDmlCommand(cmd);
														reply = rsp == null ? "" : rsp.printValueEscapedWithoutTypeNames(); //$NON-NLS-1$
													} catch (ErrorMessageException e) {
														reply = e.getMessage();
														// For 1.2 clients to deal with the behaviour of 1.1 servers (which automatically end the transaction in the event of errors)
														if (serverVersionBeforeONE_TWO) {
															transactionChain.clear();
															// scriptTransaction = startTransaction(dbc, autoCommit, readOnlyTransaction, ""); //$NON-NLS-1$
															startTransaction(dbc, autoCommit, readOnlyTransaction, "", null); //$NON-NLS-1$
														}
													}
													out.println(replyprompt + reply);
												}
											}
										}
									}
								}
							} else {
								out.println();
							}
						}
					} catch (EOFException e1) {
						proceed = false;
					}
				} while (proceed);

				// scriptTransaction.end(true);
				while (transactionChain.size() > 0) {
					transactionChain.removeLast().end(true);
				}

				out.println("END"); //$NON-NLS-1$
			} finally {
				dbc.close();
			}
		} catch (Exception e) {
			String formattedMessage = MyMessageFormat.format(Messages.getString("ProcessScript.Exception"), new String[] { e.getClass().getName(), e.getMessage() }); //$NON-NLS-1$
			e.printStackTrace(out);
			out.println(formattedMessage);
			e.printStackTrace(System.err);
			System.err.println(formattedMessage);
		} finally {
			out.close();
			try {
				fin.close();
			} catch (IOException e2) {

			}
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.SIRAPRISE.security.PrivateKeyProvider#sign(java.security.Signature, byte[])
	 */
	public byte[] sign (Signature signature, byte[] signMessage) throws InvalidKeyException, SignatureException, NotFoundException {
		signature.initSign(getPrivateKey(signature.getAlgorithm()));
		signature.update(signMessage);
		return signature.sign();
	}
}