/*
 * Created on 12-sep-2007
 */
package be.SIRAPRISE.client;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.Arrays;
import java.util.TreeSet;
import java.util.Map.Entry;

import be.SIRAPRISE.util.CommandLineParameters;
import be.SIRAPRISE.util.SiraPriseConjunctFileNameFilterSet;
import be.SIRAPRISE.util.SiraPriseRegexFilter;
import be.SIRAPRISE.util.SiraPriseSuffixFilter;

/**
 * Invokes the Script processor for a whole series of scripts
 */

public class RunAllScripts {

	/**
	 * Executes all SIRA_PRISE scripts found in a directory to be specified
	 * 
	 * @param args
	 *            The options for this execution of the scripts batch. Options can be specified using either the syntax '-optionName:optionValue' or else 'opiontName(optionValue)'
	 *            <P>
	 *            The following options can be specified :
	 *            <TABLE border="1" cellpadding="4">
	 *            <TR>
	 *            <TH>Option name</TH>
	 *            <TH>Description</TH>
	 *            </TR>
	 *            <TR>
	 *            <TD>SCRIPTSDIRECTORY</TD>
	 *            <TD>The directory containing the SIRA_PRISE scripts to be executed</TD>
	 *            </TR>
	 *            <TR>
	 *            <TD>SCRIPTS</TD>
	 *            <TD>A pattern defining which scripts will be executed</TD>
	 *            </TR>
	 *            <TR>
	 *            <TD>OUTPUTDIRECTORY</TD>
	 *            <TD>The directory where the results of executing the SIRA_PRISE scripts are to be stored</TD>
	 *            </TR>
	 *            <TR>
	 *            <TD>HOST</TD>
	 *            <TD>The IP address or DNS name of the SIRA_PRISE server</TD>
	 *            </TR>
	 *            <TR>
	 *            <TD>PORT</TD>
	 *            <TD>The port to connect to</TD>
	 *            </TR>
	 *            <TR>
	 *            <TD>USER</TD>
	 *            <TD>The identification of the user owning the transactions that will be started on the server to perform the script commands.</TD>
	 *            </TR>
	 *            </TABLE>
	 */
	public static void main (String[] args) {
		try {
			CommandLineParameters commandLineParameters = CommandLineParameters.getCommandLineParameters(args, ProcessScript.SCRIPTSDIRECTORY, null);

			// Get the directory contents
			String scriptsDirectoryName = commandLineParameters.getProperty(ProcessScript.SCRIPTSDIRECTORY);
			if (scriptsDirectoryName == null) {
				System.out.println(Messages.getString("RunAllScripts.DirectorynameRequired")); //$NON-NLS-1$
				BufferedReader conin = new BufferedReader(new InputStreamReader(System.in));
				scriptsDirectoryName = conin.readLine();
			}

			File scriptsDirectory = new File(scriptsDirectoryName);
			if (!scriptsDirectory.isDirectory()) {
				throw new FileNotFoundException(scriptsDirectoryName + Messages.getString("RunAllScripts.NotADirectory")); //$NON-NLS-1$
			}
			String fullScriptsDirectoryName = scriptsDirectory.getAbsolutePath();

			// Get the directory contents
			String outputDirectoryName = commandLineParameters.getProperty(ProcessScript.OUTPUTDIRECTORY);
			if (outputDirectoryName == null) {
				System.out.println(Messages.getString("RunAllScripts.OutputDirectorynameRequired")); //$NON-NLS-1$
				BufferedReader conin = new BufferedReader(new InputStreamReader(System.in));
				outputDirectoryName = conin.readLine();
			}

			File outputDirectory = new File(outputDirectoryName);
			if (!outputDirectory.exists()) {
				outputDirectory.mkdirs();
			} else {
				if (!outputDirectory.isDirectory()) {
					throw new FileNotFoundException(outputDirectoryName + Messages.getString("RunAllScripts.NotADirectory")); //$NON-NLS-1$
				}
			}
			String fullOutputDirectoryName = outputDirectory.getAbsolutePath();

			TreeSet<String> scriptFileNames = new TreeSet<String>(Arrays.asList(scriptsDirectory.list(new SiraPriseConjunctFileNameFilterSet(Arrays.asList(new FilenameFilter[] { new SiraPriseSuffixFilter(ProcessScript.SPS), new SiraPriseRegexFilter(commandLineParameters.getProperty("SCRIPTS", ".*")) }))))); //$NON-NLS-1$//$NON-NLS-2$

			for (String scriptFileName : scriptFileNames) {
				String[] processScriptArgs = new String[commandLineParameters.size() + 2];
				int i = 1;
				for (Entry<Object, Object> entry : commandLineParameters.entrySet()) {
					processScriptArgs[i++] = (String) entry.getKey() + '(' + (String) entry.getValue() + ')';
				}

				System.out.println(scriptFileName);
				processScriptArgs[0] = fullScriptsDirectoryName + File.separatorChar + scriptFileName;
				processScriptArgs[i] = "OUTFILE(" + fullOutputDirectoryName + File.separatorChar + scriptFileName + ".out.txt)";  //$NON-NLS-1$//$NON-NLS-2$
				new ProcessScript(processScriptArgs).run();
				try {
					// close doesn't wait for a response and returns immediately. So the server might still be busy ending the
					// transaction while we're already finishing and heading off for the next script, which might then yield a maximumConnectionsExceeded
					Thread.sleep(150);
				} catch (InterruptedException e1) {

				}
			}
		} catch (FileNotFoundException e) {
			e.printStackTrace(System.err);
		} catch (IOException e) {
			e.printStackTrace(System.err);
		}
	}

	/**
	 * prevent instantiation
	 */
	private RunAllScripts ( ) {
		throw new RuntimeException();
	}
}