/**
 * 
 */
package be.SIRAPRISE.client;

import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;

import be.SIRAPRISE.client.NAMES.COMMANDNAMES;
import be.SIRAPRISE.util.BracketParser;
import be.SIRAPRISE.util.InvalidEscapedCharacterException;
import be.SIRAPRISE.util.MissingEscapedCharacterException;
import be.SIRAPRISE.util.NameValuePair;
import be.SIRAPRISE.util.NameValueResult;
import be.SIRAPRISE.util.NoClosingBracketException;
import be.SIRAPRISE.util.NoOpeningBracketException;
import be.SIRAPRISE.util.RelationValueSelectorFactory;
import be.erwinsmout.MyMessageFormat;
import be.erwinsmout.NotFoundException;

/**
 * A ServerCommand is any command that can get sent to the SIRA_PRISE server
 * 
 * @author Erwin
 */
public abstract class ServerCommand {

	/**
	 *
	 */
	private static final List<String> commandNames = Arrays.asList(new String[] { COMMANDNAMES.ADD, COMMANDNAMES.ASSERT, COMMANDNAMES.DELETE, COMMANDNAMES.UNASSERT, COMMANDNAMES.UPDATE, COMMANDNAMES.INQUIRE, COMMANDNAMES.INSERT, COMMANDNAMES.CREATE, COMMANDNAMES.REMOVE, COMMANDNAMES.MODIFY });

	/**
	 * @param commandText
	 * @return
	 * @throws IllegalServerCommandException
	 */
	private static ServerSingleAssignmentCommand getIndividualAssignmentCommandObjectFrom (String commandText) throws IllegalServerCommandException {
		ServerCommand serverCommand;
		try {
			serverCommand = getCommandObjectFrom(commandText);
		} catch (NotFoundException e) {
			throw new IllegalServerCommandException(e);
		}
		if (!(serverCommand instanceof ServerSingleAssignmentCommand)) {
			throw new IllegalServerCommandException(Messages.getString("ServerCommand.IndividualAssignmentsOnly")); //$NON-NLS-1$
		}
		return (ServerSingleAssignmentCommand) serverCommand;
	}

	/**
	 * Factory method for obtaining command objects, given a command name and the corresponding command options in texutal format
	 * 
	 * @param commandName
	 *            The name, or alias, of the command
	 * @param commandContent
	 *            The command options to be represented in the returned ServerCommand object
	 * @return A ServerCommand object representing the given commandText
	 * @throws NotFoundException
	 * @throws IllegalServerCommandException
	 */
	public static ServerCommand getCommandObjectFor (String commandName, String commandContent) throws NotFoundException, IllegalServerCommandException {
		int x = commandNames.indexOf(commandName.toUpperCase());
		switch (x) {
			case 0:
			case 6:
			case 7: {
				// ADD, INSERT, CREATE
				String targetRelvarName, addRelation;
				int i1 = commandContent.indexOf(',');
				if (i1 < 1) {
					throw new IllegalServerCommandException(MyMessageFormat.format(Messages.getString("ServerCommand.relvarNameMissing"), new String[] { commandContent })); //$NON-NLS-1$
				}

				targetRelvarName = commandContent.substring(0, i1).trim();
				addRelation = commandContent.substring(i1 + 1).trim();

				if (addRelation.length() < 1) {
					throw new IllegalServerCommandException(MyMessageFormat.format(Messages.getString("ServerCommand.relationalExpressionMissing"), new String[] { commandContent })); //$NON-NLS-1$
				}

				return new ServerAddCommand(targetRelvarName, addRelation);
			}
			case 1: {
				// ASSERT
				String targetRelvarName, assertRelation;
				int i1 = commandContent.indexOf(',');
				if (i1 < 1) {
					throw new IllegalServerCommandException(MyMessageFormat.format(Messages.getString("ServerCommand.relvarNameMissing"), new String[] { commandContent })); //$NON-NLS-1$
				}

				targetRelvarName = commandContent.substring(0, i1).trim();
				assertRelation = commandContent.substring(i1 + 1).trim();

				if (assertRelation.length() < 1) {
					throw new IllegalServerCommandException(MyMessageFormat.format(Messages.getString("ServerCommand.relationalExpressionMissing"), new String[] { commandContent })); //$NON-NLS-1$
				}

				return new ServerAssertCommand(targetRelvarName, assertRelation);
			}
			case 2:
			case 8: {
				// DELETE, REMOVE
				String targetRelvarName, deleteRelation;
				int i1 = commandContent.indexOf(',');
				if (i1 < 1) {
					throw new IllegalServerCommandException(MyMessageFormat.format(Messages.getString("ServerCommand.relvarNameMissing"), new String[] { commandContent })); //$NON-NLS-1$
				}

				targetRelvarName = commandContent.substring(0, i1).trim();
				deleteRelation = commandContent.substring(i1 + 1).trim();

				if (deleteRelation.length() < 1) {
					throw new IllegalServerCommandException(MyMessageFormat.format(Messages.getString("ServerCommand.relationalExpressionMissing"), new String[] { commandContent })); //$NON-NLS-1$
				}

				return new ServerDeleteCommand(targetRelvarName, deleteRelation);
			}
			case 3: {
				// UNASSERT
				String targetRelvarName, unassertRelation;
				int i1 = commandContent.indexOf(',');
				if (i1 < 1) {
					throw new IllegalServerCommandException(MyMessageFormat.format(Messages.getString("ServerCommand.relvarNameMissing"), new String[] { commandContent })); //$NON-NLS-1$
				}

				targetRelvarName = commandContent.substring(0, i1).trim();
				unassertRelation = commandContent.substring(i1 + 1).trim();

				if (unassertRelation.length() < 1) {
					throw new IllegalServerCommandException(MyMessageFormat.format(Messages.getString("ServerCommand.relationalExpressionMissing"), new String[] { commandContent })); //$NON-NLS-1$
				}

				return new ServerUnassertCommand(targetRelvarName, unassertRelation);
			}
			case 4:
			case 9: {
				// UPDATE, MODIFY
				String targetRelvarName, deleteRelationAndUpdates;
				int i1 = commandContent.indexOf(',');
				if (i1 < 1) {
					throw new IllegalServerCommandException(MyMessageFormat.format(Messages.getString("ServerCommand.relvarNameMissing"), new String[] { commandContent })); //$NON-NLS-1$
				}

				targetRelvarName = commandContent.substring(0, i1).trim();
				deleteRelationAndUpdates = commandContent.substring(i1 + 1).trim();

				if (deleteRelationAndUpdates.length() < 1) {
					throw new IllegalServerCommandException(MyMessageFormat.format(Messages.getString("ServerCommand.relationalExpressionMissing"), new String[] { commandContent })); //$NON-NLS-1$
				}

				int c1 = deleteRelationAndUpdates.indexOf(',');
				if (c1 == 0) {
					// stupid thingy to be able to specify e.g. ATTRIBUTE,,updateexpressionlist for a full relation update
					deleteRelationAndUpdates = targetRelvarName + deleteRelationAndUpdates;
					c1 = deleteRelationAndUpdates.indexOf(',');
				}
				int ob1 = deleteRelationAndUpdates.indexOf('(');
				if (c1 < 1 || c1 == deleteRelationAndUpdates.length() - 1 || ob1 < 0) {
					throw new IllegalServerCommandException(MyMessageFormat.format(Messages.getString("ServerCommand.scalarExpressionsMissing"), new String[] { deleteRelationAndUpdates })); //$NON-NLS-1$
				}
				String updates;
				String relationalExpression;
				if (c1 < ob1) {
					// opening bracket is preceded by a comma
					// ===> relation specification is just a name (most likely the same name as the first one, but not necessarily so !!!),
					// not a relational expression
					relationalExpression = deleteRelationAndUpdates.substring(0, c1).trim();
					updates = deleteRelationAndUpdates.substring(c1 + 1).trim();
				} else {
					try {
						NameValueResult nvr = BracketParser.getNameAndMandatoryValueFromEscaped(deleteRelationAndUpdates, 0);
						relationalExpression = deleteRelationAndUpdates.substring(0, nvr.getNextParsePos());
						updates = deleteRelationAndUpdates.substring(nvr.getNextParsePos()).trim();
						if (updates.length() < 1 || updates.charAt(0) != ',') {
							throw new IllegalServerCommandException(MyMessageFormat.format(Messages.getString("ServerCommand.scalarExpressionsMissing"), new String[] { deleteRelationAndUpdates })); //$NON-NLS-1$
						}
						updates = updates.substring(1).trim();
					} catch (NoOpeningBracketException impossible) {
						throw new RuntimeException();
					} catch (NoClosingBracketException e) {
						throw new IllegalServerCommandException(MyMessageFormat.format(Messages.getString("ServerCommand.syntaxError"), new String[] { deleteRelationAndUpdates }), e); //$NON-NLS-1$
					} catch (InvalidEscapedCharacterException e) {
						throw new IllegalServerCommandException(MyMessageFormat.format(Messages.getString("ServerCommand.syntaxError"), new String[] { deleteRelationAndUpdates }), e); //$NON-NLS-1$
					} catch (MissingEscapedCharacterException e) {
						throw new IllegalServerCommandException(MyMessageFormat.format(Messages.getString("ServerCommand.syntaxError"), new String[] { deleteRelationAndUpdates }), e); //$NON-NLS-1$
					}
				}

				return new ServerUpdateCommand(targetRelvarName, relationalExpression, BracketParser.removeSuperfluousOuterBrackets(updates));
			}
			case 5: {
				// INQUIRE
				String commandParameters = commandContent;
				int cp = commandParameters.indexOf(',');
				String orderingAttributesList = ""; //$NON-NLS-1$
				String queryExpression;
				LinkedList<String> orderByAttributeNames;
				if (cp < 0) {
					// No ordering attributes specified, only expression
					return new ServerInquireCommand(commandContent);
				} else {
					int obp = commandParameters.indexOf('(');
					if (obp < 0) {
						// No opening bracket, but a comma separator : e.g. DBMSFILE,FILENAME
						// Syntax error : ordering spec must always be in brackets
						throw new IllegalServerCommandException(MyMessageFormat.format(Messages.getString("ServerCommand.MissingParenthesesAroundOrderingSpec"), new String[] { commandParameters.substring(cp + 1) })); //$NON-NLS-1$
					}
					if (obp < cp) {
						// Comma possibly within the expression, get the expression and then check whether there is anything (ordering specification) still left
						NameValueResult nvr;
						try {
							nvr = BracketParser.getNameAndMandatoryValueFromEscaped(commandParameters, 0);
						} catch (NoOpeningBracketException e) {
							throw new IllegalServerCommandException(MyMessageFormat.format(Messages.getString("ServerCommand.SyntaxError"), new String[] { commandParameters }), e); //$NON-NLS-1$
						} catch (NoClosingBracketException e) {
							throw new IllegalServerCommandException(MyMessageFormat.format(Messages.getString("ServerCommand.SyntaxError"), new String[] { commandParameters }), e); //$NON-NLS-1$
						} catch (InvalidEscapedCharacterException e) {
							throw new IllegalServerCommandException(MyMessageFormat.format(Messages.getString("ServerCommand.SyntaxError"), new String[] { commandParameters }), e); //$NON-NLS-1$
						} catch (MissingEscapedCharacterException e) {
							throw new IllegalServerCommandException(MyMessageFormat.format(Messages.getString("ServerCommand.SyntaxError"), new String[] { commandParameters }), e); //$NON-NLS-1$
						}
						queryExpression = commandParameters.substring(0, nvr.getNextParsePos()).trim();
						String orderingSpec = commandParameters.substring(nvr.getNextParsePos()).trim();
						if (orderingSpec.length() == 0) {
							orderByAttributeNames = new LinkedList<String>();
						} else {
							if (!orderingSpec.startsWith(",")) { //$NON-NLS-1$
								throw new IllegalServerCommandException(MyMessageFormat.format(Messages.getString("ServerCommand.MissingCommaBeforeOrderingSpec"), new String[] { orderingSpec })); //$NON-NLS-1$
							}
							try {
								NameValuePair nameValuePair = BracketParser.getNameAndMandatoryValueFromEscaped(orderingSpec, 0).getNameValuePair();
								orderingAttributesList = nameValuePair.getValue();
								orderByAttributeNames = RelationValueSelectorFactory.getNamesInEntryOrder(orderingAttributesList.toUpperCase());
							} catch (NoOpeningBracketException e1) {
								throw new IllegalServerCommandException(MyMessageFormat.format(Messages.getString("ServerCommand.SyntaxError"), new String[] { orderingSpec }), e1); //$NON-NLS-1$
							} catch (NoClosingBracketException e1) {
								throw new IllegalServerCommandException(MyMessageFormat.format(Messages.getString("ServerCommand.SyntaxError"), new String[] { orderingSpec }), e1); //$NON-NLS-1$
							} catch (InvalidEscapedCharacterException e) {
								throw new IllegalServerCommandException(MyMessageFormat.format(Messages.getString("ServerCommand.SyntaxError"), new String[] { orderingSpec }), e); //$NON-NLS-1$
							} catch (MissingEscapedCharacterException e) {
								throw new IllegalServerCommandException(MyMessageFormat.format(Messages.getString("ServerCommand.SyntaxError"), new String[] { orderingSpec }), e); //$NON-NLS-1$
							}
						}
					} else {
						queryExpression = commandParameters.substring(0, cp);
						String orderingSpec = commandParameters.substring(cp + 1);
						try {
							NameValuePair nameValuePair = BracketParser.getNameAndMandatoryValueFromEscaped(orderingSpec, 0).getNameValuePair();
							orderingAttributesList = nameValuePair.getValue();
							orderByAttributeNames = RelationValueSelectorFactory.getNamesInEntryOrder(orderingAttributesList.toUpperCase());
						} catch (NoOpeningBracketException e1) {
							throw new IllegalServerCommandException(MyMessageFormat.format(Messages.getString("ServerCommand.SyntaxError"), new String[] { orderingSpec }), e1); //$NON-NLS-1$
						} catch (NoClosingBracketException e1) {
							throw new IllegalServerCommandException(MyMessageFormat.format(Messages.getString("ServerCommand.SyntaxError"), new String[] { orderingSpec }), e1); //$NON-NLS-1$
						} catch (InvalidEscapedCharacterException e) {
							throw new IllegalServerCommandException(MyMessageFormat.format(Messages.getString("ServerCommand.SyntaxError"), new String[] { orderingSpec }), e); //$NON-NLS-1$
						} catch (MissingEscapedCharacterException e) {
							throw new IllegalServerCommandException(MyMessageFormat.format(Messages.getString("ServerCommand.SyntaxError"), new String[] { orderingSpec }), e); //$NON-NLS-1$
						}
					}
				}

				return new ServerInquireCommand(queryExpression, orderByAttributeNames);
			}
		}
		throw new NotFoundException(MyMessageFormat.format(Messages.getString("ServerCommand.invalidCommand"), new String[] { commandNames.toString(), commandName })); //$NON-NLS-1$
	}

	/**
	 * Factory method for obtaining command objects, given a fulltext command such as "ASSERT R,R(...)" or "CMD(...)CMD(...)"
	 * 
	 * @param commandText
	 *            A complete SIRA_PRISE command
	 * @return A ServerCommand object representing the given commandText
	 * @throws NotFoundException
	 * @throws IllegalServerCommandException
	 */
	public static ServerCommand getCommandObjectFrom (String commandText) throws NotFoundException, IllegalServerCommandException {
		if (commandText.length() > 2 && commandText.substring(0, 3).equalsIgnoreCase(COMMANDNAMES.CMD)) {
			LinkedList<String> commandList;
			try {
				commandList = BracketParser.createListFromEscaped(commandText);
			} catch (NoClosingBracketException e) {
				throw new IllegalServerCommandException(commandText, e);
			} catch (InvalidEscapedCharacterException e) {
				throw new IllegalServerCommandException(commandText, e);
			} catch (MissingEscapedCharacterException e) {
				throw new IllegalServerCommandException(commandText, e);
			} catch (NoOpeningBracketException e) {
				throw new IllegalServerCommandException(commandText, e);
			}

			int i = 0;
			ServerSingleAssignmentCommand[] serverAssignmentCommands = new ServerSingleAssignmentCommand[commandList.size()];
			for (String individualCommandText : commandList) {
				serverAssignmentCommands[i++] = ServerCommand.getIndividualAssignmentCommandObjectFrom(individualCommandText);
			}

			if (serverAssignmentCommands.length == 1) {
				return serverAssignmentCommands[0];
			} else {
				return new ServerMultipleAssignmentCommand(serverAssignmentCommands);
			}

		} else {

			int bp = commandText.indexOf(' ');
			String commandName;
			String commandContent;
			if (bp < 0) {
				commandName = commandText;
				commandContent = ""; //$NON-NLS-1$
			} else {
				commandName = commandText.substring(0, bp).trim();
				commandContent = commandText.substring(bp + 1).trim();
			}

			return ServerCommand.getCommandObjectFor(commandName, commandContent);
		}
	}

	/**
	 * Gets the full textual representation of the command
	 * 
	 * @return the full textual representation of the command
	 */
	public abstract String getFullCommandText ( );

	/**
	 * Gets the full textual representation of the command's parameters
	 * 
	 * @return the full textual representation of the command's parameters
	 */
	public abstract String getCommandParametersText ( );

	/**
	 * Gets the MessageTypeID of the corresponding message type to send to the server to get this (type of) command executed
	 * 
	 * @return the MessageTypeID of the corresponding message type to send to the server to get this (type of) command executed
	 */
	abstract int getCorrespondingMessagetypeid ( );

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString ( ) {
		return getFullCommandText();
	}
}
