/*
 * Created on 14-sep-2007
 */
package be.SIRAPRISE.client;

import java.util.Collection;
import java.util.Iterator;
import java.util.Map;

import be.SIRAPRISE.client.jsba.ClassDoesNotImplementDBObjectException;

/**
 * The TupleContainer interface defines all methods that must be implemented by tuple containers of any kind.
 * 
 * @author Erwin Smout
 */
public interface TupleContainer extends Iterable<Tuple> {

	/**
	 * Adds all tuples of the given TupleContainer to this one. The assignment-compatibility check between the Heading of the given tupleContainer and this one is to be carried out. It is assumed that all tuples contained in the given tupleContainer conform to that container's Heading, so no assignment compatibility checks are needed between the 'source' tuples' Headings and this Container's Heading.
	 * 
	 * @param tupleContainer
	 *            The TupleContainer whose tuples are all to be added to this one
	 */
	public void addTuplesWithHeadingCheck (TupleContainer tupleContainer);

	/**
	 * Adds all tuples of the given TupleContainer to this one. The assignment-compatibility check between the involved Headings is skipped. This means that it must be guaranteed 'externally' that the tuples to be added do conform to this container's Heading.
	 * 
	 * @param tupleContainer
	 *            The TupleContainer whose tuples are all to be added to this one
	 */
	public void addTuplesWithoutHeadingCheck (TupleContainer tupleContainer);

	/**
	 * Adds a Tuple to the container. The tuple's Heading is checked for conformance with the Container's Heading.
	 * 
	 * @param tuple
	 *            The tuple to be added to the container
	 */
	public void addTupleWithHeadingCheck (Tuple tuple);

	/**
	 * Adds a Tuple to the container without checking the conformance of the tuple to the declared heading. The conformance should be guaranteed externally to this method.
	 * 
	 * @param tuple
	 *            The tuple to be added to the container
	 */
	public void addTupleWithoutHeadingCheck (Tuple tuple);

	/**
	 * Empties the TupleContainer
	 */
	public void clear ( );

	/**
	 * Checks whether the given tuple appears in this container
	 * 
	 * @param t
	 *            The tuple value to be tested for presence in the container
	 * @return true if the tuple value is present in this container
	 */
	public boolean containsTuple (Tuple t);

	/**
	 * Checks whether all tuples that appear in the given container also appear in this one
	 * 
	 * @param tupleContainer
	 *            The tuple container whose tuples must all be contained in this one for the method to return true
	 * @return true if all tuples that appear in the given container also appear in this one
	 */
	public boolean containsTuples (TupleContainer tupleContainer);

	/**
	 * Gets the Heading of the tuple container
	 * 
	 * @return the Heading of the tuples container
	 */
	public Heading getHeading ( );

	/**
	 * Gets an iterator over the tuples in the container. The objects returned by the iterator's next() method will be Tuple obejcts.
	 * 
	 * @return an iterator over the tuples in the container
	 */
	public Iterator<Tuple> iterator ( );

	/**
	 * Gets this relation value in SIRAPRISE syntactical format (i.e. including the SIRA_PRISE escape token preceding any appearance of the tokens '(', ')' and '\' in scalar attribute values in the relation.
	 * 
	 * @return this relation value in SIRAPRISE syntactical format
	 */
	public String printValueEscapedWithoutTypeNames ( );

	/**
	 * Gets this relation value in SIRAPRISE syntactical format (i.e. including the SIRA_PRISE escape token preceding any appearance of the tokens '(', ')' and '\' in scalar attribute values in the relation.
	 * 
	 * @return this relation value in SIRAPRISE syntactical format
	 */
	public String printValueEscapedWithTypeNames ( );

	/**
	 * Gets this relation value in 'plain text' format (i.e. if a scalar attribute value includes any of the tokens '(', ')' or '\', these will appear 'as is' in the string value and will not be replaced by the appropriate SIRA_PRISE escape sequence.
	 * 
	 * @return this relation value in 'plain text' format.
	 */
	public String printValueWithoutTypeNames ( );

	/**
	 * Gets this relation value in 'plain text' format (i.e. if a scalar attribute value includes any of the tokens '(', ')' or '\', these will appear 'as is' in the string value and will not be replaced by the appropriate SIRA_PRISE escape sequence.
	 * 
	 * @return this relation value in 'plain text' format.
	 */
	public String printValueWithTypeNames ( );

	/**
	 * Gets this relation value in 'XML text' format.
	 * 
	 * @return this relation value in 'plain text' format.
	 */
	public String printValueXML ( );

	/**
	 * Removes the given tuple from this container if that tuple appears there
	 * 
	 * @param t
	 *            The tuple to be removed from the container
	 * @return true if the tuple was removed, false if it was not found in this TupleContainer
	 */
	public boolean removeTuple (Tuple t);

	/**
	 * Removes all tuples appearing in the given tupleContainer from this tupleContainer if that tuple appears there
	 * 
	 * @param tupleContainer
	 *            The tupleContainer holding all the tuples to be removed from the container
	 * @return the number of tuples effectively removed
	 */
	public int removeTuples (TupleContainer tupleContainer);

	/**
	 * Gets The number of tuples that appear in the relation.
	 * 
	 * @return The number of tuples that appear in the relation.
	 */
	public int size ( );

	/**
	 * Get an array of objects holding the same information as that which is held in this TupleContainer. One object is created for each tuple.
	 * 
	 * @param <C>
	 *            The name of the objectClass, also naming the class/object type of the objects making up the returned array
	 * @param objectClass
	 *            A Class object denoting the class of the objects to be returned. objectClass must denote a public, non-abstract class that has a public no-arg constructor, and that has a unique setter method for each attribute in the heading. The name of this unique method must be the concatenation of the word "set" (all lowercase), followed by the uppercased attribute name (i.e. first character uppercase, all others lowercase), followed by "FromDB". The argument list of this setter method must consist of excatly one java.lang.String argument.
	 * @return An array of objects of the given object Class.
	 * @throws ConstructorMissingException
	 *             if objectClass does not denote a public class, or it denotes an abstract class, or the class does not have a public no-arg constructor
	 * @throws SettersMissingException
	 *             if a needed setter is missing in the class denoted by objectClass
	 * @throws ClassDoesNotImplementDBObjectException
	 *             if objectClass does not implement the required DBObject interface
	 * @deprecated - Use the replacing DBObjectFactory method instead.
	 */
	public <C> C[] toObjectArray (Class<C> objectClass) throws ConstructorMissingException, SettersMissingException, ClassDoesNotImplementDBObjectException;

	/**
	 * Get an array of objects holding a (potentially proper) subset of the information held in this TupleContainer. One object is created for each tuple. Setters matching an attribute in the heading will be invoked, attributes for which no corresponding setter is found, are ignored.
	 * 
	 * @param <C>
	 *            The name of the objectClass, also naming the class/object type of the objects making up the returned array
	 * @param objectClass
	 *            A Class object denoting the class of the objects to be returned. objectClass must denote a public, non-abstract class that has a public no-arg constructor, and that has at most one unique setter method for each attribute in the heading. The name of this unique method must be the concatenation of the word "set" (all lowercase), followed by the uppercased attribute name (i.e. first character uppercase, all others lowercase), followed by "FromDB". The argument list of this setter method must consist of excatly one java.lang.String argument.
	 * @return An array of objects of the given object Class.
	 * @throws ConstructorMissingException
	 *             if objectClass does not denote a public class, or it denotes an abstract class, or the class does not have a public no-arg constructor
	 * @throws ClassDoesNotImplementDBObjectException
	 *             if objectClass does not implement the required DBObject interface
	 * @deprecated - Use the replacing DBObjectFactory method instead.
	 */
	public <C> C[] toObjectArrayIgnoringMissingSetters (Class<C> objectClass) throws ConstructorMissingException, ClassDoesNotImplementDBObjectException;

	/**
	 * Get a collection of objects holding the same information as that which is held in this TupleContainer. One object is created for each tuple.
	 * 
	 * @param <C>
	 * @param objectClass
	 *            A Class object denoting the class of the objects to be returned. objectClass must denote a public, non-abstract class that has a public no-arg constructor, and that has a unique setter method for each attribute in the heading. The name of this unique method must be the concatenation of the word "set" (all lowercase), followed by the uppercased attribute name (i.e. first character uppercase, all others lowercase), followed by "FromDB". The argument list of this setter method must consist of excatly one java.lang.String argument.
	 * @return An array of objects of the given object Class.
	 * @throws ConstructorMissingException
	 *             if objectClass does not denote a public class, or it denotes an abstract class, or the class does not have a public no-arg constructor
	 * @throws SettersMissingException
	 *             if a needed setter is missing in the class denoted by objectClass
	 * @throws ClassDoesNotImplementDBObjectException
	 *             if objectClass does not implement the required DBObject interface
	 */
	public <C> Collection<C> toObjectCollection (Class<C> objectClass) throws ConstructorMissingException, SettersMissingException, ClassDoesNotImplementDBObjectException;

	/**
	 * Get a collection of objects holding a (potentially proper) subset of the information held in this TupleContainer. One object is created for each tuple. Setters matching an attribute in the heading will be invoked, attributes for which no corresponding setter is found, are ignored.
	 * 
	 * @param <C>
	 * @param objectClass
	 *            A Class object denoting the class of the objects to be returned. objectClass must denote a public, non-abstract class that has a public no-arg constructor, and that has at most one unique setter method for each attribute in the heading. The name of this unique method must be the concatenation of the word "set" (all lowercase), followed by the uppercased attribute name (i.e. first character uppercase, all others lowercase), followed by "FromDB". The argument list of this setter method must consist of excatly one java.lang.String argument.
	 * @return An array of objects of the given object Class.
	 * @throws ConstructorMissingException
	 *             if objectClass does not denote a public class, or it denotes an abstract class, or the class does not have a public no-arg constructor
	 * @throws ClassDoesNotImplementDBObjectException
	 *             if objectClass does not implement the required DBObject interface
	 */
	public <C> Collection<C> toObjectCollectionIgnoringMissingSetters (Class<C> objectClass) throws ConstructorMissingException, ClassDoesNotImplementDBObjectException;

	/**
	 * Get an array of Maps of objects holding the same information as that which is held in this TupleContainer. One Map is returned for each tuple in the container. The entries in the map have one of the objectClass[] objects as their key, and an object of that class as their value. The object of that class holds the attribute values of the tuple for which a corresponding setter method was found in the class.
	 * 
	 * @param objectClass
	 *            An array of Class objects denoting the classes of the objects to be returned for each tuple. Each objectClass must denote a public, non-abstract class that has a public no-arg constructor, and that has a unique setter method for some attribute in the heading. The name of this unique method must be the concatenation of the word "set" (all lowercase), followed by the uppercased attribute name (i.e. first character uppercase, all others lowercase), followed by "FromDB". The argument list of this setter method must consist of excatly one java.lang.String argument.
	 * @return The array of maps of objects holding the same information as this relation.
	 * @throws ConstructorMissingException
	 *             if some objectClass does not denote a public class, or it denotes an abstract class, ordenotes a class that does not have a public no-arg constructor
	 * @throws SettersMissingException
	 *             if some attribute in the heading exists for which no corresponding setter could be found in any of the classes denoted by objectClass[]
	 * @throws ClassDoesNotImplementDBObjectException
	 *             if objectClass does not implement the required DBObject interface
	 */
	public Map<Class<?>, Object>[] toObjectsArray (Class<?>[] objectClass) throws ConstructorMissingException, SettersMissingException, ClassDoesNotImplementDBObjectException;

	/**
	 * Get a collection of Maps of objects holding the same information as that which is held in this TupleContainer. One Map is returned for each tuple in the container. The entries in the map have one of the objectClass[] objects as their key, and an object of that class as their value. The object of that class holds the attribute values of the tuple for which a corresponding setter method was found in the class.
	 * 
	 * @param objectClass
	 *            An array of Class objects denoting the classes of the objects to be returned for each tuple. Each objectClass must denote a public, non-abstract class that has a public no-arg constructor, and that has a unique setter method for some attribute in the heading. The name of this unique method must be the concatenation of the word "set" (all lowercase), followed by the uppercased attribute name (i.e. first character uppercase, all others lowercase), followed by "FromDB". The argument list of this setter method must consist of excatly one java.lang.String argument.
	 * @return The array of maps of objects holding the same information as this relation.
	 * @throws ConstructorMissingException
	 *             if some objectClass does not denote a public class, or it denotes an abstract class, ordenotes a class that does not have a public no-arg constructor
	 * @throws SettersMissingException
	 *             if some attribute in the heading exists for which no corresponding setter could be found in any of the classes denoted by objectClass[]
	 * @throws ClassDoesNotImplementDBObjectException
	 *             if objectClass does not implement the required DBObject interface
	 */
	public Collection<Map<Class<?>, Object>> toObjectsCollection (Class<?>[] objectClass) throws ConstructorMissingException, SettersMissingException, ClassDoesNotImplementDBObjectException;
}