/*
 * Created on 24-jul-2008
 */
package be.SIRAPRISE.client;

/**
 * Class to denote SIRA_PRISE versions and/or message type versions.
 * 
 * @author Erwin Smout
 */
public class Version implements Comparable<Version> {

	/**
	 * 
	 */
	public static final Version ONE_FOUR = new Version((short) 1, (short) 4);

	/**
	 * 
	 */
	public static final Version ONE_ONE = new Version((short) 1, (short) 1);

	/**
	 * 
	 */
	public static final Version ONE_THREE = new Version((short) 1, (short) 3);

	/**
	 * 
	 */
	public static final Version ONE_TWO = new Version((short) 1, (short) 2);

	/**
	 * in order to not have to create identical object within each message type
	 */
	public static final Version ONE_ZERO = new Version((short) 1, (short) 0);

	/**
	 * The major version
	 */
	private short majorVersion;

	/**
	 * The minor version
	 */
	private short minorVersion;

	/**
	 * Creates a Version object from a full numeric version indicator
	 * 
	 * @param fullVersion
	 *            A full numeric version indicator holding the major version number in its first two bytes and the minor version in its last two bytes.
	 */
	public Version (int fullVersion) {
		this((short) (fullVersion >>> 16), (short) (fullVersion & 0x0000ffff));
	}

	/**
	 * Creates a Version object from the given major and minor version numbers.
	 * 
	 * @param majorVersion
	 *            The major version
	 * @param minorVersion
	 */
	public Version (short majorVersion, short minorVersion) {
		if (majorVersion < 1 || minorVersion < 0) {
			throw new IllegalArgumentException(Messages.getString("Version.NegativeComponent")); //$NON-NLS-1$
		}
		this.majorVersion = majorVersion;
		this.minorVersion = minorVersion;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.lang.Comparable#compareTo(java.lang.Object)
	 */
	public final int compareTo (Version o) {
		if (equals(o)) {
			return 0;
		}
		if (isBefore(o)) {
			return -1;
		} else {
			return 1;
		}
	}

	public final boolean equals (Object obj) {
		if (obj instanceof Version) {
			Version otherVersion = (Version) obj;
			return otherVersion.majorVersion == majorVersion && otherVersion.minorVersion == minorVersion;
		}
		return false;
	}

	/**
	 * Gets the full numeric indicator of the version value.
	 * 
	 * @return the full numeric indicator of the version value.
	 */
	public final int getFullVersion ( ) {
		return majorVersion << 16 | minorVersion;
	}

	/**
	 * Compares two SIRA_PRISE Version indicators and returns the lowest (which is the one whose spec can be expected to define the functionality that the two versions have in common).
	 * 
	 * @param otherVersion
	 *            the other SIRA_PRISE Version indicator
	 * @return the Version object indicating the lowest (oldest) version number
	 */
	public final Version getLowest (Version otherVersion) {
		if (isBefore(otherVersion))
			return this;
		return otherVersion;
	}

	/**
	 * Gets The major version
	 * 
	 * @return The major version
	 */
	public final short getMajorVersion ( ) {
		return majorVersion;
	}

	/**
	 * Gets The minor version
	 * 
	 * @return The minor version
	 */
	public final short getMinorVersion ( ) {
		return minorVersion;
	}

	public final int hashCode ( ) {
		return getFullVersion();
	}

	/**
	 * Checks whether this version indicates a version that is at least (i.e the same, or later than) the one indicated by the given arguments
	 * 
	 * @param majorVersion2
	 *            The major version number to which this version is to be compared
	 * @param minorVersion2
	 *            The minor version number to which this version is to be compared
	 * @return true if this version indicates a version that is at least (i.e the same, or later than) the one indicated by the given arguments
	 */
	public final boolean isAtLeast (short majorVersion2, short minorVersion2) {
		return (majorVersion > majorVersion2 || (majorVersion == majorVersion2 && minorVersion >= minorVersion2));
	}

	/**
	 * Checks whether this version indicates a version that is at least (i.e the same, or later than) the one indicated by the given arguments
	 * 
	 * @param otherVersion
	 *            A Version object designating another SIRA_PRISE version this one is to be compared to.
	 * @return true if this version indicates a version that is at least (i.e the same, or later than) the one indicated by the given arguments
	 */
	public final boolean isAtLeast (Version otherVersion) {
		return (this.majorVersion > otherVersion.majorVersion || (this.majorVersion == otherVersion.majorVersion && this.minorVersion >= otherVersion.minorVersion));
	}

	/**
	 * Checks whether this version indicates an earlier version than the one indicated by the given arguments
	 * 
	 * @param majorVersion2
	 *            The major version number to which this version is to be compared
	 * @param minorVersion2
	 *            The minor version number to which this version is to be compared
	 * @return true if this version indicates an earlier version than the one indicated by the given arguments
	 */
	public final boolean isBefore (short majorVersion2, short minorVersion2) {
		return (majorVersion < majorVersion2 || (majorVersion == majorVersion2 && minorVersion < minorVersion2));
	}

	/**
	 * Checks if this Version object designates an earlier version than the other one
	 * 
	 * @param otherVersion
	 *            A Version object designating another SIRA_PRISE version this one is to be compared to for "earlierness".
	 * @return true if this version designates an earlier version than the other Version
	 */
	public final boolean isBefore (Version otherVersion) {
		return (this.majorVersion < otherVersion.majorVersion || (this.majorVersion == otherVersion.majorVersion && this.minorVersion < otherVersion.minorVersion));
	}

	public final String toString ( ) {
		return Integer.toString(majorVersion) + '.' + Integer.toString(minorVersion);
	}
}