/*
 * Created on 13-jan-2009
 */
package be.SIRAPRISE.client.jsba;

import be.SIRAPRISE.client.DmlAddCommand;
import be.SIRAPRISE.client.DmlAssertCommand;
import be.SIRAPRISE.client.DmlAssignmentCommand;
import be.SIRAPRISE.client.DmlDeleteCommand;
import be.SIRAPRISE.client.DmlMultipleAssignmentCommand;
import be.SIRAPRISE.client.DmlUnassertCommand;

/**
 * An abstract class that can be inherited from by classes that need to implement DBObject, providing default implementations for the Java/SIRA_PRISE bridge methods
 * 
 * @author Erwin Smout
 */
public abstract class AbstractDBObject implements DBObject {

	/**
	 * Gets the command for adding an entry in the named relvar (in the database the transaction is connected to), for the given object. The tuple that is registered in the database is constructed by invoking all the object's methods that satisfy the following properties :
	 * <ul>
	 * <li>The name of the method is the word "get" (all lowercase), followed by the capitalized relvar name (i.e. first character of the relvar name uppercase, all remaining characters of the relvar name lowercase), followed by a capitalized attribute name (i.e. the first character of the remaining part of the method name must be uppercase and all the other remaining characters must be lowercase)</LI>
	 * <li>The method is a public method that takes no arguments</LI>
	 * <li>The return type of the method is java.lang.String</LI>
	 * </UL>
	 * 
	 * @param relvarName
	 *            The name of the relvar into which a tuple is to be added for the given object
	 * @param o
	 *            The object from which the tuple to be registered in the named relvar is to be drawn.
	 * @return The add command to be issued for adding an entry in the named relvar with attribute values drawn from the given object
	 */
	public static DmlAddCommand dbAddCommand (String relvarName, DBObject o) {
		return new DmlAddCommand(relvarName, o);
	}

	/**
	 * Gets the Assert command for asserting an entry in the named relvar (in the database the transaction is connected to), for the given object. The tuple that is registered in the database is constructed by invoking all the object's methods that satisfy the following properties :
	 * <ul>
	 * <li>The name of the method is the word "get" (all lowercase), followed by the capitalized relvar name (i.e. first character of the relvar name uppercase, all remaining characters of the relvar name lowercase), followed by a capitalized attribute name (i.e. the first character of the remaining part of the method name must be uppercase and all the other remaining characters must be lowercase)</LI>
	 * <li>The method is a public method that takes no arguments</LI>
	 * <li>The return type of the method is java.lang.String</LI>
	 * </UL>
	 * 
	 * @param relvarName
	 *            The name of the relvar into which a tuple is to be added for the given object
	 * @param o
	 *            The object from which the tuple to be registered in the named relvar is to be drawn.
	 * @return The assert command to be issued for adding an entry in the named relvar with attribute values drawn from the given object
	 */
	public static DmlAssertCommand dbAssertCommand (String relvarName, DBObject o) {
		return new DmlAssertCommand(relvarName, o);
	}

	/**
	 * Gets the command for deleting the tuple in the named relvar (in the database the transaction is connected to), for the given object. The tuple that is registered in the database is constructed by invoking all the object's methods that satisfy the following properties :
	 * <ul>
	 * <li>The name of the method is the word "get" (all lowercase), followed by the capitalized relvar name (i.e. first character of the relvar name uppercase, all remaining characters of the relvar name lowercase), followed by a capitalized attribute name (i.e. the first character of the remaining part of the method name must be uppercase and all the other remaining characters must be lowercase)</LI>
	 * <li>The method is a public method that takes no arguments</LI>
	 * <li>The return type of the method is java.lang.String</LI>
	 * </UL>
	 * 
	 * @param relvarName
	 *            The name of the relvar into which a tuple is to be added for the given object
	 * @param o
	 *            The object from which the tuple to be registered in the named relvar is to be drawn.
	 * @return the command for deleting the tuple in the named relvar (in the database the transaction is connected to), for the given object.
	 */
	public static DmlDeleteCommand dbDeleteCommand (String relvarName, DBObject o) {
		return new DmlDeleteCommand(relvarName, o);
	}

	/**
	 * Gets the command for unasserting the tuple in the named relvar (in the database the transaction is connected to), for the given object. The tuple that is registered in the database is constructed by invoking all the object's methods that satisfy the following properties :
	 * <ul>
	 * <li>The name of the method is the word "get" (all lowercase), followed by the capitalized relvar name (i.e. first character of the relvar name uppercase, all remaining characters of the relvar name lowercase), followed by a capitalized attribute name (i.e. the first character of the remaining part of the method name must be uppercase and all the other remaining characters must be lowercase)</LI>
	 * <li>The method is a public method that takes no arguments</LI>
	 * <li>The return type of the method is java.lang.String</LI>
	 * </UL>
	 * 
	 * @param relvarName
	 *            The name of the relvar into which a tuple is to be added for the given object
	 * @param o
	 *            The object from which the tuple to be registered in the named relvar is to be drawn.
	 * @return the command for unasserting the tuple in the named relvar (in the database the transaction is connected to), for the given object.
	 */
	public static DmlUnassertCommand dbUnAssertCommand (String relvarName, DBObject o) {
		return new DmlUnassertCommand(relvarName, o);
	}

	/**
	 * Gets the command for adding (in the database the transaction is connected to) an entry in the single relvar named in this object's class' @forrelvar annotation, for the given object.
	 * 
	 * @return the command for adding (in the database the transaction is connected to) an entry in the single relvar named in this object's class' @forrelvar annotation, for the given object.
	 */
	public final DmlAddCommand dbAddCommand () {
		final ForRelvars annotation = this.getClass().getAnnotation(ForRelvars.class);
		if (annotation == null) {
			throw new DBObjectNotAnnotatedForASingleRelvarException();
		}
		String[] annotationRelvarNames = annotation.relvarNames();
		if (annotationRelvarNames.length != 1) {
			throw new DBObjectNotAnnotatedForASingleRelvarException(annotationRelvarNames);
		}
		return new DmlAddCommand(annotationRelvarNames[0], this);
	}

	/**
	 * Gets the command for adding an entry in the named relvar (in the database the transaction is connected to), for the given object. The tuple that is registered in the database is constructed by invoking all the object's methods that satisfy the following properties :
	 * <ul>
	 * <li>The name of the method is the word "get" (all lowercase), followed by the capitalized relvar name (i.e. first character of the relvar name uppercase, all remaining characters of the relvar name lowercase), followed by a capitalized attribute name (i.e. the first character of the remaining part of the method name must be uppercase and all the other remaining characters must be lowercase)</LI>
	 * <li>The method is a public method that takes no arguments</LI>
	 * <li>The return type of the method is java.lang.String</LI>
	 * </UL>
	 * 
	 * @param relvarName
	 *            The name of the relvar into which a tuple is to be added for the given object
	 * @return The add command to be issued for adding an entry in the named relvar with attribute values drawn from the given object
	 */
	public final DmlAddCommand dbAddCommand (String relvarName) {
		return new DmlAddCommand(relvarName, this);
	}

	/**
	 * Gets the command for adding (in the database the transaction is connected to) an entry in all the relvars named in this object's class' @forrelvar annotation, for the given object.
	 * 
	 * @return the command for adding (in the database the transaction is connected to) an entry in all the relvars named in this object's class' @forrelvar annotation, for the given object.
	 */
	public final DmlAssignmentCommand dbAddCommandAll () {
		final ForRelvars annotation = this.getClass().getAnnotation(ForRelvars.class);
		if (annotation == null) {
			throw new DBObjectNotAnnotatedForASingleRelvarException();
		}
		String[] annotationRelvarNames = annotation.relvarNames();
		if (annotationRelvarNames.length == 1) {
			return new DmlAddCommand(annotationRelvarNames[0], this);
		} else {
			DmlAddCommand[] addCommandsForAll = new DmlAddCommand[annotationRelvarNames.length];
			for (int i = 0; i < annotationRelvarNames.length; i++) {
				addCommandsForAll[i] = new DmlAddCommand(annotationRelvarNames[i], this);
			}
			return new DmlMultipleAssignmentCommand(addCommandsForAll);
		}
	}

	/**
	 * Gets the Assert command for asserting (in the database the transaction is connected to) an entry in the single relvar named in this object's class' @forrelvar annotation, for the given object.
	 * 
	 * @return the Assert command for asserting (in the database the transaction is connected to) an entry in the single relvar named in this object's class' @forrelvar annotation, for the given object.
	 */
	public final DmlAssertCommand dbAssertCommand () {
		final ForRelvars annotation = this.getClass().getAnnotation(ForRelvars.class);
		if (annotation == null) {
			throw new DBObjectNotAnnotatedForASingleRelvarException();
		}
		String[] annotationRelvarNames = annotation.relvarNames();
		if (annotationRelvarNames.length != 1) {
			throw new DBObjectNotAnnotatedForASingleRelvarException(annotationRelvarNames);
		}
		return new DmlAssertCommand(annotationRelvarNames[0], this);
	}

	/**
	 * Gets the Assert command for asserting an entry in the named relvar (in the database the transaction is connected to), for the given object. The tuple that is registered in the database is constructed by invoking all the object's methods that satisfy the following properties :
	 * <ul>
	 * <li>The name of the method is the word "get" (all lowercase), followed by the capitalized relvar name (i.e. first character of the relvar name uppercase, all remaining characters of the relvar name lowercase), followed by a capitalized attribute name (i.e. the first character of the remaining part of the method name must be uppercase and all the other remaining characters must be lowercase)</LI>
	 * <li>The method is a public method that takes no arguments</LI>
	 * <li>The return type of the method is java.lang.String</LI>
	 * </UL>
	 * 
	 * @param relvarName
	 *            The name of the relvar into which a tuple is to be added for the given object
	 * @return The assert command to be issued for adding an entry in the named relvar with attribute values drawn from the given object
	 */
	public final DmlAssertCommand dbAssertCommand (String relvarName) {
		return new DmlAssertCommand(relvarName, this);
	}

	/**
	 * Gets the Assert command for asserting (in the database the transaction is connected to) an entry in the all relvars named in this object's class' @forrelvar annotation, for the given object.
	 * 
	 * @return the Assert command for asserting (in the database the transaction is connected to) an entry in the all relvars named in this object's class' @forrelvar annotation, for the given object.
	 */
	public final DmlAssignmentCommand dbAssertCommandAll () {
		final ForRelvars annotation = this.getClass().getAnnotation(ForRelvars.class);
		if (annotation == null) {
			throw new DBObjectNotAnnotatedForASingleRelvarException();
		}
		String[] annotationRelvarNames = annotation.relvarNames();
		if (annotationRelvarNames.length == 1) {
			return new DmlAssertCommand(annotationRelvarNames[0], this);
		} else {
			DmlAssertCommand[] assertCommandsForAll = new DmlAssertCommand[annotationRelvarNames.length];
			for (int i = 0; i < annotationRelvarNames.length; i++) {
				assertCommandsForAll[i] = new DmlAssertCommand(annotationRelvarNames[i], this);
			}
			return new DmlMultipleAssignmentCommand(assertCommandsForAll);
		}
	}

	/**
	 * Gets the command for deleting (in the database the transaction is connected to) the tuple in the relvar named in the @forrelvar annotation of this object's class, for the given object.
	 * 
	 * @return the command for deleting (in the database the transaction is connected to) the tuple in the relvar named in the @forrelvar annotation of this object's class, for the given object.
	 */
	public final DmlDeleteCommand dbDeleteCommand ( ) {
		final ForRelvars annotation = this.getClass().getAnnotation(ForRelvars.class);
		if (annotation == null) {
			throw new DBObjectNotAnnotatedForASingleRelvarException();
		}
		String[] annotationRelvarNames = annotation.relvarNames();
		if (annotationRelvarNames.length != 1) {
			throw new DBObjectNotAnnotatedForASingleRelvarException(annotationRelvarNames);
		}
		return new DmlDeleteCommand(annotationRelvarNames[0], this);
	}

	/**
	 * Gets the command for deleting the tuple in the named relvar (in the database the transaction is connected to), for the given object. The tuple that is registered in the database is constructed by invoking all the object's methods that satisfy the following properties :
	 * <ul>
	 * <li>The name of the method is the word "get" (all lowercase), followed by the capitalized relvar name (i.e. first character of the relvar name uppercase, all remaining characters of the relvar name lowercase), followed by a capitalized attribute name (i.e. the first character of the remaining part of the method name must be uppercase and all the other remaining characters must be lowercase)</LI>
	 * <li>The method is a public method that takes no arguments</LI>
	 * <li>The return type of the method is java.lang.String</LI>
	 * </UL>
	 * 
	 * @param relvarName
	 *            The name of the relvar into which a tuple is to be added for the given object
	 * @return the command for deleting the tuple in the named relvar (in the database the transaction is connected to), for the given object.
	 */
	public final DmlDeleteCommand dbDeleteCommand (String relvarName) {
		return new DmlDeleteCommand(relvarName, this);
	}

	/**
	 * Gets the command for deleting the tuple in the named relvar (in the database the transaction is connected to), for the given object.
	 * 
	 * @return the command for deleting the tuple in the named relvar (in the database the transaction is connected to), for the given object.
	 */
	public final DmlAssignmentCommand dbDeleteCommandAll ( ) {
		final ForRelvars annotation = this.getClass().getAnnotation(ForRelvars.class);
		if (annotation == null) {
			throw new DBObjectNotAnnotatedForASingleRelvarException();
		}
		String[] annotationRelvarNames = annotation.relvarNames();
		if (annotationRelvarNames.length == 1) {
			return new DmlDeleteCommand(annotationRelvarNames[0], this);
		} else {
			DmlDeleteCommand[] deleteCommandsForAll = new DmlDeleteCommand[annotationRelvarNames.length];
			for (int i = 0; i < annotationRelvarNames.length; i++) {
				deleteCommandsForAll[i] = new DmlDeleteCommand(annotationRelvarNames[i], this);
			}
			return new DmlMultipleAssignmentCommand(deleteCommandsForAll);
		}
	}

	/**
	 * Gets the command for unasserting the tuple in the relvar named in the @forrelvar annotation of this class.
	 * 
	 * @return the command for unasserting the tuple in the named relvar (in the database the transaction is connected to), for the given object.
	 */
	public final DmlUnassertCommand dbUnAssertCommand ( ) {
		final ForRelvars annotation = this.getClass().getAnnotation(ForRelvars.class);
		if (annotation == null) {
			throw new DBObjectNotAnnotatedForASingleRelvarException();
		}
		String[] annotationRelvarNames = annotation.relvarNames();
		if (annotationRelvarNames.length != 1) {
			throw new DBObjectNotAnnotatedForASingleRelvarException(annotationRelvarNames);
		}
		return new DmlUnassertCommand(annotationRelvarNames[0], this);
	}

	/**
	 * Gets the command for unasserting the tuple in the named relvar (in the database the transaction is connected to), for the given object. The tuple that is registered in the database is constructed by invoking all the object's methods that satisfy the following properties :
	 * <ul>
	 * <li>The name of the method is the word "get" (all lowercase), followed by the capitalized relvar name (i.e. first character of the relvar name uppercase, all remaining characters of the relvar name lowercase), followed by a capitalized attribute name (i.e. the first character of the remaining part of the method name must be uppercase and all the other remaining characters must be lowercase)</LI>
	 * <li>The method is a public method that takes no arguments</LI>
	 * <li>The return type of the method is java.lang.String</LI>
	 * </UL>
	 * 
	 * @param relvarName
	 *            The name of the relvar into which a tuple is to be added for the given object
	 * @return the command for unasserting the tuple in the named relvar (in the database the transaction is connected to), for the given object.
	 */
	public final DmlUnassertCommand dbUnAssertCommand (String relvarName) {
		return new DmlUnassertCommand(relvarName, this);
	}

	/**
	 * Gets the command for unasserting the tuple in all the relvars named in the @forrelvar annotation of this class.
	 * 
	 * @return the command for unasserting the tuple in all the named relvar (in the database the transaction is connected to), for the given object.
	 */
	public final DmlAssignmentCommand dbUnAssertCommandAll ( ) {
		final ForRelvars annotation = this.getClass().getAnnotation(ForRelvars.class);
		if (annotation == null) {
			throw new DBObjectNotAnnotatedForASingleRelvarException();
		}
		String[] annotationRelvarNames = annotation.relvarNames();
		if (annotationRelvarNames.length == 1) {
			return new DmlUnassertCommand(annotationRelvarNames[0], this);
		} else {
			DmlUnassertCommand[] unassertCommandsForAll = new DmlUnassertCommand[annotationRelvarNames.length];
			for (int i = 0; i < annotationRelvarNames.length; i++) {
				unassertCommandsForAll[i] = new DmlUnassertCommand(annotationRelvarNames[i], this);
			}
			return new DmlMultipleAssignmentCommand(unassertCommandsForAll);
		}
	}
}