/**
 * 
 */
package be.SIRAPRISE.client.jsba;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import be.SIRAPRISE.client.AbstractTupleContainer;
import be.SIRAPRISE.client.ConstructorMissingException;
import be.SIRAPRISE.client.Heading;
import be.SIRAPRISE.client.NonScalarTypeDeclaration;
import be.SIRAPRISE.client.Relation;
import be.SIRAPRISE.client.SettersMissingException;
import be.SIRAPRISE.client.Tuple;
import be.SIRAPRISE.client.TypeDeclaration;

/**
 * @author Erwin
 * 
 */
public abstract class DBObjectFactory {

	/**
	 * Checks whether the class represented by objectClass implements the interface defined by Class interfaceClass
	 * 
	 * @param objectClass
	 *            The Class object representing the clase of which it is to be tested whether that class implements the interface represented by interfaceClass.
	 * @param interfaceClass
	 *            The Class object representing the interface of which it is to be tested whether the class represented by objectClass implements it.
	 * @return true if the objectClass implements the interface defined by Class interfaceClass, false otherwise
	 */
	private static boolean classImplements (Class<?> objectClass, Class<?> interfaceClass) {
		Class<?>[] interfaces = objectClass.getInterfaces();
		int interfacesCount = interfaces.length;
		int i = 0;
		while (i < interfacesCount) {
			if (interfaces[i] == interfaceClass || classImplements(interfaces[i], interfaceClass)) {
				return true;
			}
			i++;
		}

		return false;
	}

	/**
	 * Get an objects holding the same information as that which is held in this tuple.
	 * 
	 * @param tuple
	 *            The tuple from which an object is to be built
	 * @param objectClass
	 *            A Class object denoting the class of the objects to be returned. objectClass must denote a public, non-abstract class that has a public no-arg constructor, and that has a unique setter method for each attribute in the heading. The name of this unique method must be the concatenation of the word "set" (all lowercase), the attribute name (capitalized), and the suffix "FromDB". The argument list of this setter method must consist of excatly one java.lang.String argument.
	 * @return An array of objects of the given object Class.
	 * @throws SettersMissingException
	 *             if a needed setter is missing in the class denoted by objectClass
	 * @throws ConstructorMissingException
	 *             if objectClass does not denote a public class, or it denotes an abstract class, or the class does not have a public no-arg constructor
	 * @throws ClassDoesNotImplementDBObjectException
	 *             if objectClass does not implement the required DBObject interface
	 */
	public static <C> C getObject (Tuple tuple, Class<C> objectClass) throws ConstructorMissingException, ClassDoesNotImplementDBObjectException, SettersMissingException {
		getPublicNoArgConstructor(objectClass);

		final Heading heading = tuple.getHeading();
		Set<String> attributeNames = heading.keySet();
		HashMap<String, SetterMethod> setterMethods = getSetterMethods(objectClass, heading);

		if (setterMethods.size() != attributeNames.size()) {
			ArrayList<String> missingSetters = new ArrayList<String>();
			missingSetters.addAll(attributeNames);
			missingSetters.removeAll(setterMethods.keySet());
			throw new SettersMissingException(missingSetters.toString() + '/' + objectClass.getName());
		}

		return getObject(tuple, objectClass, setterMethods);
	}

	/**
	 * Get an array of objects holding the same information as that which is held in this TupleContainer. One object is created for each tuple.
	 * 
	 * @param c
	 *            The tuple container whose tuples are to be 'transformed' into objects
	 * @param <C>
	 *            The name of the objectClass, also naming the class/object type of the objects making up the returned array
	 * @param objectClass
	 *            A Class object denoting the class of the objects to be returned. objectClass must denote a public, non-abstract class that has a public no-arg constructor, and that has a unique setter method for each attribute in the heading. The name of this unique method must be the concatenation of the word "set" (all lowercase), followed by the uppercased attribute name (i.e. first character uppercase, all others lowercase), followed by "FromDB". The argument list of this setter method must consist of excatly one java.lang.String argument.
	 * @return An array of objects of the given object Class.
	 * @throws ConstructorMissingException
	 *             if objectClass does not denote a public class, or it denotes an abstract class, or the class does not have a public no-arg constructor
	 * @throws SettersMissingException
	 *             if a needed setter is missing in the class denoted by objectClass
	 * @throws ClassDoesNotImplementDBObjectException
	 *             if objectClass does not implement the required DBObject interface
	 */
	@SuppressWarnings("unchecked")
	public static <C> C[] getObjectArray (AbstractTupleContainer c, Class<C> objectClass) throws SettersMissingException, ConstructorMissingException, ClassDoesNotImplementDBObjectException {
		return (C[]) (c.toObjectCollection(objectClass)).toArray();
	}

	/**
	 * Get an array of objects holding a (potentially proper) subset of the information held in this TupleContainer. One object is created for each tuple. Setters matching an attribute in the heading will be invoked, attributes for which no corresponding setter is found, are ignored.
	 * 
	 * @param <C>
	 *            The name of the objectClass, also naming the class/object type of the objects making up the returned array
	 * @param c
	 *            The tuple container whose tuples are to be 'transformed' into objects of the given class
	 * @param objectClass
	 *            A Class object denoting the class of the objects to be returned. objectClass must denote a public, non-abstract class that has a public no-arg constructor, and that has at most one unique setter method for each attribute in the heading. The name of this unique method must be the concatenation of the word "set" (all lowercase), followed by the uppercased attribute name (i.e. first character uppercase, all others lowercase), followed by "FromDB". The argument list of this setter method must consist of excatly one java.lang.String argument.
	 * @return An array of objects of the given object Class.
	 * @throws ConstructorMissingException
	 *             if objectClass does not denote a public class, or it denotes an abstract class, or the class does not have a public no-arg constructor
	 * @throws ClassDoesNotImplementDBObjectException
	 *             if objectClass does not implement the required DBObject interface
	 * @throws ConstructorMissingException
	 * @throws ClassDoesNotImplementDBObjectException
	 */
	@SuppressWarnings("unchecked")
	public static <C> C[] getObjectArrayIgnoringMissingSetters (AbstractTupleContainer c, Class<C> objectClass) throws ConstructorMissingException, ClassDoesNotImplementDBObjectException {
		return (C[]) (c.toObjectCollectionIgnoringMissingSetters(objectClass)).toArray();
	}

	/**
	 * Get an objects holding the same information as that which is held in this tuple.
	 * 
	 * @param tuple
	 *            The tuple from which an Object is to be built
	 * @param objectClass
	 *            A Class object denoting the class of the objects to be returned. objectClass must denote a public, non-abstract class that has a public no-arg constructor, and that has a unique setter method for each attribute in the heading. The name of this unique method must be the concatenation of the word "set" (all lowercase), the attribute name (capitalized), and the suffix "FromDB". The argument list of this setter method must consist of excatly one java.lang.String argument.
	 * 
	 * @param <C>
	 * 
	 * @return An array of objects of the given object Class.
	 * @throws ConstructorMissingException
	 *             if objectClass does not denote a public class, or it denotes an abstract class, or the class does not have a public no-arg constructor
	 * @throws ClassDoesNotImplementDBObjectException
	 *             if objectClass does not implement the required DBObject interface
	 */
	public static final <C> C getObjectIgnoringMissingSetters (Tuple tuple, Class<C> objectClass) throws ConstructorMissingException, ClassDoesNotImplementDBObjectException {
		getPublicNoArgConstructor(objectClass);
		return getObject(tuple, objectClass, DBObjectFactory.getSetterMethods(objectClass, tuple.getHeading()));
	}

	/**
	 * Get a Map of objects holding the same information as that which is held in this Tuple. The entries in the map have one of the objectClass[] objects as their key, and an object of that class as their value. The object of that class holds the attribute values of the tuple for which a corresponding setter method was found in the class.
	 * 
	 * @param tuple
	 *            The tuple from which objects are to be contructed
	 * @param objectClass
	 *            An array of Class objects denoting the classes of the objects to be returned for each tuple. Each objectClass must denote a public, non-abstract class that has a public no-arg constructor, and that has a unique setter method for some attribute in the heading. The name of this unique method must be the concatenation of the word "set" (all lowercase), the attribute name (capitalized), and the suffix "FromDB". The argument list of this setter method must consist of excatly one java.lang.String argument.
	 * 
	 * @return A map of objects holding the same information as this tuple. The key in each map entry is one of the objectClass objects given as an argument, the corresponding value in the map entry is the created object of that class.
	 * @throws SettersMissingException
	 *             if some attribute in the heading exists for which no corresponding setter could be found in any of the classes denoted by objectClass[]
	 * @throws ConstructorMissingException
	 *             if objectClass[] contains any class that does not denote a public class, or denotes an abstract class, or does not have a public no-arg constructor
	 * @throws ClassDoesNotImplementDBObjectException
	 *             if objectClass[] contains any class that does not implement the required DBObject interface
	 */
	public static Map<Class<?>, Object> getObjects (Tuple tuple, Class<?>[] objectClass) throws ConstructorMissingException, ClassDoesNotImplementDBObjectException, SettersMissingException {
		final Heading heading = tuple.getHeading();
		Set<String> attributeNames = heading.keySet();
		HashSet<String> unmatchedAttributeNames = new HashSet<String>();
		unmatchedAttributeNames.addAll(attributeNames);
		Map<Class<?>, HashMap<String, SetterMethod>> processingMap = new HashMap<Class<?>, HashMap<String, SetterMethod>>();
		// Each objectClass must have a public no-arg constructor
		for (int i = 0; i < objectClass.length; i++) {
			getPublicNoArgConstructor(objectClass[i]);
			HashMap<String, SetterMethod> setterMethods = getSetterMethods(objectClass[i], heading);
			processingMap.put(objectClass[i], setterMethods);
			unmatchedAttributeNames.removeAll(processingMap.keySet());
		}

		if (unmatchedAttributeNames.size() > 0) {
			throw new SettersMissingException(unmatchedAttributeNames.toString());
		}

		return toObjects(tuple, processingMap);
	}

	/**
	 * Gets the public no-arg constructor from a class
	 * 
	 * @param objectClass
	 *            the class
	 * @return the public no-arg constructor of the class
	 * @throws ConstructorMissingException
	 *             if objectClass does not denote a public class, or it denotes an abstract class, or the class does not have a public no-arg constructor
	 * @throws ClassDoesNotImplementDBObjectException
	 *             if objectClass does not implement the required DBObject interface
	 */
	public static Constructor<?> getPublicNoArgConstructor (Class<?> objectClass) throws ConstructorMissingException, ClassDoesNotImplementDBObjectException {
		try {
			Constructor<?> noargConstructor = objectClass.getConstructor(new Class[] {});
			final int objectClassModifiers = objectClass.getModifiers();
			if (Modifier.isAbstract(objectClassModifiers) || !(Modifier.isPublic(noargConstructor.getModifiers()) && Modifier.isPublic(objectClass.getModifiers()))) {
				throw new ConstructorMissingException();
			}

			if (DBObjectFactory.classImplements(objectClass, DBObject.class)) {
				return noargConstructor;
			}

			throw new ClassDoesNotImplementDBObjectException(objectClass);
		} catch (NoSuchMethodException e) {
			throw new ConstructorMissingException(e);
		}
	}

	/**
	 * Gets the 'Tuple' constructor from a class
	 * 
	 * @param objectClass
	 *            the class
	 * @return the public constructor of the class that takes a single argument of a type/class that implements the Tuple interface
	 * @throws ConstructorMissingException
	 *             if objectClass does not denote a public class, or it denotes an abstract class, or the class does not have a public single-arg Tuple constructor
	 * @throws ClassDoesNotImplementDBObjectException
	 *             if objectClass does not implement the required DBObject interface
	 */
	public static Constructor<?> getPublicTupleConstructor (Class<?> objectClass) throws ConstructorMissingException, ClassDoesNotImplementDBObjectException {
		final int objectClassModifiers = objectClass.getModifiers();
		if (Modifier.isAbstract(objectClassModifiers) || !Modifier.isPublic(objectClass.getModifiers())) {
			throw new ConstructorMissingException();
		}

		if (!classImplements(objectClass, DBObject.class)) {
			throw new ClassDoesNotImplementDBObjectException(objectClass);
		}

		Constructor<?>[] tupleConstructors = objectClass.getConstructors();
		for (int i = 0; i < tupleConstructors.length; i++) {
			Constructor<?> tupleConstructor = tupleConstructors[i];
			final Class<?>[] tupleConstructorParameterTypes = tupleConstructor.getParameterTypes();
			if (!tupleConstructor.isVarArgs() && tupleConstructorParameterTypes.length == 1) {
				if (classImplements(tupleConstructorParameterTypes[0], Tuple.class)) {
					return tupleConstructor;
				}
			}
		}

		throw new ConstructorMissingException();
	}

	/**
	 * Gets the setter methods from a class applicable for the given set of attribute names
	 * 
	 * @param objectClass
	 *            The class to extract the setter methods from
	 * @param heading
	 *            The heading of the tuple container, in which to inspect whether an attribute is relation-valued, so that a "relation" setter may also match
	 * @return A Map in which the key is the attribute name of some attribute in the heading of a TupleContainer, and the value is the setter method for that attribute obtained from the objectClass.
	 */
	public static HashMap<String, SetterMethod> getSetterMethods (Class<?> objectClass, Heading heading) {
		HashMap<String, SetterMethod> setterMethods = new HashMap<String, SetterMethod>();

		for (Entry<String, TypeDeclaration> me : heading.entrySet()) {
			String attributeName = me.getKey();
			String setterMethodName = "set" + Character.toUpperCase(attributeName.charAt(0)) + attributeName.substring(1).toLowerCase() + "FromDB"; //$NON-NLS-1$//$NON-NLS-2$
			TypeDeclaration typeDeclaration = me.getValue();
			boolean searchStringSetter = true;
			if (typeDeclaration instanceof NonScalarTypeDeclaration) {
				// Search for a relation setter first
				try {
					Method relationSetterMethod = objectClass.getMethod(setterMethodName, Relation.class);
					setterMethods.put(attributeName, new RelationSetterMethod(relationSetterMethod));
					searchStringSetter = false;
				} catch (NoSuchMethodException e) {

				}
			}
			if (searchStringSetter) {
				try {
					Method stringSetterMethod = objectClass.getMethod(setterMethodName, String.class);
					setterMethods.put(attributeName, new StringSetterMethod(stringSetterMethod));
				} catch (NoSuchMethodException e) {
				}
			}
		}

		// Class<?> inspectedClass = objectClass;
		// boolean setterMethodFound = false;
		// // objectClass must have a public set...() method for each attribute in the heading. That method must take one single String argument. ... must be the same name as the attribute in the heading it corresponds to, disregarding case (so setName() is the method corresponding to the attribute named NAME).
		// do {
		// Method[] methods = inspectedClass.getDeclaredMethods();
		// for (int i = 0; i < methods.length; i++) {
		// Method method = methods[i];
		// String methodName = method.getName();
		// Class<?>[] methodParameterTypes = method.getParameterTypes();
		//				if (Modifier.isPublic(method.getModifiers()) && methodName.startsWith("set") && methodName.endsWith("FromDB") && methodParameterTypes.length == 1) { //$NON-NLS-1$ //$NON-NLS-2$
		// if (methodParameterTypes[0] == String.class) {
		// String setterName = methodName.substring(3);
		// setterName = setterName.substring(0, setterName.length() - 6);
		// String setterName1 = setterName.substring(1);
		// if (Character.isUpperCase(setterName.charAt(0)) && setterName1.equals(setterName1.toLowerCase())) {
		// // iterate over the attribute names specified in the heading. Don't assume the names contained there are all uppercase.
		// Iterator<String> i_attributeNames = attributeNames.iterator();
		// while (i_attributeNames.hasNext()) {
		// String attributeName = i_attributeNames.next();
		// if (setterName.equalsIgnoreCase(attributeName)) {
		// // if (setterMethods.containsKey(attributeName)) {
		// // Method methodAlreadyFound = setterMethods.get(attributeName);
		// // if (methodAlreadyFound.getName().equals(methodName)) {
		// // // Then it is a super implementation of a method of the same name. That isn't a problem.
		// // } else {
		// // throw new DuplicateSetterException(attributeName);
		// // }
		// // }
		// setterMethods.put(attributeName, new JSBAStringSetterMethod(method));
		// setterMethodFound = true;
		// }
		// }
		// }
		// }
		// }
		// }
		//
		// inspectedClass = inspectedClass.getSuperclass();
		// } while (inspectedClass != null && !setterMethodFound);

		return setterMethods;
	}

	/**
	 * Gets an object of the given class, using the given set of setter methods and the attribute values from this tuple
	 * 
	 * @param tuple
	 *            The tuple from which an object is to be built
	 * @param objectClass
	 *            The Class of the object to be created and returned
	 * @param setterMethods
	 *            The setter methods to be invoked to set attribute values in the returned object
	 * 
	 * @param <C>
	 * 
	 * @return An object of the given Class holding the attribute values obtained from the given tuple
	 */
	public static final <C> C getObject (Tuple tuple, final Class<C> objectClass, HashMap<String, SetterMethod> setterMethods) {
		try {
			C object = objectClass.newInstance();
			for (Entry<String, SetterMethod> me : setterMethods.entrySet()) {
				String attributeName = me.getKey();
				SetterMethod jsbaSetterMethod = me.getValue();
				Method setterMethod = jsbaSetterMethod.getMethod();
				try {
					if (jsbaSetterMethod instanceof RelationSetterMethod) {
						setterMethod.invoke(object, new Object[] { tuple.rvaValue(attributeName) });
					} else {
						setterMethod.invoke(object, new Object[] { tuple.value(attributeName) });
					}
				} catch (IllegalAccessException e2) {
					throw new RuntimeException(e2);
				} catch (InvocationTargetException e2) {
					throw new RuntimeException(e2);
				}
			}

			if (object instanceof UpdatableDBObject) {
				((UpdatableDBObject) object).setPreUpdateState();
			}

			return object;
		} catch (InstantiationException e1) {
			throw new RuntimeException(e1);
		} catch (IllegalAccessException e1) {
			throw new RuntimeException(e1);
		}
	}

	/**
	 * Gets a Map of objects holding the same information as that which is held in this tuple.
	 * 
	 * @param tuple
	 *            The tuple from which objects are to be constrcuted
	 * @param classesProcessingMap
	 *            The processing map that defines how the conversion of a tuple to a series of objects should proceed. Each entry in the map is a Class object, defining the class of an object to be instantiated for containing values of the tuple. The value mapped to this class object is a Map defining the setter methods to be invoked for setting the appropriate values in the instantiated object.
	 * 
	 * @return a Map of objects holding the same information as that which is held in this tuple. The entries in the map have Class objects as their key, and an object of that class as their value. The object of that class holds the attribute values of the tuple for which a corresponding setter method was found in the class.
	 */
	public static final Map<Class<?>, Object> toObjects (Tuple tuple, final Map<Class<?>, HashMap<String, SetterMethod>> classesProcessingMap) {
		Map<Class<?>, Object> map = new HashMap<Class<?>, Object>();
		for (Entry<Class<?>, HashMap<String, SetterMethod>> me : classesProcessingMap.entrySet()) {
			Class<?> objectClass = me.getKey();
			map.put(objectClass, getObject(tuple, objectClass, me.getValue()));
		}
		return map;
	}

	/**
	 * prevent instantiation
	 * 
	 */
	private DBObjectFactory ( ) {

	}

	/**
	 * Get a collection of objects holding the same information as that which is held in this TupleContainer. One object is created for each tuple.
	 * 
	 * @param <C>
	 * @param tupleContainer
	 *            The TupleContainer holding the tuples to be used for constructing DBObjects
	 * @param objectClass
	 *            A Class object denoting the class of the objects to be returned. objectClass must denote a public, non-abstract class that has a public no-arg constructor, and that has a unique setter method for each attribute in the heading. The name of this unique method must be the concatenation of the word "set" (all lowercase), followed by the uppercased attribute name (i.e. first character uppercase, all others lowercase), followed by "FromDB". The argument list of this setter method must consist of excatly one java.lang.String argument.
	 * @return An array of objects of the given object Class.
	 * @throws ConstructorMissingException
	 *             if objectClass does not denote a public class, or it denotes an abstract class, or the class does not have a public no-arg constructor
	 * @throws SettersMissingException
	 *             if a needed setter is missing in the class denoted by objectClass
	 * @throws ClassDoesNotImplementDBObjectException
	 *             if objectClass does not implement the required DBObject interface
	 */
	public static <C> Collection<C> getObjectCollection (AbstractTupleContainer tupleContainer, Class<C> objectClass) throws ConstructorMissingException, ClassDoesNotImplementDBObjectException, SettersMissingException {
		final Heading heading = tupleContainer.getHeading();
	
		getPublicNoArgConstructor(objectClass);
	
		HashMap<String, SetterMethod> setterMethods = getSetterMethods(objectClass, heading);
	
		Set<String> attributeNames = heading.keySet();
		if (setterMethods.size() != attributeNames.size()) {
			ArrayList<String> missingSetters = new ArrayList<String>();
			missingSetters.addAll(attributeNames);
			missingSetters.removeAll(setterMethods.keySet());
			throw new SettersMissingException(missingSetters.toString() + '/' + objectClass.getName());
		}
	
		Collection<C> objects = new ArrayList<C>();
		for (Tuple tuple : tupleContainer) {
			objects.add(getObject(tuple, objectClass, setterMethods));
		}
	
		return objects;
	}
}
