/*
 * Created on 12-sep-2007
 */
package be.SIRAPRISE.client.jsba;

import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.security.InvalidKeyException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.Signature;
import java.security.SignatureException;
import java.security.UnrecoverableKeyException;
import java.security.cert.CertificateException;
import java.util.Set;

import be.SIRAPRISE.client.NAMES.ATTRIBUTENAMES;
import be.SIRAPRISE.client.NAMES.COMMANDNAMES;
import be.SIRAPRISE.client.AbstractRelation;
import be.SIRAPRISE.client.ConnectionClosedException;
import be.SIRAPRISE.client.DBConnection;
import be.SIRAPRISE.client.DBConnectionProperties;
import be.SIRAPRISE.client.DBException;
import be.SIRAPRISE.client.DBTransaction;
import be.SIRAPRISE.client.ErrorMessageException;
import be.SIRAPRISE.client.OneShotDBConnection;
import be.SIRAPRISE.client.Signer;
import be.SIRAPRISE.client.Tuple;
import be.SIRAPRISE.util.CommandLineParameters;
import be.SIRAPRISE.util.MyKeyStore;
import be.SIRAPRISE.util.UserPrivateKeyProvider;
import be.erwinsmout.MyMessageFormat;
import be.erwinsmout.NotFoundException;

/**
 * <P>
 * The DBObjectInterfaceGenerator utility helps the developer in writing classes for the java/SIRA_PRISE bridging architecture (abbreviated JSBA) by generating java source files that define an interface for JSBA classes to implement. Without this utility, a programmer must carefully choose the names of the methods that are required by JSBA for a class to cooperate with a SIRA_PRISE relvar. With this utility, the developer can generate the interface definition that corresponds to a given SIRA_PRISE relvar, include that interface definition in his java project and implement the interface methods in his class(es).
 * </P>
 * <P>
 * Please see the main() method documentation for details on running the utility
 * </P>
 * 
 * @author Erwin Smout
 */
public class DBObjectInterfaceGenerator implements Signer {

	/**
	 * 
	 */
	//	private static final String DBOBJECTINTERFACEGENERATOR_KEYSTORE = "dbobjectinterfacegenerator.keystore"; //$NON-NLS-1$

	/**
	 * The name for the default option when invoking the DBObjectInterfaceGenerator
	 */
	public static final String RELVAR = "RELVAR"; //$NON-NLS-1$

	/**
	 * <P>
	 * The DBObjectInterfaceGenerator utility helps the developer in writing classes for the java/SIRA_PRISE bridging architecture (abbreviated JSBA) by generating java source files that define an interface for JSBA classes to implement. Without this utility, a programmer must carefully choose the names of the methods that are required by JSBA for a class to cooperate with a SIRA_PRISE relvar. With this utility, the developer can generate the interface definition that corresponds to a given SIRA_PRISE relvar, include that interface definition in his java project and implement the interface methods in his class(es).
	 * </P>
	 * <P>
	 * This utility writes two files in the specified output directory (see the discussion of the args[] argument). One file is named "<code>DBObjectTo</code>", suffixed with the capitalized relvar name (i.e. the first character in uppercase, all the others in lowercase). The other file is named "<code>ToDBObject</code>", prefixed with the capitalized relvar name.
	 * </P>
	 * 
	 * @param args
	 *            The options for the utility can be passed via the command line using the following syntax for each option : <code><i>optionname(optionvalue)</i></code>. The following options can be passed :
	 *            <table BORDER width=100%>
	 *            <tr>
	 *            <th width="20%" valign="TOP">
	 *            <p >
	 *            Option name</th>
	 *            <th width="60%" valign="TOP">
	 *            <p >
	 *            Description</th>
	 *            <th width="20%" valign="TOP">
	 *            <p >
	 *            Default value</th>
	 *            </tr>
	 *            <tr>
	 *            <td>
	 *            <p>
	 *            <CODE>HOST</code></td>
	 *            <td>
	 *            <p >
	 *            The hostname or IP address of the SIRA_PRISE server to which to connect.</td>
	 *            <td>
	 *            <p>
	 *            LOCALHOST</td>
	 *            </tr>
	 *            <tr>
	 *            <td>
	 *            <p>
	 *            <CODE>PORT</code></td>
	 *            <td>
	 *            <p >
	 *            The IP port number on the server to which to connect</td>
	 *            <td>
	 *            <p>
	 *            50000</td>
	 *            </tr>
	 *            <tr>
	 *            <td>
	 *            <p>
	 *            <CODE>RELVAR</code></td>
	 *            <td>
	 *            <p >
	 *            The name of the relvar for which to generate an interface definition.</td>
	 *            <td>
	 *            <p>
	 *            no default.</td>
	 *            </tr>
	 *            <tr>
	 *            <td>
	 *            <p>
	 *            <CODE>OUTPUTDIRECTORY</code></td>
	 *            <td>
	 *            <p >
	 *            The directory where the script processor's output is to be written. This option can only be set on the command line, not in a properties file.</td>
	 *            <td>
	 *            <p >
	 *            No default</td>
	 *            </tr>
	 *            </table>
	 *            <P>
	 *            The RELVAR argument can be passed, alternatively, as the first argument and without having to be enclosed in RELVAR() parentheses, as in : <code>DBObjectInterfaceGenerator myrelvar host(...) port(...)</code>
	 *            </P>
	 *            <P>
	 *            Options that are not specified via the command line, are set from the DBObjectInterfaceGeneratorProperties object.
	 *            </P>
	 * @throws IOException
	 */
	public static void main (String args[]) throws IOException {
		new DBObjectInterfaceGenerator(args).run();
	}

	/**
	 * Comment for <code>canonicalizedRelvarName</code>
	 */
	private String canonicalizedRelvarName;

	/**
	 * The scriptprocessor's configured client name for obtaining database connections
	 */
	private final String clientName = "DBOBJECTINTERFACEGENERATOR"; //$NON-NLS-1$

	/**
	 * Comment for <code>cryptoAlgorithmNames</code>
	 */
	private Set<String> cryptoAlgorithmNames;

	/**
	 * Comment for <code>host</code>
	 */
	private String host;

	/**
	 * Comment for <code>interfaceName1</code>
	 */
	private String interfaceName1;

	/**
	 * Comment for <code>interfaceName2</code>
	 */
	private String interfaceName2;

	/**
	 * Comment for <code>lowercaseRelvarName</code>
	 */
	private String lowercaseRelvarName;

	/**
	 * Comment for <code>out</code>
	 */
	private PrintWriter out1;

	/**
	 * Comment for <code>out2</code>
	 */
	private PrintWriter out2;

	/**
	 * Comment for <code>packageNameDirectory</code>
	 */
	private String packageDirectoryName;

	/**
	 * Comment for <code>packageName</code>
	 */
	private String packageName;

	/**
	 * Comment for <code>port</code>
	 */
	private int port;

	/**
	 * The name of the relvar for which a DBObject interface is to be generated
	 */
	private String relvarName;

	/**
	 * Comment for <code>signingAlgorithmNames</code>
	 */
	private Set<String> signingAlgorithmNames;

	/**
	 * Comment for <code>user</code>
	 */
	private String user;

	/**
	 * Creates a DBObjectInterfaceGenerator
	 * 
	 * @param args
	 *            The command line args
	 * @throws IOException
	 */
	DBObjectInterfaceGenerator (String[] args) throws IOException {

		CommandLineParameters commandLineParameters = CommandLineParameters.getCommandLineParameters(args, RELVAR, DBObjectInterfaceGeneratorProperties.getInstance());
		relvarName = commandLineParameters.getProperty(RELVAR);
		if (relvarName == null) {
			BufferedReader conin = new BufferedReader(new InputStreamReader(System.in));
			relvarName = conin.readLine();
		}

		canonicalizedRelvarName = canonicalizeName(relvarName);
		lowercaseRelvarName = relvarName.toLowerCase();

		user = commandLineParameters.getProperty(DBConnectionProperties.USER, DBConnectionProperties.USERDEFAULT);
		host = commandLineParameters.getProperty(DBConnectionProperties.HOST, DBConnectionProperties.HOSTDEFAULT);
		port = Integer.parseInt(commandLineParameters.getProperty(DBConnectionProperties.PORT, DBConnectionProperties.PORTDEFAULT));

		String outputDirectoryName = commandLineParameters.getProperty("OUTPUTDIRECTORY", "."); //$NON-NLS-1$ //$NON-NLS-2$

		packageName = commandLineParameters.getProperty("PACKAGE", ""); //$NON-NLS-1$ //$NON-NLS-2$
		packageDirectoryName = outputDirectoryName + (outputDirectoryName.endsWith(File.separator) ? "" : File.separator) + packageName.replace('.', File.separatorChar) + File.separatorChar; //$NON-NLS-1$

		interfaceName1 = "DBObjectTo" + canonicalizedRelvarName; //$NON-NLS-1$
		out1 = new PrintWriter(new BufferedOutputStream(new FileOutputStream(((packageDirectoryName + interfaceName1) + ".java")))); //$NON-NLS-1$
		interfaceName2 = canonicalizedRelvarName + "ToDBObject"; //$NON-NLS-1$
		out2 = new PrintWriter(new BufferedOutputStream(new FileOutputStream(((packageDirectoryName + interfaceName2) + ".java")))); //$NON-NLS-1$

		String signingAlgorithmNameList = commandLineParameters.getProperty(DBConnectionProperties.SIGNINGALGORITHMS, ""); //$NON-NLS-1$
		signingAlgorithmNames = DBConnectionProperties.getSigningAlgorithmNameSet(signingAlgorithmNameList);

		String cryptoAlgorithmNameList = commandLineParameters.getProperty(DBConnectionProperties.CRYPTOALGORITHMS, ""); //$NON-NLS-1$
		cryptoAlgorithmNames = DBConnectionProperties.getCryptoAlgorithmNameSet(cryptoAlgorithmNameList);
	}

	/**
	 * Canonicalizes the given name
	 * 
	 * @param name
	 *            A name to canonicalize
	 * @return the canonicalized name, i.e. the same name with the first character in uppercase and the remaining characters in lowercase
	 */
	private String canonicalizeName (String name) {
		return Character.toUpperCase(name.charAt(0)) + name.substring(1).toLowerCase();
	}

	/**
	 * Gets The private key
	 * 
	 * @param algorithm
	 * @return The private key
	 * @throws NotFoundException
	 */
	private PrivateKey getPrivateKey (String algorithm) throws NotFoundException {
		try {
			// Get the keystore
			KeyStore jks = MyKeyStore.getKeyStore(this.getClass());

			// Get the private Key from the keystore. This is the key that is stored in the keystore for the alias that is equal to the clientID, suffixed with the algorithm name
			return (PrivateKey) jks.getKey(clientName + algorithm, this.getClass().getSimpleName().toCharArray());
		} catch (KeyStoreException e) {
			throw new NotFoundException(e.getClass().getName() + e.getMessage() != null ? e.getMessage() : ""); //$NON-NLS-1$
		} catch (NoSuchAlgorithmException e) {
			throw new NotFoundException(e.getClass().getName() + e.getMessage() != null ? e.getMessage() : ""); //$NON-NLS-1$
		} catch (CertificateException e) {
			throw new NotFoundException(e.getClass().getName() + e.getMessage() != null ? e.getMessage() : ""); //$NON-NLS-1$
		} catch (IOException e) {
			throw new NotFoundException(e.getClass().getName() + e.getMessage() != null ? e.getMessage() : ""); //$NON-NLS-1$
		} catch (UnrecoverableKeyException e) {
			throw new NotFoundException(e.getClass().getName() + e.getMessage() != null ? e.getMessage() : ""); //$NON-NLS-1$
		}
	}

	/**
	 * Gets A transaction object from the connection
	 * 
	 * @param dbc
	 *            The connection to start a transaction on
	 * @return The transaction object representing the started transaction
	 * @throws DBException
	 * @throws ConnectionClosedException
	 * @throws ErrorMessageException
	 */
	private DBTransaction startTransaction (DBConnection dbc) throws DBException, ConnectionClosedException, ErrorMessageException {
		if (user.length() > 0) {
			try {
				UserPrivateKeyProvider userPrivateKeyProvider = new UserPrivateKeyProvider(user);
				return dbc.startTransaction(user, userPrivateKeyProvider);
			} catch (NotFoundException e3) {
				out1.println(MyMessageFormat.format(Messages.getString("DBObjectInterfaceGenerator.UserNotAuthenticable"), new String[] { user })); //$NON-NLS-1$
				return dbc.startTransaction();
			}
		} else {
			return dbc.startTransaction();
		}
	}

	/**
	 * Runs the generation process
	 */
	void run ( ) {

		out1.println("/*\r\n * Created on 16-jan-2009\r\n */\r\npackage " + packageName + ";\r\n\r\nimport " + DBObject.class.getName() + ";\r\n\r\n/**\r\n * @author " + clientName + "\r\n */\r\npublic interface " + interfaceName1 + " extends DBObject {\r\n\r\n"); //$NON-NLS-1$//$NON-NLS-2$//$NON-NLS-3$//$NON-NLS-4$ //$NON-NLS-5$
		out2.println("/*\r\n * Created on 16-jan-2009\r\n */\r\npackage " + packageName + ";\r\n\r\nimport " + DBObject.class.getName() + ";\r\n\r\n/**\r\n * @author " + clientName + "\r\n */\r\npublic interface " + interfaceName2 + " extends DBObject {\r\n\r\n"); //$NON-NLS-1$//$NON-NLS-2$//$NON-NLS-3$//$NON-NLS-4$ //$NON-NLS-5$

		try {
			DBConnection dbc = new OneShotDBConnection(host, port, signingAlgorithmNames, clientName, cryptoAlgorithmNames, this);
			try {
				Tuple metaInfoTuple = startTransaction(dbc).execDmlCommandAndEndTransaction(COMMANDNAMES.INQUIRE + " EXPRESSIONINFO(STRING(" + relvarName + "))").iterator().next(); //$NON-NLS-1$ //$NON-NLS-2$
				AbstractRelation attributesRelation = metaInfoTuple.rvaValue("NONSCALARHEADING"); //$NON-NLS-1$  attributesRelationText = BODY(...)
				for (Tuple tuple : attributesRelation) {
					String attributeName = tuple.value(ATTRIBUTENAMES.ATTRIBUTENAME);
					String canonicalAttributeNameForMethods = canonicalizeName(attributeName);
					String lowercaseAttributeName = attributeName.toLowerCase();

					// write the DBObject-to-database part in interface 1
					out1.println("	/**\r\n	 * Gets the " + lowercaseAttributeName + " for " + lowercaseRelvarName + "\r\n	 * \r\n	 * @return The " + lowercaseAttributeName + " for " + lowercaseRelvarName + "\r\n	 */\r\n	public String get" + canonicalizedRelvarName + canonicalAttributeNameForMethods + "();\r\n\r\n"); //$NON-NLS-1$//$NON-NLS-2$//$NON-NLS-3$ //$NON-NLS-4$ //$NON-NLS-5$ //$NON-NLS-6$

					// write the database-to-DBObject part in interface 2
					out2.println("	/**\r\n	 * Sets " + lowercaseAttributeName + " from the DB\r\n	 * \r\n	 * @param " + lowercaseAttributeName + " The DB value for " + lowercaseAttributeName + "\r\n	 */\r\n	public void set" + canonicalAttributeNameForMethods + "FromDB(String " + lowercaseAttributeName + ");\r\n\r\n"); //$NON-NLS-1$//$NON-NLS-2$//$NON-NLS-3$ //$NON-NLS-4$ //$NON-NLS-5$ //$NON-NLS-6$
				}
			} finally {
				dbc.close();
			}

			out1.println("\r\n}\r\n"); //$NON-NLS-1$
			out2.println("\r\n}\r\n"); //$NON-NLS-1$
		} catch (Exception e) {
			String formattedMessage = MyMessageFormat.format(Messages.getString("DBObjectInterfaceGenerator.Exception"), new String[] { e.getClass().getName(), e.getMessage() }); //$NON-NLS-1$
			e.printStackTrace(out1);
			out1.println(formattedMessage);
			e.printStackTrace(System.err);
			System.err.println(formattedMessage);
		} finally {
			out1.close();
			out2.close();
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.SIRAPRISE.security.PrivateKeyProvider#sign(java.security.Signature, byte[])
	 */
	public byte[] sign (Signature signature, byte[] signMessage) throws InvalidKeyException, SignatureException, NotFoundException {
		signature.initSign(getPrivateKey(signature.getAlgorithm()));
		signature.update(signMessage);
		return signature.sign();
	}
}