/**
 * 
 */
package be.SIRAPRISE.client.jsba;

import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.HashMap;
import be.SIRAPRISE.util.MyReadOnlyMap;

/**
 * The JSBAGetterMethodCache caches the sets of getter methods that can be derived by the JSBA architecture from a given combination of relvar name and java class.
 * 
 * @author Erwin
 * 
 */
public final class GetterMethodCache {

	/**
	 * 
	 */
	private static GetterMethodCache instance = new GetterMethodCache();

	/**
	 * Gets the instance of the cache
	 * 
	 * @return the instance of the cache
	 */
	public static GetterMethodCache getInstance ( ) {
		return instance;
	}

	/**
	 * The cache
	 */
	private HashMap<GetterMethodKey, MyReadOnlyMap<String, JSBAMethodInvoker>> getterMethodsCache = new HashMap<GetterMethodKey, MyReadOnlyMap<String, JSBAMethodInvoker>>();

	/**
	 * Creates the JSBAGetterMethodCache
	 * 
	 */
	private GetterMethodCache ( ) {

	}

	/**
	 * Gets A map of getter methods that could be found in the given class for the given relvar name from the cache. If the given class/relvarname combination is not yet cached, determines the info and caches it prior to returning it.
	 * 
	 * @param clazz
	 *            A class implementing the JSBA DBObject interface that is to be inspected for the presence of JSBA-compliant getter methods for the given relvar name.
	 * @param jsbaCanonicalRelvarName
	 *            A relvar name in JSBA canonical format (i.e. first token uppercase, all others lowercase), for which JSBA-compliant getter methods are to be searched in the given class.
	 * @return A map of getter methods that could be found in the given class for the given relvar name.
	 */
	public MyReadOnlyMap<String, JSBAMethodInvoker> getGetterMethods (Class<?> clazz, String jsbaCanonicalRelvarName) {
		GetterMethodKey jsbaGetterMethodKey = new GetterMethodKey(clazz, jsbaCanonicalRelvarName);
		MyReadOnlyMap<String, JSBAMethodInvoker> getterMethods = getterMethodsCache.get(jsbaGetterMethodKey);
		if (getterMethods == null) {
			HashMap<String, JSBAMethodInvoker> workGetterMethods = new HashMap<String, JSBAMethodInvoker>();
			getterMethods = new MyReadOnlyMap<String, JSBAMethodInvoker>(workGetterMethods);
			getterMethodsCache.put(jsbaGetterMethodKey, getterMethods);


			//			final ForRelvars annotation = this.getClass().getAnnotation(ForRelvars.class);
			// if (annotation != null) {
			
			Class<?> inspectedClass = clazz;
			do {
				Method[] methods = inspectedClass.getDeclaredMethods();
				for (int i = 0; i < methods.length; i++) {
					Method method = methods[i];
					if (Modifier.isPublic(method.getModifiers()) && method.getParameterTypes().length == 0 && method.getReturnType() == String.class) {
						GetterMethod jsbaGetterMethodAnnotation = method.getAnnotation(GetterMethod.class);
						if (jsbaGetterMethodAnnotation != null) {
							int relvarNameIndexInAnnotation;
							if ((relvarNameIndexInAnnotation = getRelvarNameIndexInAnnotation(jsbaCanonicalRelvarName, jsbaGetterMethodAnnotation)) >= 0) {
								String attributeNameFromAnnotation = getAttributeNameFromAnnotation(jsbaGetterMethodAnnotation, relvarNameIndexInAnnotation);
								final String ucAttributeNameFromAnnotation = attributeNameFromAnnotation.toUpperCase();
								if (!workGetterMethods.containsKey(ucAttributeNameFromAnnotation)) { // can be the case if we're inspecting a superclass that has a matching method that was overridden by some subclass
									workGetterMethods.put(ucAttributeNameFromAnnotation, new AnnotatedMethodInvoker(method, jsbaGetterMethodAnnotation, relvarNameIndexInAnnotation));
								}
							}
						}
					}

				}

				inspectedClass = inspectedClass.getSuperclass();
			} while (inspectedClass != null);

			// } else {
			
			// old-style getter methods collection, based on naming conventions of the getter method : name is "get" + Capitalized relvar name + Capitalized attribute name ; method is public ; takes no parameters ; returns a String
			String getterMethodNamePrefix = "get" + jsbaCanonicalRelvarName; //$NON-NLS-1$
			int getterMethodNamePrefixLength = getterMethodNamePrefix.length();

			inspectedClass = clazz;
			// search for all public getRelvarnameAttributename() methods that return a java.lang.String.
			do {
				Method[] methods = inspectedClass.getDeclaredMethods();
				for (int i = 0; i < methods.length; i++) {
					Method method = methods[i];
					String methodName = method.getName();
					if (Modifier.isPublic(method.getModifiers()) && methodName.startsWith(getterMethodNamePrefix) && method.getParameterTypes().length == 0 && method.getReturnType() == String.class && methodName.length() > getterMethodNamePrefixLength) {
						String attributeNameUnCased = methodName.substring(getterMethodNamePrefixLength);
						String attributeNameUnCasedSubstring = attributeNameUnCased.substring(1);
						if (Character.isUpperCase(attributeNameUnCased.charAt(0)) && attributeNameUnCasedSubstring.equals(attributeNameUnCasedSubstring.toLowerCase())) {
							if (!workGetterMethods.containsKey(attributeNameUnCased.toUpperCase())) { // can be the case if we're inspecting a superclass that has a matching method that was overridden by some subclass
								workGetterMethods.put(attributeNameUnCased.toUpperCase(), new OldStyleDirectMethodInvoker(method));
							}
						}
					}
				}

				inspectedClass = inspectedClass.getSuperclass();
			} while (inspectedClass != null);
			
			// }

		}

		return getterMethods;
	}

	/**
	 * @param jsbaGetterMethodAnnotation
	 * @param relvarNameIndexInAnnotation
	 * @return
	 */
	private String getAttributeNameFromAnnotation (GetterMethod jsbaGetterMethodAnnotation, int relvarNameIndexInAnnotation) {
		String[] attributeNames = jsbaGetterMethodAnnotation.attributeNames();
		if (attributeNames.length == 1) {
			return attributeNames[0];
		}
		return attributeNames[relvarNameIndexInAnnotation];
	}

	/**
	 * @param jsbaCanonicalRelvarName
	 * @param jsbaGetterMethodAnnotation
	 * @return
	 */
	private int getRelvarNameIndexInAnnotation (String jsbaCanonicalRelvarName, GetterMethod jsbaGetterMethodAnnotation) {
		String[] relvarNames = jsbaGetterMethodAnnotation.relvarNames();
		for (int i = 0; i < relvarNames.length; i++) {
			if (relvarNames[i].equalsIgnoreCase(jsbaCanonicalRelvarName)) {
				return i;
			}
		}
		return -1;
	}
}
