/*
 * Created on 13-jan-2009
 */
package be.SIRAPRISE.client.jsba;

import be.SIRAPRISE.util.BracketParser;
import be.SIRAPRISE.util.InvalidEscapedCharacterException;
import be.SIRAPRISE.util.MissingEscapedCharacterException;

/**
 * Class provided for purposes of documenting the methods involved in the SIRA_PRISE/Java bridge.
 * 
 * @author Erwin Smout
 */
public class UpdatableDBObjectTemplate implements UpdatableDBObject {

	/**
	 * some attribute of this class
	 */
	private int attribute1;

	/**
	 * some other attribute of this class
	 */
	private String attribute2;

	/**
	 * The object that saves the "state" of the concerned tuple/object. Setter methods setAttribute1(), setAttribute2() may subsequently be invoked by clients. The JSBA's dmlUpdateCommand() method will then find all attribute values that have changed through an invocation of such a setter method, and build the appropriate update command for the database accordingly.
	 */
	private UpdatableDBObjectTemplate preUpdateState;

	/**
	 * The public no-arg constructor is required in any (class that implements) DBObject.
	 */
	public UpdatableDBObjectTemplate ( ) {

	}

	/**
	 * Other constructors may exist to serve the application programs
	 * 
	 * @param attribute1
	 *            some attribute of this class
	 * @param attribute2
	 *            some other attribute of this class
	 */
	public UpdatableDBObjectTemplate (int attribute1, String attribute2) {
		this.attribute1 = attribute1;
		this.attribute2 = attribute2;
	}

	/**
	 * Gets some attribute of this class
	 * 
	 * @return some attribute of this class
	 */
	public final int getAttribute1 ( ) {
		return attribute1;
	}

	/**
	 * Gets some other attribute of this class
	 * 
	 * @return some other attribute of this class
	 */
	public final String getAttribute2 ( ) {
		return attribute2;
	}

	/**
	 * Gets the value of some attribute of this class for use in the Java-to-SIRA_PRISE part of the bridge.
	 * 
	 * @return some attribute of this class. The return type of the getter methods that form the java-to-SIRA_PRISE part of the bridge must always be java.lang.String.
	 */
	public final String getRelvar1Attribute1 ( ) {
		return Integer.toString(attribute1);
	}

	/**
	 * Gets the value of some other attribute of this class for use in the Java-to-SIRA_PRISE part of the bridge. Note the call to BracketParser.meta(). This call is needed in the case that the value of the attribute can itself contain characters that are part of the SIRA_PRISE language, notably '(', ')' and '\' characters.
	 * 
	 * @return some other attribute of this class
	 */
	public final String getRelvar1Attribute2 ( ) {
		return BracketParser.meta(attribute2);
	}

	/**
	 * Gets the value of some attribute of this class for use in the Java-to-SIRA_PRISE part of the bridge.
	 * 
	 * @return some attribute of this class.
	 */
	public final String getRelvar2Attribute1 ( ) {
		return Integer.toString(attribute1);
	}

	/**
	 * Sets some attribute of this class
	 * 
	 * @param attribute1
	 *            some attribute of this class
	 */
	public final void setAttribute1 (int attribute1) {
		this.attribute1 = attribute1;
	}

	/**
	 * Sets some attribute of this class, given a SIRA_PRISE string representation of the value to set.
	 * 
	 * @param attribute1
	 *            The SIRA_PRISE string representation of a value for some attribute of this class
	 */
	public final void setAttribute1FromDB (String attribute1) {
		this.attribute1 = Integer.parseInt(attribute1);
	}

	/**
	 * Sets some other attribute of this class
	 * 
	 * @param attribute2
	 *            some other attribute of this class
	 */
	public final void setAttribute2 (String attribute2) {
		this.attribute2 = attribute2;
	}

	/**
	 * Sets some other attribute of this class, given a SIRA_PRISE string representation of the value to set.
	 * 
	 * @param attribute2
	 *            The SIRA_PRISE string representation of a value for some other attribute of this class. Note in particular the call to BracketParser.unMeta(). This is because the String values that are passed to these setter methods are always in 'escaped' format, i.e. if one of the characters '(', ')' or '\' is part of the actual value of the attribute, then any appearance of these characters is represented by the character sequences '\(', '\)' or '\\' respectively (note that in a java source, the latter is actually written as four consecutive backslashes, because java itself also 'escapes' backslash characters). This allows in
	 *            particular to correctly treat values that contain opening and closing parentheses to denote component values of the value. E.g. a date value could be passed to this setter method as the String "D(31)M(12)Y(1999)". Thus, string values that represent filenames or directory names can also be dealt with appropriately : the set...FromDB() method will be invoked with a String value of, e.g. "directory\\filename", and the "regular" setter method available to the application could be invoked with a string value of "directory\filename", as the application is likely to want to have it.
	 */
	public final void setAttribute2FromDB (String attribute2) {
		try {
			this.attribute2 = BracketParser.unMeta(attribute2);
		} catch (InvalidEscapedCharacterException e) {
			throw new RuntimeException(e);
		} catch (MissingEscapedCharacterException e) {
			throw new RuntimeException(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.SIRAPRISE.client.UpdatableDBObject#setPreUpdateState()
	 */
	public final void setPreUpdateState ( ) {
		preUpdateState = new UpdatableDBObjectTemplate(attribute1, attribute2);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.SIRAPRISE.client.UpdatableDBObject#getPreUpdateState()
	 */
	public final UpdatableDBObject getPreUpdateState ( ) {
		return preUpdateState;
	}
}