/*
 * Created on 24-jul-2008
 */
package be.SIRAPRISE.messages;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

import be.SIRAPRISE.client.Version;
import be.SIRAPRISE.util.MyDataInputStream;
import be.SIRAPRISE.util.MyDataOutputStream;

/**
 * ClientHelloMessageTypeV1_0 is the 1.0 version of the client hello message type. This message is structured as follows :
 * <table border="1" cellpadding="2" cellspacing="1" width="100%">
 * <tbody>
 * <tr>
 * <th width="24%">Zone</th>
 * <th width="12%">Format</th>
 * <th width="9%">Length</th>
 * <th width="55%">Description</th>
 * </tr>
 * <tr>
 * <td>SPCOUNT</td>
 * <td>INTEGER</td>
 * <td>2</td>
 * <td>Number of signing protocol specifications that follow. Can be zero.</td>
 * </tr>
 * <tr>
 * <td>SP</td>
 * <td>STRING</td>
 * <td>&nbsp;</td>
 * <td>The identification of a signing protocol that the client is willing and able to use. This field appears as many times as indicated in SPCOUNT.</td>
 * </tr>
 * <tr>
 * <td>EPCOUNT</td>
 * <td>INTEGER</td>
 * <td>2</td>
 * <td>Number of encryption protocol specificiations that follow. Can be zero.</td>
 * </tr>
 * <tr>
 * <td>EP</td>
 * <td>STRING</td>
 * <td>&nbsp;</td>
 * <td>The identification of an encryption protocol that the client is willing and able to use. This field appears as many times as indicated in EPCOUNT.</td>
 * </tr>
 * <tr>
 * <td>IDLETIME</td>
 * <td>INTEGER</td>
 * <td>8</td>
 * <td>The time in milliseconds that the server is requested to allow this connection to remain "idle". A connection is "idle" either if it has been opened, but no transaction has currently been started on it, or a transaction has been started, and that transaction is awaiting client input. "idle time forever" is not supported, but it can be approximated by requesting the value 0x7FFFFFFFFFFFFFFF.</td>
 * </tr>
 * <tr>
 * <td>MAJORCLVERSION</td>
 * <td>INTEGER</td>
 * <td>2</td>
 * <td>The major version number of the SIRA_PRISE specification that the client was designed to handle (currently only the value 1 is valid).</td>
 * </tr>
 * <tr>
 * <td>MINORCLVERSION</td>
 * <td>INTEGER</td>
 * <td>2</td>
 * <td>The minor version number of the SIRA_PRISE specification that the client was designed to handle.</td>
 * </tr>
 * </tbody>
 * </table>
 * <P>
 * If the SPCOUNT field is set to zero, this means that the client will not be signing his commands. Likewise, if the EPCOUNT field is set to zero, this means that the client will not be encrypting his commands.
 * </P>
 * 
 * @author Erwin Smout
 */
public final class ClientHelloMessageTypeV1_0 extends ClientHelloMessageType {

	/**
	 * the instance
	 */
	private static final ClientHelloMessageType instance = new ClientHelloMessageTypeV1_0();

	/**
	 * Gets the instance
	 * 
	 * @return the instance
	 */
	static ClientHelloMessageType getInstance ( ) {
		return instance;
	}

	/**
	 * 
	 */
	private ClientHelloMessageTypeV1_0 ( ) {
		super(Version.ONE_ZERO, Version.ONE_ZERO);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.SIRAPRISE.server.ServerMessageType#getServerMessage(java.io.DataInputStream)
	 */
	ServerMessage typeSpecificFromStream (DataInputStream in) throws IOException {

		int clientSignatureProtocolCount = in.readShort();
		HashSet<String> clientSignatureProtocols = new HashSet<String>();

		while (--clientSignatureProtocolCount >= 0) {
			short charLength = in.readShort();
			short byteLength = in.readShort();
			clientSignatureProtocols.add(MyDataInputStream.getUTFString(in, charLength, byteLength));
		}

		int clientEncryptionProtocolCount = in.readShort();
		HashSet<String> clientEncryptionProtocols = new HashSet<String>();
		while (--clientEncryptionProtocolCount >= 0) {
			short charLength = in.readShort();
			short byteLength = in.readShort();
			clientEncryptionProtocols.add(MyDataInputStream.getUTFString(in, charLength, byteLength));
		}

		long idleTime = in.readLong();
		short majorClientSiraPriseVersion = in.readShort();
		short minorClientSiraPriseVersion = in.readShort();

		return new ClientHelloMessageV1_0(this, clientSignatureProtocols, clientEncryptionProtocols, idleTime, new Version(majorClientSiraPriseVersion, minorClientSiraPriseVersion));
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.client.ServerMessageType#typeSpecificToStream(be.erwinsmout.SIRA_PRISE.client.ServerMessage, java.io.DataOutputStream)
	 */
	void typeSpecificToStream (ServerMessage message, DataOutputStream outputStream) throws IOException {
		if (!(message instanceof ClientHelloMessageV1_0)) {
			throw new IllegalArgumentException();
		}

		ClientHelloMessageV1_0 clientHelloMessageV1_0 = (ClientHelloMessageV1_0) message;

		Set<String> clientSignatureProtocols = clientHelloMessageV1_0.getClientSignatureProtocolNames();
		outputStream.writeShort(clientSignatureProtocols.size());
		Iterator<String> i_clientSignatureProtocols = clientSignatureProtocols.iterator();
		while (i_clientSignatureProtocols.hasNext()) {
			String clientSignatureProtocol = i_clientSignatureProtocols.next();
			MyDataOutputStream.writeSmallUTF(clientSignatureProtocol, outputStream);
		}

		Set<String> clientEncryptionProtocols = clientHelloMessageV1_0.getClientCryptoProtocolNames();
		outputStream.writeShort(clientEncryptionProtocols.size());
		Iterator<String> i_clientEncryptionProtocols = clientEncryptionProtocols.iterator();
		while (i_clientEncryptionProtocols.hasNext()) {
			String clientEncryptionProtocol = i_clientEncryptionProtocols.next();
			MyDataOutputStream.writeSmallUTF(clientEncryptionProtocol, outputStream);
		}

		outputStream.writeLong(clientHelloMessageV1_0.getIdleTime());

		outputStream.writeShort(clientHelloMessageV1_0.getClientSiraPriseVersion().getMajorVersion());
		outputStream.writeShort(clientHelloMessageV1_0.getClientSiraPriseVersion().getMinorVersion());
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.messages.ClientHelloMessageType#message(java.util.Set, java.util.Set, long, be.erwinsmout.SIRA_PRISE.client.Version)
	 */
	public ClientHelloMessage message (Set<String> signatureAlgorithmNames, Set<String> encryptionProtocolNames, long requestConnectionIdleTime, Version version) {
		return new ClientHelloMessageV1_0(this, signatureAlgorithmNames, encryptionProtocolNames, requestConnectionIdleTime, version);
	}
}
