/*
 * Created on 26-jul-2008
 */
package be.SIRAPRISE.messages;

/**
 * A ServerErrorMessage is any kind of message holding an indication that some kind of error has occurred on the server side. It can be subclassed to create "classes of server errors", each with their own specifics.
 * 
 * @author Erwin Smout
 */
public abstract class ServerErrorMessage extends ServerMessage {

	/**
	 * The client response text
	 */
	private String clientResponseText;

	/**
	 * The error number, whether a constraint number, a sira-prise runtime error, or a nil
	 */
	private long errorNumber;

	/**
	 * Flag indicating whether the error was 'fatal', i.e. whether the server has aborted its side of the transaction as a consequence of the error
	 */
	private boolean fatal;

	/**
	 * the server's original exception message
	 */
	private String originalExceptionMessage;

	/**
	 * Creates the ErrorMessage, setting its type, error number and client response text
	 * 
	 * @param type
	 *            the error message's type, indicating whether the error is a server runtime error, a sira_prise runtime error, a constraint violation, ...
	 * @param errorNumber
	 *            The error number, whether a constraint number, a sira-prise runtime error, or a nil
	 * @param clientResponseText
	 *            The response text to be shown to the client/user
	 * @param originalExceptionMessage
	 *            the server's original exception message
	 * @param fatal
	 *            Flag indicating whether the error was 'fatal', i.e. whether the server has aborted its side of the transaction as a consequence of the error
	 */
	ServerErrorMessage (ErrorMessageType type, long errorNumber, String clientResponseText, String originalExceptionMessage, boolean fatal) {
		super(type);
		this.errorNumber = errorNumber;
		this.clientResponseText = clientResponseText;
		this.originalExceptionMessage = originalExceptionMessage;
		this.fatal = fatal;
	}

	/**
	 * Gets The response text to be shown to the client/user
	 * 
	 * @return The response text to be shown to the client/user
	 */
	public final String getClientResponseText ( ) {
		return clientResponseText;
	}

	/**
	 * Gets a description of the error class
	 * 
	 * @return a description of the error class
	 */
	public abstract String getErrorClass ( );

	/**
	 * Gets The error number, whether a constraint number, a sira-prise runtime error, or a nil
	 * 
	 * @return The error number, whether a constraint number, a sira-prise runtime error, or a nil
	 */
	public final long getErrorNumber ( ) {
		return errorNumber;
	}

	/**
	 * Gets the server's original exception message
	 * 
	 * @return the server's original exception message
	 */
	public final String getOriginalExceptionMessage ( ) {
		return originalExceptionMessage;
	}

	/**
	 * Gets a flag indicating whether the error was 'fatal', i.e. whether the server has aborted its side of the transaction as a consequence of the error
	 * 
	 * @return Flag indicating whether the error was 'fatal', i.e. whether the server has aborted its side of the transaction as a consequence of the error
	 */
	public final boolean isFatal ( ) {
		return fatal;
	}
}