/*
 * Created on 24-jul-2008
 */
package be.SIRAPRISE.messages;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.security.NoSuchAlgorithmException;
import java.security.Signature;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.Set;

import javax.crypto.Cipher;
import javax.crypto.NoSuchPaddingException;

import be.SIRAPRISE.client.Version;
import be.SIRAPRISE.security.JCECipher;
import be.SIRAPRISE.security.ProprietaryOrJCECipher;
import be.SIRAPRISE.security.SPE;
import be.SIRAPRISE.util.MyDataInputStream;
import be.SIRAPRISE.util.MyDataOutputStream;

/**
 * ServerHelloMessageTypeV1_0 is the V1.0 version of the server hello message type.
 * <p>
 * The layout of this message is as follows :
 * </p>
 * <table border="1" cellpadding="2" cellspacing="1" width="100%">
 * <tbody>
 * <tr>
 * <th width="24%">Zone</th>
 * <th width="12%">Format</th>
 * <th width="9%">Length</th>
 * <th width="55%">Description</th>
 * </tr>
 * <tr>
 * <td>SPINDICATOR</td>
 * <td>INTEGER</td>
 * <td>1</td>
 * <td>0 if no signing protocol was retained (field SP will not be present in the response).<br>
 * 1 if a signing protocol is retained.</td>
 * </tr>
 * <tr>
 * <td>SP</td>
 * <td>STRING</td>
 * <td>&nbsp;</td>
 * <td>The name of the signing protocol that the server elected as the most appropriate to be used.</td>
 * </tr>
 * <tr>
 * <td>EPINDICATOR</td>
 * <td>INTEGER</td>
 * <td>1</td>
 * <td>0 if no encryption protocol was retained (fields EP and SERVEREPINFO will not be present in the response).<br>
 * 1 if an encryption protocol is retained (fields EP and SERVEREPINFO will be present in the response).</td>
 * </tr>
 * <tr>
 * <td>EP</td>
 * <td>STRING</td>
 * <td>&nbsp;</td>
 * <td>The name of the encryption protocol that the server elected as the one to be used on this connection.</td>
 * </tr>
 * <tr>
 * <td>SERVEREPINFO</td>
 * <td>BYTES</td>
 * <td>&nbsp;</td>
 * <td>This field contains info pertinent to the particular encryption protocol elected by the server. The format of this field is protocol-specific and explained in separate tables below.</td>
 * </tr>
 * <tr>
 * <td>IDLETIME</td>
 * <td>INTEGER</td>
 * <td>8</td>
 * <td>The idle time that the server has decided to allow for this connection. Clients must inspect this field and cannot simply assume that the server will retain the IDLETIME setting that has been requested by the client in the session protocol negotiation message.</td>
 * </tr>
 * <tr>
 * <td>MAJORVERSION</td>
 * <td>INTEGER</td>
 * <td>2</td>
 * <td>The major version of the SIRA_PRISE specification that the client is to use when selecting the message versions to send to the server and to interpret the responses from the server. This may be a "lesser" version of the SIRA_PRISE specification if the client package is "more recent" than the server that the client is connecting to.</td>
 * </tr>
 * <tr>
 * <td>MINORVERSION</td>
 * <td>INTEGER</td>
 * <td>2</td>
 * <td>The minor version of the SIRA_PRISE specification that the client is to use.</td>
 * </tr>
 * <tr>
 * <td>ALTSIGNATURES</td>
 * <td>INTEGER</td>
 * <td>2</td>
 * <td>The number of alternative signature algorithms that the server supports, and that can be used in e.g. transaction user identity signing. Can be zero.</td>
 * </tr>
 * <tr>
 * <td>ALTSIGNATURE</td>
 * <td>STRING</td>
 * <td>&nbsp;</td>
 * <td>The name of an alternative signing algorithm that the server supports and that can be used in e.g. user identity signing when starting transactions. Appears as many times as indicated in the ALTSIGNATURES field.</td>
 * </tr>
 * </tbody>
 * </table>
 * <P>
 * The layout of the SERVEREPINFO field depends on the particular encryption protocol elected by the server and named in EP. Currently, only one encryption protocol is supported, namely SIRA_PRISE's proprietary SPE protocol. The layout of the SERVEREPINFO field for this protocol is as follows :
 * </P>
 * <table border="1" cellpadding="2" cellspacing="1" width="100%">
 * <tbody>
 * <tr>
 * <th width="24%">Zone</th>
 * <th width="12%">Format</th>
 * <th width="9%">Length</th>
 * <th width="55%">Description</th>
 * </tr>
 * <tr>
 * <td>KEY</td>
 * <td>INTEGER</td>
 * <td>4096</td>
 * <td>An array of 2048 2-byte INTEGER values.</td>
 * </tr>
 * </tbody>
 * </table>
 * Note that SPE is a symmetric algorithm. The key to be used for encryption is generated by the SIRA_PRISE server for each new connection created.
 * 
 * @author Erwin Smout
 * @since SIRA_PRISE 1.1
 */
public final class ServerHelloMessageTypeV1_0 extends ServerHelloMessageType {

	/**
	 * the instance
	 */
	private static final ServerMessageType instance = new ServerHelloMessageTypeV1_0();

	/**
	 * Gets the instance
	 * 
	 * @return the instance
	 */
	static ServerMessageType getInstance ( ) {
		return instance;
	}

	/**
     * 
     */
	private ServerHelloMessageTypeV1_0 ( ) {
		super(Version.ONE_ZERO, Version.ONE_ZERO);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.client.ServerMessageType#typeSpecificFromStream(java.io.DataInputStream)
	 */
	ServerMessage typeSpecificFromStream (DataInputStream in) throws IOException {

		Signature clientSignatureProtocol = null;
		ProprietaryOrJCECipher cryptoProtocol = null;

		byte clientSignatureProtocolIndicator = in.readByte();
		if (clientSignatureProtocolIndicator != 0) {
			String clientSignatureProtocolName = MyDataInputStream.getSmallUTFString(in);
			try {
				clientSignatureProtocol = Signature.getInstance(clientSignatureProtocolName);
			} catch (NoSuchAlgorithmException impossible) {
				throw new IOException(Messages.getString("ServerHelloMessageTypeV1_0.SignatureProtocolCriticalFailure")); //$NON-NLS-1$
			}
		}

		byte clientEncryptionProtocolIndicator = in.readByte();
		if (clientEncryptionProtocolIndicator != 0) {
			String clientCryptoProtocolName = MyDataInputStream.getSmallUTFString(in);
			try {
				cryptoProtocol = new JCECipher(Cipher.getInstance(clientCryptoProtocolName));
			} catch (NoSuchAlgorithmException e) {
				if (clientCryptoProtocolName.equalsIgnoreCase("SPE")) { //$NON-NLS-1$
					cryptoProtocol = new SPE();
				} else {
					throw new IOException(Messages.getString("ServerHelloMessageTypeV1_0.EncryptionProtocolCriticalFailure")); //$NON-NLS-1$
				}
			} catch (NoSuchPaddingException impossible) {
				throw new IOException(Messages.getString("ServerHelloMessageTypeV1_0.EncryptionProtocolCriticalFailure")); //$NON-NLS-1$
			}

			// Set the certificate holding the server's public key
			cryptoProtocol.getPublishedKey(in);
		}

		long idleTime = in.readLong();
		short majorClientSiraPriseVersion = in.readShort();
		short minorClientSiraPriseVersion = in.readShort();

		Set<String> alternativeSigningProtocols = new LinkedHashSet<String>();
		int alternativeSigningProtocolsCount = in.readShort();
		while (alternativeSigningProtocolsCount-- > 0) {
			String alternativeSigningProtocolName = MyDataInputStream.getSmallUTFString(in);
			alternativeSigningProtocols.add(alternativeSigningProtocolName);
		}

		return new ServerHelloMessageV1_0(this, clientSignatureProtocol, cryptoProtocol, idleTime, new Version(majorClientSiraPriseVersion, minorClientSiraPriseVersion), alternativeSigningProtocols);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.client.ServerMessageType#typeSpecificToStream(be.erwinsmout.SIRA_PRISE.client.ServerMessage, java.io.DataOutputStream)
	 */
	void typeSpecificToStream (ServerMessage message, DataOutputStream outputStream) throws IOException {
		if (!(message instanceof ServerHelloMessageV1_0)) {
			throw new IllegalArgumentException();
		}

		ServerHelloMessageV1_0 serverHelloMessageV1_0 = (ServerHelloMessageV1_0) message;

		Signature sp = serverHelloMessageV1_0.getSigningProtocol();
		if (sp == null) {
			outputStream.writeByte(0);
		} else {
			outputStream.writeByte(1);
			MyDataOutputStream.writeSmallUTF(sp.getAlgorithm(), outputStream);
		}

		ProprietaryOrJCECipher encryptionDecryptionProtocol = serverHelloMessageV1_0.getCryptoProtocol();
		if (encryptionDecryptionProtocol == null) {
			outputStream.writeByte(0);
		} else {
			outputStream.writeByte(1);
			MyDataOutputStream.writeSmallUTF(encryptionDecryptionProtocol.getAlgorithm(), outputStream);
			// And the engine's certificate holding the public key
			encryptionDecryptionProtocol.publishKey(outputStream);
		}

		// IDLETIME
		outputStream.writeLong(serverHelloMessageV1_0.getIdleTime());

		// SIRAPRISE VERSION
		outputStream.writeShort(serverHelloMessageV1_0.getVersion().getMajorVersion());
		outputStream.writeShort(serverHelloMessageV1_0.getVersion().getMinorVersion());

		// ALTERNATIVE SIGNATURE ALGORITHMS
		Set<String> alternativeSigningProtocols = serverHelloMessageV1_0.getAlternativeSigningProtocols();
		outputStream.writeShort(alternativeSigningProtocols.size());
		Iterator<String> i_alternativeSigningProtocols = alternativeSigningProtocols.iterator();
		while (i_alternativeSigningProtocols.hasNext()) {
			String alternativeSigningProtocolName = i_alternativeSigningProtocols.next();
			MyDataOutputStream.writeSmallUTF(alternativeSigningProtocolName, outputStream);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.client.ServerHelloMessageType#message(java.security.Signature, be.erwinsmout.SIRA_PRISE.security.ProprietaryOrJCECipher, long, be.erwinsmout.SIRA_PRISE.client.Version)
	 */
	public ServerHelloMessage message (Signature signingProtocol, ProprietaryOrJCECipher cryptoProtocol, long idleTime, Version clientSiraPriseVersion, Set<String> alternativeSigningProtocols) {
		return new ServerHelloMessageV1_0(this, signingProtocol, cryptoProtocol, idleTime, clientSiraPriseVersion, alternativeSigningProtocols);
	}
}
