/*
 * Created on 24-jul-2008
 */
package be.SIRAPRISE.messages;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;

import be.SIRAPRISE.client.Version;

/**
 * A ServerMessageType defines a type of message that can be sent to/by a SIRA_PRISE engine
 * 
 * @author Erwin Smout
 */
public abstract class ServerMessageType {

	/**
	 * Gets the SIRA_PRISE specification version in which this message type (version !) was introduced.
	 */
	private Version introducedInSiraPriseVersion;

	/**
	 * the numeric ID of a particular message type
	 */
	private int messageTypeID;

	/**
	 * the Version indicator of the message type
	 */
	private Version messageVersion;

	/**
	 * Creates the ServerMessageType, setting its needed attributes
	 * 
	 * @param messageTypeID
	 *            the numeric ID of a particular message type
	 * @param messageVersion
	 *            the Version indicator of the message type
	 * @param introducedInSiraPriseVersion
	 *            Gets the SIRA_PRISE specification version in which this message type (version !) was introduced.
	 */
	ServerMessageType (int messageTypeID, Version messageVersion, Version introducedInSiraPriseVersion) {
		this.messageTypeID = messageTypeID;
		this.messageVersion = messageVersion;
		this.introducedInSiraPriseVersion = introducedInSiraPriseVersion;
	}

	/**
	 * Gets the major version of a particular message type
	 * 
	 * @return the major version of a particular message type
	 */
	final short getMajorVersion ( ) {
		return messageVersion.getMajorVersion();
	}

	/**
	 * Gets the numeric ID of a particular message type
	 * 
	 * @return the numeric ID of a particular message type
	 */
	final int getMessageTypeID ( ) {
		return messageTypeID;
	}

	/**
	 * Gets the Version indicator of the message type
	 * 
	 * @return the Version indicator of the message type
	 */
	final Version getMessageVersion ( ) {
		return messageVersion;
	}

	/**
	 * Gets the minor version of a particular message type
	 * 
	 * @return the minor version of a particular message type
	 */
	final short getMinorVersion ( ) {
		return messageVersion.getMinorVersion();
	}

	/**
	 * The method used by readers to obtain a (structured) message from the given stream.
	 * 
	 * @param in
	 *            The inputstream from which the ServerMessage to be returned is to be constructed
	 * @return The ServerMessage of this type
	 * @throws IOException
	 */
	abstract ServerMessage typeSpecificFromStream (DataInputStream in) throws IOException;

	/**
	 * Writes the type- (and version-) specific part of the message to the given stream
	 * 
	 * @param message
	 *            The message to be written to the stream
	 * @param outputStream
	 *            The stream where the message is to go
	 * @throws IOException
	 */
	abstract void typeSpecificToStream (ServerMessage message, DataOutputStream outputStream) throws IOException;

	/**
	 * Gets the SIRA_PRISE specification version in which this message type (version !) was introduced. e.g. ClientHelloMessageType V 1_0 was introduced in SIRA_PRISE version 1_0, e.g. ClientHelloMessageType V 1_1 was introduced in SIRA_PRISE version 2_3, meaning that ClientHelloMessageType V 1_0 applied to all SIRA_PRISE versions up to and not including V 2_3
	 * 
	 * @return The version in which this message type (version !) was introduced
	 */
	final Version introducedInSiraPriseVersion ( ) {
		return introducedInSiraPriseVersion;
	}
}