/*
 * Created on 26-jul-2008
 */
package be.SIRAPRISE.messages;

import java.util.Iterator;
import java.util.SortedMap;
import java.util.TreeMap;

import be.SIRAPRISE.client.Version;
import be.SIRAPRISE.client.IDS.VERSION;
import be.erwinsmout.NotFoundException;

/**
 * @author Erwin Smout
 */
public class ServerMessageTypes {

	/**
	 * The instance
	 */
	private static final ServerMessageTypes instance = new ServerMessageTypes();

	/**
	 * Gets The instance
	 * 
	 * @return The instance
	 */
	public static ServerMessageTypes getInstance ( ) {
		return instance;
	}

	/**
	 * The map containing all known message types and versions
	 */
	private TreeMap<Long, ServerMessageType> serverMessageTypes = new TreeMap<Long, ServerMessageType>();

	/**
	 * The specification version that this package corresponds to
	 */
	private Version thisPackagesSiraPriseVersion;

	/**
	 * The message type id
	 */
	public static final int AUTHENTICATIONOK = 0x08115100;

	/**
	 * 
	 */
	private ServerMessageTypes ( ) {
		addServerMessageType(ConstraintViolatedErrorMessageTypeV1_0.getInstance());
		addServerMessageType(NonSiraPriseErrorMessageTypeV1_0.getInstance());
		addServerMessageType(NonSiraPriseErrorMessageTypeV1_1.getInstance());
		addServerMessageType(SiraPriseErrorMessageTypeV1_0.getInstance());
		addServerMessageType(SiraPriseErrorMessageTypeV1_1.getInstance());

		addServerMessageType(ClientAuthenticationMessageTypeV1_0.getInstance());
		addServerMessageType(ClientHelloMessageTypeV1_0.getInstance());
		addServerMessageType(CommitRollbackMessageTypeV1_0.getInstance());
		addServerMessageType(CommitRollbackMessageTypeV1_1.getInstance());
		addServerMessageType(EndConnectionMessageTypeV1_0.getInstance());
		addServerMessageType(EndTransactionMessageTypeV1_0.getInstance());
		addServerMessageType(EndTransactionMessageTypeV1_1.getInstance());
		addServerMessageType(ExecuteDMLMessageTypeV1_0.getInstance());
		addServerMessageType(ExecuteAssignmentMessageTypeV1_0.getInstance());
		addServerMessageType(ExecuteQueryMessageTypeV1_0.getInstance());
		addServerMessageType(ExecuteMonitorCommandMessageTypeV1_0.getInstance());
		addServerMessageType(StartSubTransactionMessageTypeV1_0.getInstance());
		addServerMessageType(StartTransactionMessageTypeV1_0.getInstance());
		addServerMessageType(StartTransactionMessageTypeV1_1.getInstance());
		addServerMessageType(StartTransactionMessageTypeV1_2.getInstance());

		addServerMessageType(AuthenticationOKMessageTypeV1_0.getInstance());
		addServerMessageType(CommittedMessageTypeV1_0.getInstance());
		addServerMessageType(DmlExecutedMessageTypeV1_0.getInstance());
		addServerMessageType(ServerHelloMessageTypeV1_0.getInstance());
		addServerMessageType(TransactionEndedMessageTypeV1_0.getInstance());
		addServerMessageType(TransactionStartedMessageTypeV1_0.getInstance());
		addServerMessageType(TransactionStartedMessageTypeV1_1.getInstance());

		thisPackagesSiraPriseVersion = new Version(VERSION.MAJORSPECIFICATIONVERSION, VERSION.MINORSPECIFICATIONVERSION);
	}

	/**
	 * Adds a ServerMessageType to the pool
	 * 
	 * @param serverMessageType
	 *            the server message type to be added to the pool
	 */
	private void addServerMessageType (ServerMessageType serverMessageType) {
		serverMessageTypes.put(new Long(((long) serverMessageType.getMessageTypeID() << 32) | serverMessageType.getMessageVersion().getFullVersion()), serverMessageType);
	}

	/**
	 * Gets the most recent version of the ServerMessageType of the indicated type, that is, the version that is used in the SIRA_PRISE version that this package conforms to / corresponds with.
	 * 
	 * @param messageType
	 *            the message type family to be obtained
	 * @return The ServerMessageType of the indicated type and version
	 * @throws NotFoundException
	 */
	public ServerMessageType getServerMessageType (int messageType) throws NotFoundException {
		return getServerMessageTypeForSiraPriseVersion(messageType, thisPackagesSiraPriseVersion);
	}

	/**
	 * Gets The ServerMessageType of the indicated type and version
	 * 
	 * @param messageType
	 *            the message type family to be obtained
	 * @param messageVersion
	 *            the version of the message type to be retrieved
	 * @return The ServerMessageType of the indicated type and version
	 * @throws NotFoundException
	 */
	public ServerMessageType getServerMessageTypeForMessageVersion (int messageType, int messageVersion) throws NotFoundException {
		Long key = new Long(((long) messageType << 32) | messageVersion);
		if (serverMessageTypes.containsKey(key)) {
			return serverMessageTypes.get(key);
		}
		throw new NotFoundException();
	}

	/**
	 * Gets The ServerMessageType of the indicated type that was used in the indicated SIRA_PRISE specification version
	 * 
	 * @param messageType
	 *            the message type family to be obtained
	 * @param siraPriseVersion
	 *            the SIRA_PRISE specificiation version applying to the message type to be retrieved
	 * @return The ServerMessageType of the indicated type and version
	 * @throws NotFoundException
	 */
	public ServerMessageType getServerMessageTypeForSiraPriseVersion (int messageType, Version siraPriseVersion) throws NotFoundException {
		Long lowest = new Long((long) messageType << 32);
		Long highest = new Long(((long) messageType << 32) | 0x00000000ffffffffL); // Excludes the ffffffff value, but that's harmless
		SortedMap<Long, ServerMessageType> messageTypeVersions = serverMessageTypes.subMap(lowest, highest);
		Iterator<ServerMessageType> i_messageTypeVersions = messageTypeVersions.values().iterator();
		ServerMessageType returnServerMessageType = null;
		boolean moreRecentPossible = true;
		while (i_messageTypeVersions.hasNext() && moreRecentPossible) {
			ServerMessageType serverMessageType = i_messageTypeVersions.next();
			if ((serverMessageType.introducedInSiraPriseVersion().compareTo(siraPriseVersion) < 1)) { // -1 : lower ; 0 : equal
				returnServerMessageType = serverMessageType;
			} else {
				moreRecentPossible = false;
			}
		}
		if (returnServerMessageType == null) {
			throw new NotFoundException();
		}
		return returnServerMessageType;
	}

	/**
	 * Gets The specification version that this package corresponds to
	 * 
	 * @return The specification version that this package corresponds to
	 */
	public Version getThisPackagesSiraPriseVersion ( ) {
		return thisPackagesSiraPriseVersion;
	}
}
