/*
 * Created on 29-jul-2008
 */
package be.SIRAPRISE.messages;

/**
 * @author Erwin Smout
 */
public abstract class StartTransactionMessage extends ServerInputMessage {

	/**
	 * For read-only transactions, indicates whether or not the read locks acquired by the server can be release after each individual statement executed by the transaction
	 */
	private boolean autoCommit;

	/**
	 * Flag indicating whether the server's DDL capture facility should be activated for this transaction
	 */
	private boolean captureDDL;

	/**
	 * flag indicating whether or not the new transaction is a read-only one, i.e. whether it will or will not attempt updates to the database
	 */
	private boolean readOnlyTransaction;

	/**
	 * the signature computed by the user
	 */
	private byte[] signature;

	/**
	 * Flag indicating whether the user identified by getUserID() has been authenticated by the client, meaning that if the server trusts the client, the server needn't do user authentication of its own.
	 */
	private boolean userAuthenticatedByClient;

	/**
	 * name of the user authentication algorithm to be used by the server if the decides that it will do user authentication itself
	 */
	private String userAuthenticationAlgorithm;

	/**
	 * The identification of the user owning the new transaction
	 */
	private String userID;

	/**
	 * @param type
	 *            The MessageType object representing the type (and version of type) this new Message is of
	 * @param userID
	 *            The identification of the user owning the new transaction
	 * @param userAuthenticatedByClient
	 *            indicates whether the user owning the new transaction has been authenticated by the client
	 * @param userAuthenticationAlgorithm
	 *            name of the user authentication algorithm to be used by the server if the decides that it will do user authentication itself
	 * @param signature
	 *            The signature to be verified by the server if the decides that it will do user authentication itself
	 * @param autoCommit
	 *            flag indicating whether, for read-only transactions, read locks can be released by the server after completion of each individual statement, rather than at transaction commit time
	 * @param readOnlyTransaction
	 *            flag indicating whether each individual statement is automatically committed (meaning, for a read-only transaction, that locks will immediately be released when the statement completes)
	 * @param captureDDL
	 *            flag indicating whether the server's DDL capture facility should be activated for this transaction
	 */
	StartTransactionMessage (StartTransactionMessageType type, String userID, boolean userAuthenticatedByClient, String userAuthenticationAlgorithm, byte[] signature, boolean readOnlyTransaction, boolean autoCommit, boolean captureDDL) {
		super(type);
		this.userID = userID;
		this.userAuthenticatedByClient = userAuthenticatedByClient;
		this.userAuthenticationAlgorithm = userAuthenticationAlgorithm;
		this.signature = signature;
		this.readOnlyTransaction = readOnlyTransaction;
		this.autoCommit = autoCommit;
		this.captureDDL = captureDDL;
	}

	/**
	 * Gets the signature computed by the user
	 * 
	 * @return the signature computed by the user
	 */
	public final byte[] getSignature ( ) {
		return signature;
	}

	/**
	 * Gets the name of the user authentication algorithm to be used by the server if the decides that it will do user authentication itself
	 * 
	 * @return the name of the user authentication algorithm to be used by the server if the decides that it will do user authentication itself
	 */
	public final String getUserAuthenticationAlgorithm ( ) {
		return userAuthenticationAlgorithm;
	}

	/**
	 * Gets The identification of the user owning the new transaction
	 * 
	 * @return The identification of the user owning the new transaction
	 */
	public final String getUserID ( ) {
		return userID;
	}

	/**
	 * Checks whether, for read-only transactions, the read locks acquired by the server can be released after each individual statement executed by the transaction
	 * 
	 * @return true if, for read-only transactions, the read locks acquired by the server can be released after each individual statement executed by the transaction
	 */
	public final boolean isAutoCommit ( ) {
		return autoCommit;
	}

	/**
	 * Gets the flag indicating whether the server's DDL capture facility should be activated for this transaction
	 * 
	 * @return the flag indicating whether the server's DDL capture facility should be activated for this transaction
	 */
	public final boolean isCaptureDDL ( ) {
		return captureDDL;
	}

	/**
	 * Checks whether or not the new transaction is a read-only one, i.e. whether it will or will not attempt updates to the database
	 * 
	 * @return true if the new transaction is a read-only one, i.e. if it will not attempt updates to the database
	 */
	public final boolean isReadOnlyTransaction ( ) {
		return readOnlyTransaction;
	}

	/**
	 * Gets the flag indicating whether the user identified by getUserID() has been authenticated by the client, meaning that if the server trusts the client, the server needn't do user authentication of its own.
	 * 
	 * @return true if the user identified by getUserID() has been authenticated by the client, meaning that if the server trusts the client, the server needn't do user authentication of its own.
	 */
	public final boolean isUserAuthenticatedByClient ( ) {
		return userAuthenticatedByClient;
	}

}
