/*
 * Created on 29-jul-2008
 */
package be.SIRAPRISE.messages;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;

import be.SIRAPRISE.client.DDLCapture;
import be.SIRAPRISE.client.Version;
import be.SIRAPRISE.util.MyDataInputStream;
import be.SIRAPRISE.util.MyDataOutputStream;

/**
 * StartTransactionMessageTypeV1_0 is V1_0 of the StartTransactionMessageType.
 * <P>
 * This message has the following fields :
 * </P>
 * 
 * <table border="1" cellpadding="2" cellspacing="1" width="100%">
 * <tbody>
 * <tr>
 * <th width="24%">Zone</th>
 * <th width="12%">Format</th>
 * <th width="9%">Length</th>
 * <th width="55%">Description</th>
 * </tr>
 * <tr>
 * <td>USERID</td>
 * <td>STRING</td>
 * <td>&nbsp;</td>
 * <td>The name of the user starting the transaction. Can be the zero-length string, but if not, it must be a name that appears in some tuple of the USER relvar.</td>
 * </tr>
 * <tr>
 * <td>AUTHBYCLIENT</td>
 * <td>INTEGER</td>
 * <td>1</td>
 * <td>1 if the client has authenticated the user, 0 otherwise.</td>
 * </tr>
 * <tr>
 * <td>AUTHALGORITHM</td>
 * <td>STRING</td>
 * <td>&nbsp;</td>
 * <td>The name of the signing algorithm used to compute SIGNATURE. Can be the zero-length string.</td>
 * </tr>
 * <tr>
 * <td>SIGNATURELENGTH</td>
 * <td>INTEGER</td>
 * <td>2</td>
 * <td>The number of bytes in SIGNATURE. Can be zero, in which case the SIGNATURE field is omitted from the message.</td>
 * </tr>
 * <tr>
 * <td>SIGNATURE</td>
 * <td>BYTES</td>
 * <td>&nbsp;</td>
 * <td>The signature as computed by the signature algorithm identified by AUTHALGORITHM, using the user's private key corresponding to the public key that is registered for the user in the USERCERTIFICATE relvar, and applied to the USERID field.</td>
 * </tr>
 * </tbody>
 * </table>
 * 
 * <P>
 * SIRA_PRISE 1.2 introduced two new fields in this message type : a readonly flag and an autocommit flag. The default assumed by 1.2 servers for these fields when sent this message type is 'false' in both occasions, meaning that 1.2 servers getting this request from a 1.1 client, will start an update transaction with no autocommit, which is the behaviour previously displayed by 1.1 servers.
 * </P>
 * 
 * @author Erwin Smout
 * @since SIRA_PRISE 1.1
 */
public final class StartTransactionMessageTypeV1_0 extends StartTransactionMessageType {

	/**
	 * the instance
	 */
	private static StartTransactionMessageType instance = new StartTransactionMessageTypeV1_0();

	/**
	 * Gets the instance
	 * 
	 * @return the instance
	 */
	static StartTransactionMessageType getInstance ( ) {
		return instance;
	}

	/**
	 * 
	 */
	private StartTransactionMessageTypeV1_0 ( ) {
		super(Version.ONE_ZERO, Version.ONE_ZERO);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.client.ServerMessageType#typeSpecificFromStream(java.io.DataInputStream)
	 */
	ServerMessage typeSpecificFromStream (DataInputStream in) throws IOException {
		String userID = MyDataInputStream.getSmallUTFString(in);
		boolean authenticatedByClient = (in.readByte() == 1);
		String userAuthenticationAlgorithm = MyDataInputStream.getSmallUTFString(in);
		short signatureLength = in.readShort();
		byte[] signature = new byte[signatureLength];
		in.read(signature);

		return new StartTransactionMessageV1_0(this, userID, authenticatedByClient, userAuthenticationAlgorithm, signature);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.client.ServerMessageType#typeSpecificToStream(be.erwinsmout.SIRA_PRISE.client.ServerMessage, java.io.DataOutputStream)
	 */
	void typeSpecificToStream (ServerMessage message, DataOutputStream outputStream) throws IOException {
		if (!(message instanceof StartTransactionMessageV1_0)) {
			throw new IllegalArgumentException();
		}

		StartTransactionMessageV1_0 startTransactionMessageMessageV1_0 = (StartTransactionMessageV1_0) message;

		MyDataOutputStream.writeSmallUTF(startTransactionMessageMessageV1_0.getUserID(), outputStream);
		outputStream.writeByte(startTransactionMessageMessageV1_0.isUserAuthenticatedByClient() ? 1 : 0);
		MyDataOutputStream.writeSmallUTF(startTransactionMessageMessageV1_0.getUserAuthenticationAlgorithm(), outputStream);
		byte[] signature = startTransactionMessageMessageV1_0.getSignature();
		outputStream.writeShort(signature.length);
		outputStream.write(signature);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.messages.StartTransactionMessageType#message(java.lang.String, boolean, java.lang.String, byte[])
	 */
	public StartTransactionMessage message (String userID, boolean userAuthenticatedByClient, String signatureAlgorithm, byte[] signature, boolean readOnlyTransaction, boolean autoCommit, DDLCapture captureDDL) {
		if (readOnlyTransaction || autoCommit || captureDDL.isCapturingDDL()) {
			throw new IllegalArgumentException(Messages.getString("StartTransactionMessageTypeV1_0.IncompatibleWithServer")); //$NON-NLS-1$
		}
		return new StartTransactionMessageV1_0(this, userID, userAuthenticatedByClient, signatureAlgorithm, signature);
	}
}
