/*
 * Created on 7-jun-04
 */
package be.SIRAPRISE.typeimplementations;

import java.util.*;

import be.SIRAPRISE.client.NAMES.POSSREPCOMPONENTNAMES;
import be.SIRAPRISE.client.NAMES.POSSREPNAMES;
import be.SIRAPRISE.client.NAMES.TYPENAMES;
import be.SIRAPRISE.util.BracketParser;
import be.SIRAPRISE.util.DuplicateNameException;
import be.SIRAPRISE.util.InvalidEscapedCharacterException;
import be.SIRAPRISE.util.MissingEscapedCharacterException;
import be.SIRAPRISE.util.MyReadOnlyMap;
import be.SIRAPRISE.util.MyReadOnlySet;
import be.SIRAPRISE.util.NoClosingBracketException;
import be.SIRAPRISE.util.NoOpeningBracketException;
import be.erwinsmout.MyMessageFormat;

/**
 * Type 'Angle' is provided to demonstrate the type-plugin feature. Its encoding is as follows :
 * <ul>
 * <li>The angle value is encoded as an 8-byte java double</li>
 * <li>The range of this double is ]-PI - PI] (meaning that negative PI is invalid, but positive PI is valid).</li>
 * </ul>
 * 
 * @author Erwin Smout
 */
public final class DbmsAngleImplementation implements TypeImplementation, PossRepImplementation {

	/**
	 * Angleradians is the implementation of the 'Radians' possrep of an Angle value. This possrep represents angles as a floating-point value in the range ]-PI PI].
	 * 
	 * @author Erwin Smout
	 */
	final class Radians implements PossRepImplementation {

		/**
		 * The component names/typenames of this possrep
		 */
		private final Map<String, String> radiansComponentNameMap = initRadiansComponentNameMap();

		/**
		 * 
		 */
		Radians ( ) {

		}

		/**
		 * @param d
		 */
		private void checkFloatRange (double d) {
			if (d <= -Math.PI || d > Math.PI) {
				throw new IllegalArgumentException(Messages.getString("DbmsAngleImplementation.Range")); //$NON-NLS-1$
			}
		}

		/**
		 * @return
		 */
		private Map<String, String> initRadiansComponentNameMap ( ) {
			HashMap<String, String> w = new HashMap<String, String>();
			w.put(POSSREPCOMPONENTNAMES.RADIANS, TYPENAMES.FLOAT);
			return new MyReadOnlyMap<String, String>(w);
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see be.erwinsmout.SIRA_PRISE.TypeImplementations.PossRepImplementation#getComponentNameMap()
		 */
		public Map<String, String> getComponentNameMap ( ) {
			return radiansComponentNameMap;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.PossRepImplementation#valueToExternalRepresentationForComponent(java.lang.String, be.erwinsmout.SIRA_PRISE.typeimplementations.ScalarValueBuffer)
		 */
		public ValueBuffer getComponentValue (String componentName, ScalarValueBuffer valueBuffer) {
			return valueBuffer;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.PossRepImplementation#getName()
		 */
		public String getPossrepName ( ) {
			return POSSREPNAMES.RADIANS;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.PossRepImplementation#valueFromComponentValues(java.util.HashMap, int)
		 */
		public ValueBuffer valueFromComponentValues (HashMap<String, ValueBuffer> componentValueMap, int logicalSize) {
			ValueBuffer floatRadians = componentValueMap.get(POSSREPCOMPONENTNAMES.RADIANS);
			double radians = DbmsFloatImplementation.getJavaDouble(floatRadians);
			checkFloatRange(radians);
			return floatRadians;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see be.erwinsmout.DBMS.Server.PossRepImplementation#valueFromExternalRepresentation(java.lang.String)
		 */
		public ValueBuffer valueFromExternalRepresentation (String value) throws InvalidValueException {
			try {
				double d = Double.parseDouble(value);
				checkFloatRange(d);
				return DbmsFloatImplementation.getValueBuffer(d);
			} catch (NumberFormatException e) {
				throw new InvalidValueException(MyMessageFormat.format(Messages.getString("DbmsAngleImplementation.Value"), new String[] { value, e.getMessage() })); //$NON-NLS-1$
			}
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see be.erwinsmout.DBMS.Server.PossRepImplementation#valueFromExternalRepresentation(java.lang.String, int)
		 */
		public ValueBuffer valueFromExternalRepresentation (String value, int maximumLogicalLengthAllowed) throws InvalidValueException {
			return valueFromExternalRepresentation(value);
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.PossRepImplementation#valueToEscapedExternalRepresentation(be.erwinsmout.SIRA_PRISE.typeimplementations.ValueBuffer)
		 */
		public String valueToEscapedExternalRepresentation (ValueBuffer valueBuffer) {
			return valueToExternalRepresentation(valueBuffer);
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see be.erwinsmout.DBMS.Server.PossRepImplementation#valueToExternalRepresentation(java.nio.ByteBuffer)
		 */
		public String valueToExternalRepresentation (ValueBuffer valueBuffer) {
			return Double.toString(DbmsFloatImplementation.getJavaDouble(valueBuffer));
		}
	}

	/**
	 * the instance.
	 */
	private static DbmsAngleImplementation instance = new DbmsAngleImplementation();

	/**
	 * Gets A ValueBuffer holding the given value
	 * 
	 * @param degrees
	 * @param minutes
	 * @param seconds
	 * @return A ValueBuffer holding the given value
	 */
	public static ValueBuffer getAngleValuebuffer (int degrees, int minutes, double seconds) {
		if (degrees < 0 || degrees >= 360 || minutes < 0 || minutes >= 60 || seconds < 0 || seconds >= 60.0D) {
			throw new IllegalArgumentException(Messages.getString("DbmsAngleImplementation.DMSRange")); //$NON-NLS-1$
		}
		double d = Math.PI * ((seconds / 3600 + minutes / 60.0D + degrees) / 180.0D);

		return getAngleValueBuffer(d);
	}

	/**
	 * Gets a ValueBuffer holding the angle value represented by the argument
	 * 
	 * @param radians
	 *            The number of radians
	 * @return a ValueBuffer holding the angle value represented by the argument
	 */
	public static ScalarValueBuffer getAngleValueBuffer (double radians) {
		double d;
		double dPI = 2 * Math.PI;
		if (radians > dPI) {
			d = radians % dPI;
		} else {
			d = radians;
		}
		if (d < -dPI) {
			d = d % -dPI;
		}
		if (d > Math.PI) {
			d -= dPI;
		}
		if (d <= -Math.PI) {
			d += dPI;
		}

		return DbmsFloatImplementation.getValueBuffer(d);
	}

	/**
	 * Gets the instance
	 * 
	 * @return the instance
	 */
	public static DbmsAngleImplementation getInstance ( ) {
		return instance;
	}

	/**
	 * Gets the angle value as a value of the most appropriate java type
	 * 
	 * @param scalarValueBuffer
	 *            A ValueBuffer holding an Angle value
	 * @return A java double in the range [-PI - +PI[
	 */
	public static double getJavaDouble (ScalarValueBuffer scalarValueBuffer) {
		return DbmsFloatImplementation.getJavaDouble(scalarValueBuffer);
	}

	/**
	 * The components of the default possrep (=degrees minutes seconds)
	 */
	private final MyReadOnlyMap<String, String> componentNameMap = initComponentNameMap();

	/**
	 * Map of the possreps
	 */
	private final Set<PossRepImplementation> possrepImplementations = new MyReadOnlySet<PossRepImplementation>(new HashSet<PossRepImplementation>(Arrays.asList(new PossRepImplementation[] { new Radians(), this })));

	/**
	 * 
	 */
	private final Map<String, PossRepImplementation> possrepImplementationsPerComponent = initPossrepImplementationsPerComponent();

	/**
	 * Builds the DbmsAngleImplementation object
	 */
	private DbmsAngleImplementation ( ) {

	}

	/**
	 * @return
	 */
	private Map<String, PossRepImplementation> initPossrepImplementationsPerComponent ( ) {
		HashMap<String, PossRepImplementation> w = new HashMap<String, PossRepImplementation>();
		for (PossRepImplementation possrepImplementation : possrepImplementations) {
			for (String componentName : possrepImplementation.getComponentNameMap().keySet()) {
				w.put(componentName, possrepImplementation);
			}
		}
		return new MyReadOnlyMap<String, PossRepImplementation>(w);
	}

	/**
	 * @param d
	 *            A radians value
	 * @return The degrees component of the radians value represented by d
	 */
	private int getDegrees (double d) {
		return new Double(d).intValue();
	}

	/**
	 * @param d
	 *            A radians value
	 * @return The minutes component of the radians value represented by d
	 */
	private int getMinutes (double d) {
		return new Double((d % 1.0) * 60.0).intValue();
	}

	/**
	 * @param d
	 *            A radians value
	 * @return The seconds component of the radians value represented by d
	 */
	private double getSeconds (double d) {
		return (((d % 1.0) * 60.0) % 1.0) * 60.0;
	}

	/**
	 * @return
	 */
	private static MyReadOnlyMap<String, String> initComponentNameMap ( ) {
		HashMap<String, String> w = new HashMap<String, String>();
		// components DEGREES MINUTES SECONDS of the 'this' possrep (the ANGLE one)
		w.put(POSSREPCOMPONENTNAMES.DEGREES, TYPENAMES.INT);
		w.put(POSSREPCOMPONENTNAMES.MINUTES, TYPENAMES.INT);
		w.put(POSSREPCOMPONENTNAMES.SECONDS, TYPENAMES.FLOAT);
		return new MyReadOnlyMap<String, String>(w);
	}

	/**
	 * Returns a corresponding radians value in the range [0-2PI[
	 * 
	 * @param valueBuffer
	 *            The byteBuffer holding the angle value
	 * @return The degrees value in the range [0-360[
	 */
	private double strictlyPositiveNormalForm (ScalarValueBuffer valueBuffer) {
		double d = DbmsFloatImplementation.getJavaDouble(valueBuffer);
		if (d < 0) {
			d = 2 * Math.PI + d;
		}
		d = d / Math.PI * 180.0D;
		return d;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.TypeImplementations.PossRepImplementation#getComponentNameMap()
	 */
	public Map<String, String> getComponentNameMap ( ) {
		return componentNameMap;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.PossRepImplementation#extractComponentValue(java.lang.String, be.erwinsmout.SIRA_PRISE.typeimplementations.ScalarValueBuffer)
	 */
	public ValueBuffer getComponentValue (String componentName, ScalarValueBuffer valueBuffer) {
		double d = strictlyPositiveNormalForm(valueBuffer);
		if (componentName.equalsIgnoreCase(POSSREPCOMPONENTNAMES.DEGREES)) {
			return DbmsIntImplementation.getValueBuffer(getDegrees(d));
		}
		if (componentName.equalsIgnoreCase(POSSREPCOMPONENTNAMES.MINUTES)) {
			return DbmsIntImplementation.getValueBuffer(getMinutes(d));
		}
		if (componentName.equalsIgnoreCase(POSSREPCOMPONENTNAMES.SECONDS)) {
			return DbmsFloatImplementation.getValueBuffer(getSeconds(d));
		}
		throw new IllegalArgumentException(componentName);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.TypeImplementation#getDefaultPossrepImplementation()
	 */
	public PossRepImplementation getDefaultPossrepImplementation ( ) {
		return this;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.DBMS.TypeImplementations.TypeImplementation#getDefaultSize()
	 */
	public int getDfltLogicalSize ( ) {
		return 1;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.DBMS.TypeImplementations.TypeImplementation#getMaxSize()
	 */
	public int getMaxLogicalSize ( ) {
		return 1;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.DBMS.TypeImplementations.TypeImplementation#getMinSize()
	 */
	public int getMinLogicalSize ( ) {
		return 1;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.TypeImplementation#getPhysicalSizeFor(int)
	 */
	public int getPhysicalSizeFor (int logicalSize) {
		return 8;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.SIRAPRISE.typeimplementations.TypeImplementation#getPossrepImplementation(java.lang.String)
	 */
	@Override
	public PossRepImplementation getPossrepImplementation (String componentName) {
		PossRepImplementation possRepImplementation = possrepImplementationsPerComponent.get(componentName);
		if (possRepImplementation == null) {
			throw new IllegalArgumentException(MyMessageFormat.format(Messages.getString("IntervalTypeImplementation.UnknownCompoenent"), new String[] { componentName, this.getClass().getName() })); //$NON-NLS-1$
		}
		return possRepImplementation;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.TypeImplementation#getPossRepImplementation()
	 */
	public Set<PossRepImplementation> getPossrepImplementations ( ) {
		return possrepImplementations;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.PossRepImplementation#getName()
	 */
	public String getPossrepName ( ) {
		return POSSREPNAMES.ANGLE;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.PossRepImplementation#valueFromComponentValues(java.util.HashMap, int)
	 */
	public ValueBuffer valueFromComponentValues (HashMap<String, ValueBuffer> componentValueMap, int logicalSize) {
		int degrees = DbmsIntImplementation.getJavaInt((ScalarValueBuffer) componentValueMap.get(POSSREPCOMPONENTNAMES.DEGREES));
		int minutes = DbmsIntImplementation.getJavaInt((ScalarValueBuffer) componentValueMap.get(POSSREPCOMPONENTNAMES.MINUTES));
		double seconds = DbmsFloatImplementation.getJavaDouble(componentValueMap.get(POSSREPCOMPONENTNAMES.SECONDS));
		return getAngleValuebuffer(degrees, minutes, seconds);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.DBMS.TypeImplementations.TypeImplementation#valueFromExternalRepresentation(java.lang.String)
	 */
	public ValueBuffer valueFromExternalRepresentation (String value) throws InvalidValueException {
		Map<String, String> componentValuesMap;
		try {
			componentValuesMap = BracketParser.createMapFromEscapedString(value, false);
		} catch (NoClosingBracketException e1) {
			throw new InvalidValueException(Messages.getString("DbmsAngleImplementation.BracketSyntax"), e1); //$NON-NLS-1$
		} catch (InvalidEscapedCharacterException e1) {
			throw new InvalidValueException(Messages.getString("DbmsAngleImplementation.BracketSyntax"), e1); //$NON-NLS-1$
		} catch (MissingEscapedCharacterException e1) {
			throw new InvalidValueException(Messages.getString("DbmsAngleImplementation.BracketSyntax"), e1); //$NON-NLS-1$
		} catch (NoOpeningBracketException e1) {
			throw new InvalidValueException(Messages.getString("DbmsAngleImplementation.BracketSyntax"), e1); //$NON-NLS-1$
		} catch (DuplicateNameException e1) {
			throw new InvalidValueException(Messages.getString("DbmsAngleImplementation.BracketSyntax"), e1); //$NON-NLS-1$
		}

		String degrees_tx = componentValuesMap.get(POSSREPCOMPONENTNAMES.DEGREES);
		String minutes_tx = componentValuesMap.get(POSSREPCOMPONENTNAMES.MINUTES);
		String seconds_tx = componentValuesMap.get(POSSREPCOMPONENTNAMES.SECONDS);
		if (degrees_tx == null || minutes_tx == null || seconds_tx == null) {
			throw new InvalidValueException(Messages.getString("DbmsAngleImplementation.ComponentMissing")); //$NON-NLS-1$
		}
		try {
			int degrees = Integer.parseInt(degrees_tx);
			int minutes = Integer.parseInt(minutes_tx);
			double seconds = Double.parseDouble(seconds_tx);

			return getAngleValuebuffer(degrees, minutes, seconds);
		} catch (NumberFormatException e) {
			throw new InvalidValueException(Messages.getString("DbmsAngleImplementation.ValidTokensDMS")); //$NON-NLS-1$
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.DBMS.TypeImplementations.TypeImplementation#valueFromExternalRepresentation(java.lang.String, int)
	 */
	public ValueBuffer valueFromExternalRepresentation (String value, int maximumLogicalLengthAllowed) throws InvalidValueException {
		return valueFromExternalRepresentation(value);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.PossRepImplementation#valueToEscapedExternalRepresentation(be.erwinsmout.SIRA_PRISE.typeimplementations.ValueBuffer)
	 */
	public String valueToEscapedExternalRepresentation (ValueBuffer valueBuffer) {
		return valueToExternalRepresentation(valueBuffer);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.DBMS.TypeImplementations.TypeImplementation#valueToExternalRepresentation(java.nio.ByteBuffer, int, int, int)
	 */
	public String valueToExternalRepresentation (ValueBuffer valueBuffer) {
		double d = strictlyPositiveNormalForm((ScalarValueBuffer) valueBuffer);
		int degrees = new Double(d).intValue();
		d = (d % 1.0D) * 60.0;
		int minutes = new Double(d).intValue();
		d = (d % 1.0D) * 60.0;
		double seconds = d;
		return POSSREPCOMPONENTNAMES.DEGREES + "(" + Integer.toString(degrees) + ")" + POSSREPCOMPONENTNAMES.MINUTES + "(" + Integer.toString(minutes) + ")" + POSSREPCOMPONENTNAMES.SECONDS + "(" + Double.toString(seconds) + ")"; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$ //$NON-NLS-5$ //$NON-NLS-6$
	}
}