/*
 * Created on 7-jun-04
 */
package be.SIRAPRISE.typeimplementations;

import java.nio.ByteBuffer;
import java.util.*;

import be.SIRAPRISE.client.NAMES.POSSREPCOMPONENTNAMES;
import be.SIRAPRISE.client.NAMES.POSSREPNAMES;
import be.SIRAPRISE.client.NAMES.TYPENAMES;
import be.SIRAPRISE.util.BracketParser;
import be.SIRAPRISE.util.DuplicateNameException;
import be.SIRAPRISE.util.InvalidEscapedCharacterException;
import be.SIRAPRISE.util.MissingEscapedCharacterException;
import be.SIRAPRISE.util.MyReadOnlyMap;
import be.SIRAPRISE.util.MyReadOnlySet;
import be.SIRAPRISE.util.NoClosingBracketException;
import be.SIRAPRISE.util.NoOpeningBracketException;
import be.erwinsmout.MyMessageFormat;

/**
 * Type 'Average' is used when computing averages. Its physical encoding is as follows :
 * <ul>
 * <li>The byte length of an encoded AVERAGE value is 12, bytes labeled b11-b0. These bytes contain the components COUNT and VALUE.</li>
 * <li>Bytes b11-b8 contain the COUNT component as a positive java integer.</li>
 * <li>Bytes b7-b0 contain a java double representing the value of the VALUE component.</li>
 * </ul>
 * 
 * @author Erwin Smout
 */
public final class DbmsAverageImplementation implements TypeImplementation, PossRepImplementation {

	/**
	 * the instance.
	 */
	private static final DbmsAverageImplementation instance = new DbmsAverageImplementation();

	/**
	 * The physical encoding size, catering for one INT and one FLOAT
	 */
	private static final int PHYSICALENCODINGSIZE = 12;

	/**
	 * Gets A ValueBuffer holding the given value
	 * 
	 * @param count
	 * @param value
	 * @return A ValueBuffer holding the given value
	 */
	public static ValueBuffer getAverageValueBuffer (int count, double value) {
		if (count < 0) {
			throw new IllegalArgumentException(MyMessageFormat.format(Messages.getString("DbmsAverageImplementation.NegativeCount"), new String[] { Integer.toString(count) })); //$NON-NLS-1$
		}
		ByteBuffer byteBuffer = ByteBuffer.allocate(PHYSICALENCODINGSIZE);
		byteBuffer.putInt(count).putDouble(value);
		return new ScalarValueBuffer(byteBuffer);
	}

	/**
	 * Gets A ValueBuffer holding the given value
	 * 
	 * @param count
	 * @param value
	 * @return A ValueBuffer holding the given value
	 */
	public static ValueBuffer getAverageValueBufferWithoutException (int count, double value) {
		if (count < 0) {
			throw new IllegalArgumentException(MyMessageFormat.format(Messages.getString("DbmsAverageImplementation.NegativeCount"), new String[] { Integer.toString(count) })); //$NON-NLS-1$
		}
		ByteBuffer byteBuffer = ByteBuffer.allocate(PHYSICALENCODINGSIZE);
		byteBuffer.putInt(count).putDouble(value);
		return new ScalarValueBuffer(byteBuffer);
	}

	/**
	 * Gets instance
	 * 
	 * @return instance.
	 */
	public static final DbmsAverageImplementation getInstance ( ) {
		return instance;
	}

	/**
	 * The components of the default possrep (=count value)
	 */
	private final Map<String, String> componentNameMap = initComponentNameMap();

	/**
	 * Map of the possreps
	 */
	private final Set<PossRepImplementation> possrepImplementations = new MyReadOnlySet<PossRepImplementation>(new HashSet<PossRepImplementation>(Arrays.asList(new PossRepImplementation[] { this })));

	/**
	 * 
	 */
	private final Map<String, PossRepImplementation> possrepImplementationsPerComponent = initPossrepImplementationsPerComponent();

	/**
	 * Builds the DbmsAverageImplementation object
	 */
	private DbmsAverageImplementation ( ) {

	}

	/**
	 * @return
	 */
	private MyReadOnlyMap<String, String> initComponentNameMap ( ) {
		HashMap<String, String> w = new HashMap<String, String>();
		w.put(POSSREPCOMPONENTNAMES.COUNT, TYPENAMES.INT);
		w.put(POSSREPCOMPONENTNAMES.VALUE, TYPENAMES.FLOAT);
		return new MyReadOnlyMap<String, String>(w);
	}

	/**
	 * @return
	 */
	private Map<String, PossRepImplementation> initPossrepImplementationsPerComponent ( ) {
		HashMap<String, PossRepImplementation> w = new HashMap<String, PossRepImplementation>();
		for (PossRepImplementation possrepImplementation : possrepImplementations) {
			for (String componentName : possrepImplementation.getComponentNameMap().keySet()) {
				w.put(componentName, possrepImplementation);
			}
		}
		return new MyReadOnlyMap<String, PossRepImplementation>(w);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.TypeImplementations.PossRepImplementation#getComponentNameMap()
	 */
	public Map<String, String> getComponentNameMap ( ) {
		return componentNameMap;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.PossRepImplementation#extractComponentValue(java.lang.String, be.erwinsmout.SIRA_PRISE.typeimplementations.ScalarValueBuffer)
	 */
	public ValueBuffer getComponentValue (String componentName, ScalarValueBuffer valueBuffer) {
		ByteBuffer byteBuffer = valueBuffer.getByteBuffer();
		if (componentName.equalsIgnoreCase(POSSREPCOMPONENTNAMES.COUNT)) {
			return DbmsIntImplementation.getValueBuffer(byteBuffer.getInt());
		}
		if (componentName.equalsIgnoreCase(POSSREPCOMPONENTNAMES.VALUE)) {
			return DbmsFloatImplementation.getValueBuffer(((ByteBuffer) byteBuffer.position(4)).getDouble());
		}
		throw new IllegalArgumentException(componentName);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.TypeImplementation#getDefaultPossrepImplementation()
	 */
	public PossRepImplementation getDefaultPossrepImplementation ( ) {
		return this;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.DBMS.TypeImplementations.TypeImplementation#getDefaultSize()
	 */
	public int getDfltLogicalSize ( ) {
		return 1;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.DBMS.TypeImplementations.TypeImplementation#getMaxSize()
	 */
	public int getMaxLogicalSize ( ) {
		return 1;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.DBMS.TypeImplementations.TypeImplementation#getMinSize()
	 */
	public int getMinLogicalSize ( ) {
		return 1;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.TypeImplementation#getPhysicalSizeFor(int)
	 */
	public int getPhysicalSizeFor (int logicalSize) {
		return PHYSICALENCODINGSIZE;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.SIRAPRISE.typeimplementations.TypeImplementation#getPossrepImplementation(java.lang.String)
	 */
	@Override
	public PossRepImplementation getPossrepImplementation (String componentName) {
		PossRepImplementation possRepImplementation = possrepImplementationsPerComponent.get(componentName);
		if (possRepImplementation == null) {
			throw new IllegalArgumentException(MyMessageFormat.format(Messages.getString("IntervalTypeImplementation.UnknownCompoenent"), new String[] { componentName, this.getClass().getName() })); //$NON-NLS-1$
		}
		return possRepImplementation;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.TypeImplementation#getPossRepImplementation()
	 */
	public Set<PossRepImplementation> getPossrepImplementations ( ) {
		return possrepImplementations;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.PossRepImplementation#getName()
	 */
	public String getPossrepName ( ) {
		return POSSREPNAMES.AVERAGE;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.PossRepImplementation#valueFromComponentValues(java.util.HashMap, int)
	 */
	public ValueBuffer valueFromComponentValues (HashMap<String, ValueBuffer> componentValueMap, int logicalSize) {
		int count = DbmsIntImplementation.getJavaInt((ScalarValueBuffer) componentValueMap.get(POSSREPCOMPONENTNAMES.COUNT));
		double value = DbmsFloatImplementation.getJavaDouble(componentValueMap.get(POSSREPCOMPONENTNAMES.VALUE));
		return getAverageValueBuffer(count, value);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.DBMS.TypeImplementations.TypeImplementation#valueFromExternalRepresentation(java.lang.String)
	 */
	public ValueBuffer valueFromExternalRepresentation (String value) throws InvalidValueException {
		Map<String, String> componentValuesMap;
		try {
			componentValuesMap = BracketParser.createMapFromEscapedString(value, false);
		} catch (NoClosingBracketException e) {
			throw new InvalidValueException(Messages.getString("DbmsAverageImplementation.Syntax"), e); //$NON-NLS-1$
		} catch (InvalidEscapedCharacterException e) {
			throw new InvalidValueException(Messages.getString("DbmsAverageImplementation.Syntax"), e); //$NON-NLS-1$
		} catch (MissingEscapedCharacterException e) {
			throw new InvalidValueException(Messages.getString("DbmsAverageImplementation.Syntax"), e); //$NON-NLS-1$
		} catch (NoOpeningBracketException e) {
			throw new InvalidValueException(Messages.getString("DbmsAverageImplementation.Syntax"), e); //$NON-NLS-1$
		} catch (DuplicateNameException e) {
			throw new InvalidValueException(Messages.getString("DbmsAverageImplementation.Syntax"), e); //$NON-NLS-1$
		}
		String count_tx = componentValuesMap.get(POSSREPCOMPONENTNAMES.COUNT);
		String value_tx = componentValuesMap.get(POSSREPCOMPONENTNAMES.VALUE);
		if (count_tx == null || value_tx == null) {
			throw new InvalidValueException(MyMessageFormat.format(Messages.getString("DbmsAverageImplementation.ComponentMissing"), new String[] { value })); //$NON-NLS-1$
		}
		try {
			return getAverageValueBuffer(Integer.parseInt(count_tx), Double.parseDouble(value_tx));
		} catch (NumberFormatException e) {
			throw new InvalidValueException(MyMessageFormat.format(Messages.getString("DbmsAverageImplementation.NumericDigits"), new String[] { value })); //$NON-NLS-1$
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.DBMS.TypeImplementations.TypeImplementation#valueFromExternalRepresentation(java.lang.String, int)
	 */
	public ValueBuffer valueFromExternalRepresentation (String value, int maximumLogicalLengthAllowed) throws InvalidValueException {
		return valueFromExternalRepresentation(value);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.PossRepImplementation#valueToEscapedExternalRepresentation(be.erwinsmout.SIRA_PRISE.typeimplementations.ValueBuffer)
	 */
	public String valueToEscapedExternalRepresentation (ValueBuffer valueBuffer) {
		return valueToExternalRepresentation(valueBuffer);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.DBMS.TypeImplementations.TypeImplementation#valueToExternalRepresentation(java.nio.ByteBuffer, int, int, int)
	 */
	public String valueToExternalRepresentation (ValueBuffer valueBuffer) {
		ByteBuffer byteBuffer = ((ScalarValueBuffer) valueBuffer).getByteBuffer();
		int count = byteBuffer.getInt();
		double value = byteBuffer.getDouble();
		return POSSREPCOMPONENTNAMES.COUNT + "(" + Integer.toString(count) + ")" + POSSREPCOMPONENTNAMES.VALUE + "(" + Double.toString(value) + ")"; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$ 
	}
}