/*
 * Created on 19-apr-04
 */
package be.SIRAPRISE.typeimplementations;

import java.nio.ByteBuffer;
import java.util.*;

import be.SIRAPRISE.client.NAMES.POSSREPNAMES;
import be.SIRAPRISE.client.NAMES.TYPENAMES;
import be.SIRAPRISE.util.BracketParser;
import be.SIRAPRISE.util.InvalidEscapedCharacterException;
import be.SIRAPRISE.util.MissingEscapedCharacterException;
import be.SIRAPRISE.util.MyReadOnlyMap;
import be.SIRAPRISE.util.MyReadOnlySet;
import be.erwinsmout.MyMessageFormat;

/**
 * The implementation for the binary bucket type. The physical encoding for values of this type is as follows :
 * <ul>
 * <li>Being a variable-length type, the first four bytes of a BITS value hold the logical length of the value, in this case the number of raw bytes.</li>
 * <li>All the following bytes are the actual bytes of the value.</li>
 * </ul>
 * 
 * The BITS type is destined for storing "raw" or "binary" content, such as, e.g., image or audio data. Using this type requires some explanation with respect to the external representation of values of this type. The actual content of values of this type is considered by SIRA_PRISE as "just any sequence of bytes". These bytes may include values (in particular binary zero) that may make those "sequences of bytes" a bit more difficult to "transport" through certain other software platforms that communicate with SIRA_PRISE (e.g. in systems that consider a binary-zero byte as an end-of-string marker). SIRA_PRISE therefore externalizes
 * values of type BITS using a mechanism similar to uu-encoding and base64-encoding :
 * <ul>
 * <li>First of all, the sequence of bytes is subdivided in groups of three bytes. One or two additional binary-zero bytes may need to be added to complete the last group.</li>
 * <li>Then, for each group of three bytes, the 24 bits are rearranged to form 4 consecutive 6-bit values. The value 32 is added to each of these four values (yielding a value in the range [32-95]). Using these four values to select characters from the unicode character set, then yields four character tokens uniquely representing the original three bytes.</li>
 * <li>All 4-character groups are concatenated together, and the character '0', '1' or '2' is prepended to identify the number of binary-zero bytes that were padded to complete the last group.</li>
 * </ul>
 * 
 * To clarify using as an example the byte string consisting of three blanks (hex 20) :
 * 
 * <ul>
 * <li>202020</li>
 * <li>(rearrange in 6-bit groups) ===> 001000 000010 000000 100000</li>
 * <li>(add 32 to each) ===> 011000 010010 010000 110000</li>
 * <li>(select corresponding Unicode character) ===> 8 2 0 P</li>
 * <li>(Prepend number of binary-zero bytes used for padding) ===> 0820P</li>
 * </ul>
 * 
 * When specifying values of this type to SIRA_PRISE, SIRA_PRISE expects to be presented a token string that is built in exactly the same way, i.e. in order to have SIRA_PRISE record a BITS value consisting of three blanks, the value selector BITS(0820P) must be specified. The encoded value in this case will be 7 bytes long, the first four holding the integer 3, then next three holding the unicode character 0x0020.
 * 
 * @author Erwin Smout
 */
public final class DbmsBitsImplementation implements TypeImplementation, PossRepImplementation {

	/**
	 * The instance
	 */
	private static final DbmsBitsImplementation instance = new DbmsBitsImplementation();

	/**
	 * The maximum logical size of a sira_prise bit bucket. Its UU representation must fit within the maximal logical size of the type used for that UU representation (=String). The length of the UU representation is 1 + 4 * logicallength / 3,
	 */
	public static final int MAXLOGICALSIZE = ((DbmsStringImplementation.MAXLOGICALSIZE - 1) / 4) * 3; // Important to do the division first to avoid overflows.

	/**
	 * The minimum logical size of a sira_prise string
	 */
	public static final int MINLOGICALSIZE = 0;

	/**
	 * @param value
	 * @param logicalLength
	 * @return A ValueBuffer holding the given string
	 */
	private static ScalarValueBuffer getValueBufferWithoutLogicalSizeChecks (String value, int logicalLength) {
		int bufferLength = physicalSizeFor(logicalLength);
		ByteBuffer buffer = ByteBuffer.allocate(bufferLength);
		buffer.putInt(logicalLength);

		int i = 1;
		int uuBytesCount = value.length() - 1;
		while (i <= uuBytesCount) {
			int uub1 = value.charAt(i++);
			int uub2 = value.charAt(i++);
			int uub3 = value.charAt(i++);
			int uub4 = value.charAt(i++);

			if (uub1 < 32 || uub1 >= 96) {
				throw new IllegalArgumentException(MyMessageFormat.format(Messages.getString("DbmsBitsImplementation.InvalidToken"), new String[] { Byte.toString((byte) uub1) })); //$NON-NLS-1$
			}
			if (uub2 < 32 || uub2 >= 96) {
				throw new IllegalArgumentException(MyMessageFormat.format(Messages.getString("DbmsBitsImplementation.InvalidToken"), new String[] { Byte.toString((byte) uub2) })); //$NON-NLS-1$
			}
			if (uub3 < 32 || uub3 >= 96) {
				throw new IllegalArgumentException(MyMessageFormat.format(Messages.getString("DbmsBitsImplementation.InvalidToken"), new String[] { Byte.toString((byte) uub3) })); //$NON-NLS-1$
			}
			if (uub4 < 32 || uub4 >= 96) {
				throw new IllegalArgumentException(MyMessageFormat.format(Messages.getString("DbmsBitsImplementation.InvalidToken"), new String[] { Byte.toString((byte) uub4) })); //$NON-NLS-1$
			}

			uub1 -= 32;
			uub2 -= 32;
			uub3 -= 32;
			uub4 -= 32;

			int b123 = uub4 | ((uub3 | ((uub2 | (uub1 << 6)) << 6)) << 6);

			byte b1 = (byte) ((b123 & 0x00FF0000) >> 16);
			byte b2 = (byte) ((b123 & 0x0000FF00) >> 8);
			byte b3 = (byte) ((b123 & 0x000000FF));

			buffer.put(b1);
			if (buffer.remaining() > 0) {
				buffer.put(b2);
				if (buffer.remaining() > 0) {
					buffer.put(b3);
				} else {
					if (b3 != 0) {
						throw new IllegalArgumentException(MyMessageFormat.format(Messages.getString("DbmsBitsImplementation.NonZeroPadByte"), new String[] { Byte.toString(b3), value })); //$NON-NLS-1$
					}
				}
			} else {
				if (b2 != 0) {
					throw new IllegalArgumentException(MyMessageFormat.format(Messages.getString("DbmsBitsImplementation.NonZeroPadByte"), new String[] { Byte.toString(b2), value })); //$NON-NLS-1$
				}
				if (b3 != 0) {
					throw new IllegalArgumentException(MyMessageFormat.format(Messages.getString("DbmsBitsImplementation.NonZeroPadByte"), new String[] { Byte.toString(b3), value })); //$NON-NLS-1$
				}
			}
		}
		return new ScalarValueBuffer(buffer);
	}

	/**
	 * Retrieves the logical length (= the actual number of bytes) of a bits value. This takes into account the padding indicator at the beginning of the value.
	 * 
	 * @param uuEncodedValue
	 *            The String representation of a bits value, i.e. a UU-encoded string value
	 * @return The logical length of the bits value (= the actual number of bytes)
	 */
	private static int logicalLengthFrom (String uuEncodedValue) {
		if ((uuEncodedValue.length() & 0x00000003) != 1) {
			throw new IllegalArgumentException(Messages.getString("DbmsBitsImplementation.MultipleOfFour")); //$NON-NLS-1$
		}
		int uuBytesCount = uuEncodedValue.length() - 1;
		String paddingIndicator = uuEncodedValue.substring(0, 1);
		int paddedBytesCount;
		try {
			paddedBytesCount = Integer.parseInt(paddingIndicator);
		} catch (NumberFormatException e) {
			throw new IllegalArgumentException(MyMessageFormat.format(Messages.getString("DbmsBitsImplementation.PaddingIndicator"), new String[] { paddingIndicator, uuEncodedValue })); //$NON-NLS-1$
		}
		if (paddedBytesCount > Math.min(2, uuBytesCount)) {
			throw new IllegalArgumentException(MyMessageFormat.format(Messages.getString("DbmsBitsImplementation.PaddingIndicator"), new String[] { paddingIndicator, uuEncodedValue })); //$NON-NLS-1$
		}

		int logicalLength = uuBytesCount / 4 * 3 - paddedBytesCount;
		return logicalLength;
	}

	/**
	 * Gets the number of bytes needed to encode this string in sira_prise
	 * 
	 * @param logicalSize
	 *            The number of bytes in the bucket
	 * @return The number of bytes needed to encode this string in sira_prise
	 */
	private static int physicalSizeFor (int logicalSize) {
		return 4 + logicalSize;
	}

	/**
	 * Gets a java String holding the SIRA_PRISE UU String from a SIRA_PRISE scalarvaluebuffer holding a SIRA_PRISE BITS value
	 * 
	 * @param valueBuffer
	 *            The ScalarValueBuffer containing a SIRA_PRISE String value
	 * @return The String contained in the given ScalarValueBuffer
	 */
	static String getUUString (ScalarValueBuffer valueBuffer) {
		ByteBuffer buffer = valueBuffer.getByteBuffer();
		int byteLength = buffer.getInt();
		if (byteLength == 0) {
			return "0"; //$NON-NLS-1$
		}
		// Each 3 bytes produce 4 UU bytes : 3*8 = 24 bits = 4 6-bit values
		int uuLength = ((byteLength - 1) / 3 + 1) * 4;
		byte[] uuBytes = new byte[uuLength];

		int pad = 0;
		int i = 0;
		while (buffer.remaining() > 0) {
			byte b1 = buffer.get();
			byte b2;
			byte b3;
			if (buffer.remaining() > 0) {
				b2 = buffer.get();
				if (buffer.remaining() > 0) {
					b3 = buffer.get();
					pad = 0;
				} else {
					b3 = 0;
					pad = 1;
				}
			} else {
				b2 = 0;
				b3 = 0;
				pad = 2;
			}

			int b123 = (((b1 << 8) | b2) << 8) | b3;

			byte uu4 = (byte) ((b123 & 0x0000003f) + 32);
			b123 >>= 6;
			byte uu3 = (byte) ((b123 & 0x0000003f) + 32);
			b123 >>= 6;
			byte uu2 = (byte) ((b123 & 0x0000003f) + 32);
			b123 >>= 6;
			byte uu1 = (byte) ((b123 & 0x0000003f) + 32);
			b123 >>= 6;

			uuBytes[i++] = uu1;
			uuBytes[i++] = uu2;
			uuBytes[i++] = uu3;
			uuBytes[i++] = uu4;
		}

		return Integer.toString(pad) + new String(uuBytes);
	}

	/**
	 * Gets a SIRA_PRISE bitbucket valuebuffer from a java String. The java string is a sort of uuencoded value :
	 * 
	 * @param value
	 *            the java String
	 * @return A ValueBuffer holding the given string
	 */
	static ScalarValueBuffer getValueBuffer (String value) {
		return getValueBufferWithoutLogicalSizeChecks(value, logicalLengthFrom(value));
	}

	/**
	 * Gets a SIRA_PRISE scalarvaluebuffer from a java String
	 * 
	 * @param value
	 *            the java String
	 * @param maximumLogicalLengthAllowed
	 * @return A ValueBuffer holding the given string
	 */
	static ScalarValueBuffer getValueBuffer (String value, int maximumLogicalLengthAllowed) {
		int logicalLength = logicalLengthFrom(value);
		if (logicalLength > maximumLogicalLengthAllowed) {
			throw new IllegalArgumentException(MyMessageFormat.format(Messages.getString("DbmsBitsImplementation.ExceedsBufferSize"), new String[] { Integer.toString(logicalLength), Integer.toString(maximumLogicalLengthAllowed), value })); //$NON-NLS-1$
		}
		return getValueBufferWithoutLogicalSizeChecks(value, logicalLength);
	}

	/**
	 * Gets instance
	 * 
	 * @return instance.
	 */
	public static final DbmsBitsImplementation getInstance ( ) {
		return instance;
	}

	/**
	 * Gets a java String holding the SIRA_PRISE UU String from a given byte array
	 * 
	 * @param bytes
	 *            The input byte array
	 * @return The UU String representation for the byte sequence given
	 */
	public static String getUUString (byte[] bytes) {
		int byteLength = bytes.length;
		if (byteLength == 0) {
			return "0"; //$NON-NLS-1$
		}
		// Each 3 bytes produce 4 UU bytes : 3*8 = 24 bits = 4 6-bit values
		int uuLength = ((byteLength - 1) / 3 + 1) * 4;
		byte[] uuBytes = new byte[uuLength];

		int pad = 0;
		int i_bytes = 0;
		int i_uuBytes = 0;
		while (i_bytes < byteLength) {
			byte b1 = bytes[i_bytes++];
			byte b2;
			byte b3;
			if (i_bytes < byteLength) {
				b2 = bytes[i_bytes++];
				if (i_bytes < byteLength) {
					b3 = bytes[i_bytes++];
					pad = 0;
				} else {
					b3 = 0;
					pad = 1;
				}
			} else {
				b2 = 0;
				b3 = 0;
				pad = 2;
			}

			int b123 = ((((b1 & 0x000000ff) << 8) | (b2 & 0x000000ff)) << 8) | (b3 & 0x000000ff);

			byte uu4 = (byte) ((b123 & 0x0000003f) + 32);
			b123 >>= 6;
			byte uu3 = (byte) ((b123 & 0x0000003f) + 32);
			b123 >>= 6;
			byte uu2 = (byte) ((b123 & 0x0000003f) + 32);
			b123 >>= 6;
			byte uu1 = (byte) ((b123 & 0x0000003f) + 32);
			b123 >>= 6;

			uuBytes[i_uuBytes++] = uu1;
			uuBytes[i_uuBytes++] = uu2;
			uuBytes[i_uuBytes++] = uu3;
			uuBytes[i_uuBytes++] = uu4;
		}

		return Integer.toString(pad) + new String(uuBytes);
	}

	/**
	 * The possrep component map
	 */
	private final Map<String, String> componentNameMap = initComponentNameMap();

	/**
	 * The possreps
	 */
	private final Set<PossRepImplementation> possrepImplementations = new MyReadOnlySet<PossRepImplementation>(new HashSet<PossRepImplementation>(Arrays.asList(new PossRepImplementation[] { this })));

	/**
	 * 
	 */
	private final Map<String, PossRepImplementation> possrepImplementationsPerComponent = initPossrepImplementationsPerComponent();

	/**
	 * 
	 */
	private DbmsBitsImplementation ( ) {

	}

	/**
	 * @return
	 */
	private MyReadOnlyMap<String, String> initComponentNameMap ( ) {
		HashMap<String, String> w = new HashMap<String, String>();
		// "conceptual" components in this possrep : STRING(STRING)
		w.put(TYPENAMES.STRING, TYPENAMES.STRING);
		return new MyReadOnlyMap<String, String>(w);
	}

	/**
	 * @return
	 */
	private Map<String, PossRepImplementation> initPossrepImplementationsPerComponent ( ) {
		HashMap<String, PossRepImplementation> w = new HashMap<String, PossRepImplementation>();
		for (PossRepImplementation possrepImplementation : possrepImplementations) {
			for (String componentName : possrepImplementation.getComponentNameMap().keySet()) {
				w.put(componentName, possrepImplementation);
			}
		}
		return new MyReadOnlyMap<String, PossRepImplementation>(w);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.TypeImplementations.PossRepImplementation#getComponentNameMap()
	 */
	public Map<String, String> getComponentNameMap ( ) {
		return componentNameMap;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.PossRepImplementation#extractComponentValue(java.lang.String, be.erwinsmout.SIRA_PRISE.typeimplementations.ScalarValueBuffer)
	 */
	public ValueBuffer getComponentValue (String componentName, ScalarValueBuffer valueBuffer) {
		try {
			return DbmsStringImplementation.getValueBuffer(getUUString(valueBuffer));
		} catch (InvalidValueException e) {
			throw new RuntimeException(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.TypeImplementation#getDefaultPossrepImplementation()
	 */
	public PossRepImplementation getDefaultPossrepImplementation ( ) {
		return this;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.DBMS.TypeImplementations.TypeImplementation#getDefaultSize()
	 */
	public int getDfltLogicalSize ( ) {
		return 2048;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.DBMS.TypeImplementations.TypeImplementation#getMaxSize()
	 */
	public int getMaxLogicalSize ( ) {
		return MAXLOGICALSIZE;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.DBMS.TypeImplementations.TypeImplementation#getMinSize()
	 */
	public int getMinLogicalSize ( ) {
		return MINLOGICALSIZE;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.TypeImplementation#getPhysicalSizeFor(int)
	 */
	public int getPhysicalSizeFor (int logicalSize) {
		return physicalSizeFor(logicalSize);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.SIRAPRISE.typeimplementations.TypeImplementation#getPossrepImplementation(java.lang.String)
	 */
	@Override
	public PossRepImplementation getPossrepImplementation (String componentName) {
		PossRepImplementation possRepImplementation = possrepImplementationsPerComponent.get(componentName);
		if (possRepImplementation == null) {
			throw new IllegalArgumentException(MyMessageFormat.format(Messages.getString("IntervalTypeImplementation.UnknownCompoenent"), new String[] { componentName, this.getClass().getName() })); //$NON-NLS-1$
		}
		return possRepImplementation;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.TypeImplementation#getPossRepImplementation()
	 */
	public Set<PossRepImplementation> getPossrepImplementations ( ) {
		return possrepImplementations;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.PossRepImplementation#getName()
	 */
	public String getPossrepName ( ) {
		return POSSREPNAMES.BITS;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.PossRepImplementation#valueFromComponentValues(java.util.HashMap, int)
	 */
	public ValueBuffer valueFromComponentValues (HashMap<String, ValueBuffer> componentValueMap, int logicalSize) {
		String value = DbmsStringImplementation.getJavaString((ScalarValueBuffer) componentValueMap.get(TYPENAMES.STRING));
		// return getValueBuffer(BracketParser.unMeta(value), logicalSize); unMeta should already have been done in the StringImplementation
		return getValueBuffer(value, logicalSize);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.DBMS.TypeImplementations.TypeImplementation#valueFromExternalRepresentation(java.lang.String)
	 */
	public ValueBuffer valueFromExternalRepresentation (String value) throws InvalidValueException {
		try {
			return getValueBuffer(BracketParser.unMeta(value));
		} catch (InvalidEscapedCharacterException e) {
			throw new InvalidValueException(MyMessageFormat.format(Messages.getString("DbmsStringImplementation.InvalidEscape"), new String[] { value }), e); //$NON-NLS-1$
		} catch (MissingEscapedCharacterException e) {
			throw new InvalidValueException(MyMessageFormat.format(Messages.getString("DbmsStringImplementation.InvalidEscape"), new String[] { value }), e); //$NON-NLS-1$
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.DBMS.Server.TypeImplementation#fromExternal(java.lang.String, int)
	 */
	public ValueBuffer valueFromExternalRepresentation (String value, int maximumLogicalLengthAllowed) throws InvalidValueException {
		try {
			return getValueBuffer(BracketParser.unMeta(value), maximumLogicalLengthAllowed);
		} catch (InvalidEscapedCharacterException e) {
			throw new InvalidValueException(MyMessageFormat.format(Messages.getString("DbmsStringImplementation.InvalidEscape"), new String[] { value }), e); //$NON-NLS-1$
		} catch (MissingEscapedCharacterException e) {
			throw new InvalidValueException(MyMessageFormat.format(Messages.getString("DbmsStringImplementation.InvalidEscape"), new String[] { value }), e); //$NON-NLS-1$
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.PossRepImplementation#valueToEscapedExternalRepresentation(be.erwinsmout.SIRA_PRISE.typeimplementations.ValueBuffer)
	 */
	public String valueToEscapedExternalRepresentation (ValueBuffer valueBuffer) {
		String s = getUUString((ScalarValueBuffer) valueBuffer);
		return BracketParser.meta(s);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.DBMS.Server.TypeImplementation#toExternal(java.nio.ByteBuffer, int, int, int)
	 */
	public String valueToExternalRepresentation (ValueBuffer valueBuffer) {
		return getUUString((ScalarValueBuffer) valueBuffer);
	}
}