/*
 * Created on 7-jun-04
 */
package be.SIRAPRISE.typeimplementations;

import java.nio.ByteBuffer;
import java.util.*;

import be.SIRAPRISE.client.NAMES.POSSREPCOMPONENTNAMES;
import be.SIRAPRISE.client.NAMES.TYPENAMES;
import be.SIRAPRISE.util.MyReadOnlyMap;
import be.SIRAPRISE.util.MyReadOnlySet;
import be.erwinsmout.MyMessageFormat;

/**
 * The implementation for the system-defined Boolean type. The physical encoding of boolean values is a 4-byte integer that holds the value -1 for true, or the integer value 0 for false.
 * 
 * @author Erwin Smout
 */
public final class DbmsBooleanImplementation implements TypeImplementation, PossRepImplementation {

	/**
	 * The implementation for the YESNO possrep of type boolean
	 * 
	 * @author Erwin Smout
	 */
	final class YN implements PossRepImplementation {

		/**
		 * The components in this possrep
		 */
		private final Map<String, String> ynComponentNameMap = initYNComponentNameMap();

		/**
		 * 
		 */
		YN ( ) {

		}

		/**
		 * @return
		 */
		private MyReadOnlyMap<String, String> initYNComponentNameMap ( ) {
			HashMap<String, String> w = new HashMap<String, String>();
			w.put(POSSREPCOMPONENTNAMES.YN, TYPENAMES.STRING);
			return new MyReadOnlyMap<String, String>(w);
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see be.erwinsmout.SIRA_PRISE.TypeImplementations.PossRepImplementation#getComponentNameMap()
		 */
		public Map<String, String> getComponentNameMap ( ) {
			return ynComponentNameMap;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.PossRepImplementation#extractComponentValue(java.lang.String, be.erwinsmout.SIRA_PRISE.typeimplementations.ScalarValueBuffer)
		 */
		public ValueBuffer getComponentValue (String componentName, ScalarValueBuffer valueBuffer) {
			try {
				return DbmsStringImplementation.getValueBuffer(valueToExternalRepresentation(valueBuffer));
			} catch (InvalidValueException e) {
				throw new RuntimeException(e);
			}
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.PossRepImplementation#getName()
		 */
		public String getPossrepName ( ) {
			return POSSREPCOMPONENTNAMES.YN;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.PossRepImplementation#valueFromComponentValues(java.util.HashMap, int)
		 */
		public ValueBuffer valueFromComponentValues (HashMap<String, ValueBuffer> componentValueMap, int logicalSize) {
			return valueFromExternalRepresentation(DbmsStringImplementation.getJavaString((ScalarValueBuffer) componentValueMap.get(POSSREPCOMPONENTNAMES.YN)));
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see be.erwinsmout.DBMS.Server.PossRepImplementation#valueFromExternalRepresentation(java.lang.String)
		 */
		public ValueBuffer valueFromExternalRepresentation (String value) {
			ByteBuffer byteBuffer = ByteBuffer.allocate(4);
			if (value.toUpperCase().equalsIgnoreCase("Y")) { //$NON-NLS-1$
				byteBuffer.putInt(_TRUE);
			} else {
				if (value.toUpperCase().equalsIgnoreCase("N")) { //$NON-NLS-1$
					byteBuffer.putInt(_FALSE);
				} else {
					throw new IllegalArgumentException(MyMessageFormat.format(Messages.getString("DbmsBooleanImplementation.Value"), new String[] { value })); //$NON-NLS-1$
				}
			}
			return new ScalarValueBuffer(byteBuffer);
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see be.erwinsmout.DBMS.Server.PossRepImplementation#valueFromExternalRepresentation(java.lang.String, int)
		 */
		public ValueBuffer valueFromExternalRepresentation (String value, int maximumLogicalLengthAllowed) throws InvalidValueException {
			return valueFromExternalRepresentation(value);
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.PossRepImplementation#valueToEscapedExternalRepresentation(be.erwinsmout.SIRA_PRISE.typeimplementations.ValueBuffer)
		 */
		public String valueToEscapedExternalRepresentation (ValueBuffer valueBuffer) {
			return valueToExternalRepresentation(valueBuffer);
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see be.erwinsmout.DBMS.Server.PossRepImplementation#valueToExternalRepresentation(java.nio.ByteBuffer)
		 */
		public String valueToExternalRepresentation (ValueBuffer valueBuffer) {
			if (DbmsIntImplementation.getJavaInt(((ScalarValueBuffer) valueBuffer)) == _TRUE) {
				return "Y"; //$NON-NLS-1$
			} else {
				return "N"; //$NON-NLS-1$
			}
		}
	}

	/**
	 * 
	 */
	private static final DbmsBooleanImplementation instance = new DbmsBooleanImplementation();

	/**
	 * The numeric value for false
	 */
	public static final int _FALSE = 0;

	/**
	 * The numeric value for true
	 */
	public static final int _TRUE = -1;

	/**
	 * Value representing 'false'
	 */
	public static final ScalarValueBuffer falseValue = new ScalarValueBuffer(ByteBuffer.allocate(4).putInt(_FALSE));

	/**
	 * Value representing 'true'
	 */
	public static final ScalarValueBuffer trueValue = new ScalarValueBuffer(ByteBuffer.allocate(4).putInt(_TRUE));

	/**
	 * Gets instance
	 * 
	 * @return instance.
	 */
	public static DbmsBooleanImplementation getInstance ( ) {
		return instance;
	}

	/**
	 * Converts a ValueBuffer holding a boolean into a java boolean
	 * 
	 * @param buffer
	 *            The ValueBuffer holding a boolean value
	 * @return true if the bytebuffer represents the sira_prise value for true ; false otherwise
	 */
	public static boolean getJavaBoolean (ScalarValueBuffer buffer) {
		return (buffer.getByteBuffer().getInt() == _TRUE);
	}

	/**
	 * Converts a ValueBuffer holding a boolean into a java boolean representing the negation of that boolean
	 * 
	 * @param buffer
	 *            The ValueBuffer holding a boolean value
	 * @return true if the bytebuffer represents the sira_prise value for true ; false otherwise
	 */
	public static boolean getNegatedJavaBoolean (ScalarValueBuffer buffer) {
		return (buffer.getByteBuffer().getInt() == _FALSE);
	}

	/**
	 * Converts a ValueBuffer holding a boolean into a java boolean
	 * 
	 * @param buffer
	 *            The ValueBuffer holding a boolean value
	 * @return true if the bytebuffer represents the sira_prise value for true ; false otherwise
	 */
	public static ScalarValueBuffer getNegatedJavaBooleanBuffer (ScalarValueBuffer buffer) {
		return (buffer.getByteBuffer().getInt() == _FALSE) ? trueValue : falseValue;
	}

	/**
	 * The component names of the default possrep (none)
	 */
	private final Map<String, String> componentNameMap = initComponentNameMap();

	/**
	 * Map of the possrep names/implementation objects
	 */
	private final Set<PossRepImplementation> possrepImplementations = new MyReadOnlySet<PossRepImplementation>(new HashSet<PossRepImplementation>(Arrays.asList(new PossRepImplementation[] { new YN(), this })));

	/**
	 * 
	 */
	private final Map<String, PossRepImplementation> possrepImplementationsPerComponent = initPossrepImplementationsPerComponent();

	/**
	 * Builds a DbmsBooleanImplementation
	 */
	private DbmsBooleanImplementation ( ) {

	}

	/**
	 * @return
	 */
	private MyReadOnlyMap<String, String> initComponentNameMap ( ) {
		HashMap<String, String> w = new HashMap<String, String>();
		// "conceptual" component named STRING and type STRING
		w.put(TYPENAMES.STRING, TYPENAMES.STRING);
		return new MyReadOnlyMap<String, String>(w);
	}

	/**
	 * @return
	 */
	private MyReadOnlyMap<String, PossRepImplementation> initPossrepImplementationsPerComponent ( ) {
		HashMap<String, PossRepImplementation> w = new HashMap<String, PossRepImplementation>();
		for (PossRepImplementation possrepImplementation : possrepImplementations) {
			for (String componentName : possrepImplementation.getComponentNameMap().keySet()) {
				w.put(componentName, possrepImplementation);
			}
		}
		return new MyReadOnlyMap<String, PossRepImplementation>(w);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.TypeImplementations.PossRepImplementation#getComponentNameMap()
	 */
	public Map<String, String> getComponentNameMap ( ) {
		return componentNameMap;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.PossRepImplementation#extractComponentValue(java.lang.String, be.erwinsmout.SIRA_PRISE.typeimplementations.ScalarValueBuffer)
	 */
	public ValueBuffer getComponentValue (String componentName, ScalarValueBuffer valueBuffer) {
		try {
			return DbmsStringImplementation.getValueBuffer(valueToExternalRepresentation(valueBuffer));
		} catch (InvalidValueException e) {
			throw new RuntimeException(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.TypeImplementation#getDefaultPossrepImplementation()
	 */
	public PossRepImplementation getDefaultPossrepImplementation ( ) {
		return this;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.DBMS.TypeImplementations.TypeImplementation#getDefaultSize()
	 */
	public int getDfltLogicalSize ( ) {
		return 1;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.DBMS.TypeImplementations.TypeImplementation#getMaxSize()
	 */
	public int getMaxLogicalSize ( ) {
		return 1;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.DBMS.TypeImplementations.TypeImplementation#getMinSize()
	 */
	public int getMinLogicalSize ( ) {
		return 1;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.TypeImplementation#getPhysicalSizeFor(int)
	 */
	public int getPhysicalSizeFor (int logicalSize) {
		return 4;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.SIRAPRISE.typeimplementations.TypeImplementation#getPossrepImplementation(java.lang.String)
	 */
	@Override
	public PossRepImplementation getPossrepImplementation (String componentName) {
		PossRepImplementation possRepImplementation = possrepImplementationsPerComponent.get(componentName);
		if (possRepImplementation == null) {
			throw new IllegalArgumentException(MyMessageFormat.format(Messages.getString("IntervalTypeImplementation.UnknownCompoenent"), new String[] { componentName, this.getClass().getName() })); //$NON-NLS-1$
		}
		return possRepImplementation;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.TypeImplementation#getPossRepImplementation()
	 */
	public Set<PossRepImplementation> getPossrepImplementations ( ) {
		return possrepImplementations;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.PossRepImplementation#getName()
	 */
	public String getPossrepName ( ) {
		return TYPENAMES.BOOLEAN;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.PossRepImplementation#valueFromComponentValues(java.util.HashMap, int)
	 */
	public ValueBuffer valueFromComponentValues (HashMap<String, ValueBuffer> componentValueMap, int logicalSize) {
		String truefalse = DbmsStringImplementation.getJavaString((ScalarValueBuffer) componentValueMap.get(TYPENAMES.STRING));
		return valueFromExternalRepresentation(truefalse);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.DBMS.TypeImplementations.TypeImplementation#valueFromExternalRepresentation(java.lang.String)
	 */
	public ValueBuffer valueFromExternalRepresentation (String value) {
		ByteBuffer byteBuffer = ByteBuffer.allocate(4);
		if (value.toUpperCase().equalsIgnoreCase("TRUE")) { //$NON-NLS-1$
			byteBuffer.putInt(_TRUE);
		} else {
			if (value.toUpperCase().equalsIgnoreCase("FALSE")) { //$NON-NLS-1$
				byteBuffer.putInt(_FALSE);
			} else {
				throw new IllegalArgumentException(MyMessageFormat.format(Messages.getString("DbmsBooleanImplementation.BooleanTrueFalse"), new String[] { value })); //$NON-NLS-1$
			}
		}
		return new ScalarValueBuffer(byteBuffer);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.DBMS.TypeImplementations.TypeImplementation#valueFromExternalRepresentation(java.lang.String, int)
	 */
	public ValueBuffer valueFromExternalRepresentation (String value, int maximumLogicalLengthAllowed) throws InvalidValueException {
		return valueFromExternalRepresentation(value);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.PossRepImplementation#valueToEscapedExternalRepresentation(be.erwinsmout.SIRA_PRISE.typeimplementations.ValueBuffer)
	 */
	public String valueToEscapedExternalRepresentation (ValueBuffer valueBuffer) {
		return valueToExternalRepresentation(valueBuffer);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.DBMS.TypeImplementations.TypeImplementation#valueToExternalRepresentation(java.nio.ByteBuffer, int, int, int)
	 */
	public String valueToExternalRepresentation (ValueBuffer valueBuffer) {
		if (DbmsIntImplementation.getJavaInt(((ScalarValueBuffer) valueBuffer)) == _TRUE) {
			return "True"; //$NON-NLS-1$
		} else {
			return "False"; //$NON-NLS-1$
		}
	}
}
