/*
 * Created on 19-apr-04
 */
package be.SIRAPRISE.typeimplementations;

import java.nio.ByteBuffer;
import java.util.*;

import be.SIRAPRISE.client.NAMES.POSSREPCOMPONENTNAMES;
import be.SIRAPRISE.client.NAMES.POSSREPNAMES;
import be.SIRAPRISE.client.NAMES.TYPENAMES;
import be.SIRAPRISE.util.BracketParser;
import be.SIRAPRISE.util.InvalidEscapedCharacterException;
import be.SIRAPRISE.util.MissingEscapedCharacterException;
import be.SIRAPRISE.util.MyReadOnlyMap;
import be.SIRAPRISE.util.MyReadOnlySet;
import be.erwinsmout.MyMessageFormat;

/**
 * The implementation for the system-defined CHAR type. The physical encoding of the values consists of two bytes holding the numeric offset ([0-65535]) of the character in the java character set. This corresponds mostly (if not always) to the offset of the character in the BMP of the Unicode character set.
 * 
 * @author Erwin Smout
 */
public final class DbmsCharImplementation implements TypeImplementation, PossRepImplementation {

	/**
	 * @author Erwin
	 * 
	 */
	private static class UniCode implements PossRepImplementation {

		/**
		 * 
		 */
		private Map<String, String> unicodeComponentNameMap = initUnicodeComponentNameMap();

		/**
		 * Creates the UniCode
		 * 
		 */
		UniCode ( ) {

		}

		/**
		 * @return
		 */
		private MyReadOnlyMap<String, String> initUnicodeComponentNameMap ( ) {
			HashMap<String, String> w = new HashMap<String, String>();
			w.put(POSSREPCOMPONENTNAMES.UNICODE, TYPENAMES.INT);
			return new MyReadOnlyMap<String, String>(w);
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see be.SIRAPRISE.typeimplementations.PossRepImplementation#getComponentNameMap()
		 */
		public Map<String, String> getComponentNameMap ( ) {
			return unicodeComponentNameMap;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see be.SIRAPRISE.typeimplementations.PossRepImplementation#getComponentValue(java.lang.String, be.SIRAPRISE.typeimplementations.ScalarValueBuffer)
		 */
		public ValueBuffer getComponentValue (String componentName, ScalarValueBuffer valueBuffer) {
			return DbmsIntImplementation.getValueBuffer(getJavaChar(valueBuffer));
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see be.SIRAPRISE.typeimplementations.PossRepImplementation#getPossrepName()
		 */
		public String getPossrepName ( ) {
			return POSSREPNAMES.UNICODE;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see be.SIRAPRISE.typeimplementations.PossRepImplementation#valueFromComponentValues(java.util.HashMap, int)
		 */
		public ValueBuffer valueFromComponentValues (HashMap<String, ValueBuffer> componentValueMap, int logicalSize) {
			int uniCodeNumber = DbmsIntImplementation.getJavaInt((ScalarValueBuffer) componentValueMap.get(POSSREPCOMPONENTNAMES.UNICODE));
			if (uniCodeNumber < Character.MIN_VALUE || uniCodeNumber > Character.MAX_VALUE) {
				throw new IllegalArgumentException(MyMessageFormat.format(Messages.getString("DbmsCharImplementation.InvalidUniCodeNumber"), new String[] { Integer.toString(uniCodeNumber) })); //$NON-NLS-1$
			}
			return DbmsCharImplementation.getValueBuffer((char) uniCodeNumber);
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see be.SIRAPRISE.typeimplementations.PossRepImplementation#valueFromExternalRepresentation(java.lang.String)
		 */
		public ValueBuffer valueFromExternalRepresentation (String value) throws InvalidValueException {
			int uniCodeNumber = DbmsIntImplementation.getIntegerFromString(value);
			if (uniCodeNumber < Character.MIN_VALUE || uniCodeNumber > Character.MAX_VALUE) {
				throw new InvalidValueException(MyMessageFormat.format(Messages.getString("DbmsCharImplementation.InvalidUniCodeNumber"), new String[] { Integer.toString(uniCodeNumber) })); //$NON-NLS-1$
			}
			return DbmsCharImplementation.getValueBuffer((char) uniCodeNumber);
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see be.SIRAPRISE.typeimplementations.PossRepImplementation#valueFromExternalRepresentation(java.lang.String, int)
		 */
		public ValueBuffer valueFromExternalRepresentation (String value, int maximumLogicalLengthAllowed) throws InvalidValueException {
			return valueFromExternalRepresentation(value);
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see be.SIRAPRISE.typeimplementations.PossRepImplementation#valueToEscapedExternalRepresentation(be.SIRAPRISE.typeimplementations.ValueBuffer)
		 */
		public String valueToEscapedExternalRepresentation (ValueBuffer valueBuffer) {
			return valueToExternalRepresentation(valueBuffer);
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see be.SIRAPRISE.typeimplementations.PossRepImplementation#valueToExternalRepresentation(be.SIRAPRISE.typeimplementations.ValueBuffer)
		 */
		public String valueToExternalRepresentation (ValueBuffer valueBuffer) {
			return Integer.toString(getJavaChar((ScalarValueBuffer) valueBuffer));
		}

	}

	/**
	 * the instance.
	 */
	private static DbmsCharImplementation instance = new DbmsCharImplementation();

	/**
	 * The minimum, maximum & default logical size of a sira_prise char
	 */
	private static final int LOGICALSIZE = 1;

	/**
	 * 
	 */
	private static final int PHYSICALSIZE = 2;

	/**
	 * Gets a java char from a SIRA_PRISE scalarvaluebuffer
	 * 
	 * @param valueBuffer
	 *            The ScalarValueBuffer containing a SIRA_PRISE char value
	 * @return The char contained in the given ScalarValueBuffer
	 */
	static char getJavaChar (ScalarValueBuffer valueBuffer) {
		return valueBuffer.getByteBuffer().getChar();
	}

	/**
	 * Gets a SIRA_PRISE scalarvaluebuffer from a java String
	 * 
	 * @param value
	 *            the java String
	 * @return A ValueBuffer holding the given string
	 */
	static ScalarValueBuffer getValueBuffer (char value) {
		ByteBuffer buffer = ByteBuffer.allocate(PHYSICALSIZE);
		buffer.asCharBuffer().put(value);
		return new ScalarValueBuffer(buffer);
	}

	/**
	 * Gets a SIRA_PRISE scalarvaluebuffer from a java String
	 * 
	 * @param value
	 *            the java String
	 * @return A ValueBuffer holding the given string
	 */
	static ScalarValueBuffer getValueBuffer (String value) {
		int logicalLength = value.length();
		if (logicalLength != LOGICALSIZE) {
			throw new IllegalArgumentException(MyMessageFormat.format(Messages.getString("DbmsCharImplementation.InvalidSize"), new String[] { Integer.toString(logicalLength), Integer.toString(LOGICALSIZE), Integer.toString(LOGICALSIZE), value })); //$NON-NLS-1$
		}

		return getValueBuffer(value.charAt(0));
	}

	/**
	 * Gets the instance
	 * 
	 * @return the instance
	 */
	public static DbmsCharImplementation getInstance ( ) {
		return instance;
	}

	/**
	 * The possrep component map
	 */
	private Map<String, String> componentNameMap = initComponentNameMap();

	/**
	 * The possreps
	 */
	private Set<PossRepImplementation> possrepImplementations = new MyReadOnlySet<PossRepImplementation>(new HashSet<PossRepImplementation>(Arrays.asList(new PossRepImplementation[] { this, new UniCode() })));

	/**
	 * 
	 */
	private Map<String, PossRepImplementation> possrepImplementationsPerComponent = initPossrepImplementationsPerComponent();

	/**
	 * 
	 */
	private DbmsCharImplementation ( ) {

	}

	/**
	 * @return
	 */
	private Map<String, String> initComponentNameMap ( ) {
		// The default component has a "conceptual" component named STRING of type STRING
		HashMap<String, String> w = new HashMap<String, String>();
		w.put(TYPENAMES.STRING, TYPENAMES.STRING);
		return new MyReadOnlyMap<String, String>(w);
	}

	/**
	 * @return
	 */
	private MyReadOnlyMap<String, PossRepImplementation> initPossrepImplementationsPerComponent ( ) {
		HashMap<String, PossRepImplementation> w = new HashMap<String, PossRepImplementation>();
		for (PossRepImplementation possrepImplementation : possrepImplementations) {
			for (String componentName : possrepImplementation.getComponentNameMap().keySet()) {
				w.put(componentName, possrepImplementation);
			}
		}
		return new MyReadOnlyMap<String, PossRepImplementation>(w);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.TypeImplementations.PossRepImplementation#getComponentNameMap()
	 */
	public Map<String, String> getComponentNameMap ( ) {
		return componentNameMap;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.PossRepImplementation#extractComponentValue(java.lang.String, be.erwinsmout.SIRA_PRISE.typeimplementations.ScalarValueBuffer)
	 */
	public ValueBuffer getComponentValue (String componentName, ScalarValueBuffer valueBuffer) {
		try {
			return DbmsStringImplementation.getValueBuffer(valueToExternalRepresentation(valueBuffer));
		} catch (InvalidValueException e) {
			throw new RuntimeException(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.TypeImplementation#getDefaultPossrepImplementation()
	 */
	public PossRepImplementation getDefaultPossrepImplementation ( ) {
		return this;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.DBMS.TypeImplementations.TypeImplementation#getDefaultSize()
	 */
	public int getDfltLogicalSize ( ) {
		return LOGICALSIZE;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.DBMS.TypeImplementations.TypeImplementation#getMaxSize()
	 */
	public int getMaxLogicalSize ( ) {
		return LOGICALSIZE;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.DBMS.TypeImplementations.TypeImplementation#getMinSize()
	 */
	public int getMinLogicalSize ( ) {
		return LOGICALSIZE;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.TypeImplementation#getPhysicalSizeFor(int)
	 */
	public int getPhysicalSizeFor (int logicalSize) {
		return PHYSICALSIZE;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.SIRAPRISE.typeimplementations.TypeImplementation#getPossrepImplementation(java.lang.String)
	 */
	@Override
	public PossRepImplementation getPossrepImplementation (String componentName) {
		PossRepImplementation possRepImplementation = possrepImplementationsPerComponent.get(componentName);
		if (possRepImplementation == null) {
			throw new IllegalArgumentException(MyMessageFormat.format(Messages.getString("IntervalTypeImplementation.UnknownCompoenent"), new String[] { componentName, this.getClass().getName() })); //$NON-NLS-1$
		}
		return possRepImplementation;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.TypeImplementation#getPossRepImplementation()
	 */
	public Set<PossRepImplementation> getPossrepImplementations ( ) {
		return possrepImplementations;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.PossRepImplementation#getName()
	 */
	public String getPossrepName ( ) {
		return POSSREPNAMES.CHAR;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.PossRepImplementation#valueFromComponentValues(java.util.HashMap, int)
	 */
	public ValueBuffer valueFromComponentValues (HashMap<String, ValueBuffer> componentValueMap, int logicalSize) {
		return getValueBuffer(DbmsStringImplementation.getJavaString((ScalarValueBuffer) componentValueMap.get(POSSREPCOMPONENTNAMES.STRING)));
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.DBMS.TypeImplementations.TypeImplementation#valueFromExternalRepresentation(java.lang.String)
	 */
	public ValueBuffer valueFromExternalRepresentation (String value) throws InvalidValueException {
		try {
			return getValueBuffer(BracketParser.unMeta(value));
		} catch (InvalidEscapedCharacterException e) {
			throw new InvalidValueException(MyMessageFormat.format(Messages.getString("DbmsStringImplementation.InvalidEscape"), new String[] { value }), e); //$NON-NLS-1$
		} catch (MissingEscapedCharacterException e) {
			throw new InvalidValueException(MyMessageFormat.format(Messages.getString("DbmsStringImplementation.InvalidEscape"), new String[] { value }), e); //$NON-NLS-1$
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.DBMS.Server.TypeImplementation#fromExternal(java.lang.String, int)
	 */
	public ValueBuffer valueFromExternalRepresentation (String value, int maximumLogicalLengthAllowed) throws InvalidValueException {
		return valueFromExternalRepresentation(value);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.PossRepImplementation#valueToEscapedExternalRepresentation(be.erwinsmout.SIRA_PRISE.typeimplementations.ValueBuffer)
	 */
	public String valueToEscapedExternalRepresentation (ValueBuffer valueBuffer) {
		return BracketParser.meta(valueToExternalRepresentation(valueBuffer));
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.DBMS.Server.TypeImplementation#toExternal(java.nio.ByteBuffer, int, int, int)
	 */
	public String valueToExternalRepresentation (ValueBuffer valueBuffer) {
		return Character.toString(getJavaChar((ScalarValueBuffer) valueBuffer));
	}
}