/*
 * Created on 26-apr-2004
 */
package be.SIRAPRISE.typeimplementations;

import java.nio.ByteBuffer;
import java.util.*;

import be.SIRAPRISE.client.NAMES.POSSREPCOMPONENTNAMES;
import be.SIRAPRISE.client.NAMES.POSSREPNAMES;
import be.SIRAPRISE.client.NAMES.TYPENAMES;
import be.SIRAPRISE.util.BracketParser;
import be.SIRAPRISE.util.DuplicateNameException;
import be.SIRAPRISE.util.FixedLengthFormatter;
import be.SIRAPRISE.util.InvalidEscapedCharacterException;
import be.SIRAPRISE.util.MissingEscapedCharacterException;
import be.SIRAPRISE.util.MyReadOnlyMap;
import be.SIRAPRISE.util.MyReadOnlySet;
import be.SIRAPRISE.util.NoClosingBracketException;
import be.SIRAPRISE.util.NoOpeningBracketException;
import be.erwinsmout.MyMessageFormat;

/**
 * The implementation for the SIRA_PRISE DATE type. The physical encoding of values of type date is as follows : The value is encoded in 4 bytes, holding an integer that is computed using the dd ([1-31]), mm ([1-12]) and yyyy ([1-9999]) components of the DATE value as : <code>32 * (12 * yyyy + mm - 1) + dd</code>
 * 
 * @author Erwin Smout
 */
public final class DbmsDateImplementation implements TypeImplementation, PossRepImplementation {

	/**
	 * @author Erwin Smout
	 */
	final class DMY implements PossRepImplementation {

		/**
		 * The component name map
		 */
		private Map<String, String> dmyComponentNameMap = initDMYComponentNameMap();

		/**
		 * 
		 */
		DMY ( ) {

		}

		/**
		 * @return
		 */
		private MyReadOnlyMap<String, String> initDMYComponentNameMap ( ) {
			final HashMap<String, String> w = new HashMap<String, String>();
			w.put(POSSREPCOMPONENTNAMES.DAY, TYPENAMES.INT);
			w.put(POSSREPCOMPONENTNAMES.MONTH, TYPENAMES.INT);
			w.put(POSSREPCOMPONENTNAMES.YEAR, TYPENAMES.INT);
			return new MyReadOnlyMap<String, String>(w);
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.PossRepImplementation#getComponentNameMap()
		 */
		public Map<String, String> getComponentNameMap ( ) {
			return dmyComponentNameMap;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.PossRepImplementation#extractComponentValue(java.lang.String, be.erwinsmout.SIRA_PRISE.typeimplementations.ScalarValueBuffer)
		 */
		public ValueBuffer getComponentValue (String componentName, ScalarValueBuffer valueBuffer) {
			if (componentName.equalsIgnoreCase(POSSREPCOMPONENTNAMES.DAY)) {
				return DbmsIntImplementation.getValueBuffer(getDay(valueBuffer));
			}
			if (componentName.equalsIgnoreCase(POSSREPCOMPONENTNAMES.MONTH)) {
				return DbmsIntImplementation.getValueBuffer(getMonth(valueBuffer));
			}
			if (componentName.equalsIgnoreCase(POSSREPCOMPONENTNAMES.YEAR)) {
				return DbmsIntImplementation.getValueBuffer(getYear(valueBuffer));
			}
			throw new IllegalArgumentException(componentName);
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.PossRepImplementation#getPossrepName()
		 */
		public String getPossrepName ( ) {
			return POSSREPNAMES.DMY;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.PossRepImplementation#valueFromComponentValues(java.util.HashMap, int)
		 */
		public ValueBuffer valueFromComponentValues (HashMap<String, ValueBuffer> componentValueMap, int logicalSize) {
			return getValueBuffer(DbmsIntImplementation.getJavaInt((ScalarValueBuffer) componentValueMap.get(POSSREPCOMPONENTNAMES.YEAR)), DbmsIntImplementation.getJavaInt((ScalarValueBuffer) componentValueMap.get(POSSREPCOMPONENTNAMES.MONTH)), DbmsIntImplementation.getJavaInt((ScalarValueBuffer) componentValueMap.get(POSSREPCOMPONENTNAMES.DAY)));
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.PossRepImplementation#valueFromExternalRepresentation(java.lang.String)
		 */
		public ValueBuffer valueFromExternalRepresentation (String value) throws InvalidValueException {
			Map<String, String> m;
			try {
				m = BracketParser.createMapFromEscapedString(value, false);
			} catch (NoClosingBracketException e) {
				throw new InvalidValueException(MyMessageFormat.format(Messages.getString("DbmsDateImplementation.DMYSyntax"), new String[] { value }), e); //$NON-NLS-1$
			} catch (InvalidEscapedCharacterException e) {
				throw new InvalidValueException(MyMessageFormat.format(Messages.getString("DbmsDateImplementation.DMYSyntax"), new String[] { value }), e); //$NON-NLS-1$
			} catch (MissingEscapedCharacterException e) {
				throw new InvalidValueException(MyMessageFormat.format(Messages.getString("DbmsDateImplementation.DMYSyntax"), new String[] { value }), e); //$NON-NLS-1$
			} catch (NoOpeningBracketException e) {
				throw new InvalidValueException(MyMessageFormat.format(Messages.getString("DbmsDateImplementation.DMYSyntax"), new String[] { value }), e); //$NON-NLS-1$
			} catch (DuplicateNameException e) {
				throw new InvalidValueException(MyMessageFormat.format(Messages.getString("DbmsDateImplementation.DMYSyntax"), new String[] { value }), e); //$NON-NLS-1$
			}
			if (!m.keySet().equals(dmyComponentNameMap.keySet())) {
				throw new InvalidValueException(MyMessageFormat.format(Messages.getString("DbmsDateImplementation.DMYComponents"), new Object[] { m.keySet(), dmyComponentNameMap.keySet() })); //$NON-NLS-1$
			}

			return getValueBuffer(DbmsIntImplementation.getIntegerFromString(m.get(POSSREPCOMPONENTNAMES.YEAR)), DbmsIntImplementation.getIntegerFromString(m.get(POSSREPCOMPONENTNAMES.MONTH)), DbmsIntImplementation.getIntegerFromString(m.get(POSSREPCOMPONENTNAMES.DAY)));
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.PossRepImplementation#valueFromExternalRepresentation(java.lang.String, int)
		 */
		public ValueBuffer valueFromExternalRepresentation (String value, int maximumLogicalLengthAllowed) throws InvalidValueException {
			return valueFromExternalRepresentation(value);
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.PossRepImplementation#valueToEscapedExternalRepresentation(be.erwinsmout.SIRA_PRISE.typeimplementations.ValueBuffer)
		 */
		public String valueToEscapedExternalRepresentation (ValueBuffer valueBuffer) {
			return valueToExternalRepresentation(valueBuffer);
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.PossRepImplementation#valueToExternalRepresentation(be.erwinsmout.SIRA_PRISE.typeimplementations.ValueBuffer)
		 */
		public String valueToExternalRepresentation (ValueBuffer valueBuffer) {
			ByteBuffer byteBuffer = ((ScalarValueBuffer) valueBuffer).getByteBuffer();
			int i = byteBuffer.getInt();
			int dd = i & 0x0000001f;
			i = i >> 5;
			int mm = i % 12 + 1;
			int yyyy = i / 12;
			return POSSREPCOMPONENTNAMES.DAY + "(" + dd + ")" + POSSREPCOMPONENTNAMES.MONTH + "(" + mm + ")" + POSSREPCOMPONENTNAMES.YEAR + "(" + yyyy + ")"; //$NON-NLS-1$//$NON-NLS-2$//$NON-NLS-3$//$NON-NLS-4$ //$NON-NLS-5$ //$NON-NLS-6$
		}
	}

	/**
	 * 
	 */
	private static int[][] highestDays = new int[][] { new int[] { 31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 }, new int[] { 31, 29, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 } };

	/**
	 * the instance.
	 */
	private static DbmsDateImplementation instance = new DbmsDateImplementation();

	/**
	 * Gets the instance
	 * 
	 * @return the instance
	 */
	public static DbmsDateImplementation getInstance ( ) {
		return instance;
	}

	/**
	 * Creates a new ScalarValueBuffer holding the date value implied by the arguments
	 * 
	 * @param yyyy
	 *            The year
	 * @param mm
	 *            The month (in the normal human range [1-12], not that stupid java range [0-11].
	 * @param dd
	 *            The day
	 * @return A new ScalarValueBuffer holding the date value implied by the arguments
	 */
	public static ScalarValueBuffer getValueBuffer (int yyyy, int mm, int dd) {
		if (yyyy < 1) {
			throw new IllegalArgumentException(Messages.getString("DbmsDateImplementation.BC")); //$NON-NLS-1$
		}
		if (yyyy > 9999) {
			throw new IllegalArgumentException(Messages.getString("DbmsDateImplementation.Past9999")); //$NON-NLS-1$
		}
		if (mm < 1 || mm > 12) {
			throw new IllegalArgumentException(Messages.getString("DbmsDateImplementation.MonthRange")); //$NON-NLS-1$
		}
		final int highestDay = highestDays[((yyyy & 0x00000003) == 0 && (yyyy % 400 == 0 || !(yyyy % 100 == 0))) ? 1 : 0][mm - 1];
		if (dd < 1 || dd > highestDay) {
			throw new IllegalArgumentException(MyMessageFormat.format(Messages.getString("DbmsDateImplementation.DayRange"), new String[] { Integer.toString(mm), Integer.toString(yyyy), Integer.toString(highestDay) })); //$NON-NLS-1$
		}
		ByteBuffer buffer = ByteBuffer.allocate(4);
		buffer.position(0);
		buffer.putInt(32 * (12 * yyyy + mm - 1) + dd);
		return new ScalarValueBuffer(buffer);
	}

	/**
	 * The map of component/type names of the default implementation
	 */
	private Map<String, String> componentNameMap = initComponentNameMap();

	/**
	 * Map of the possreps
	 */
	private Set<PossRepImplementation> possrepImplementations = new MyReadOnlySet<PossRepImplementation>(new HashSet<PossRepImplementation>(Arrays.asList(new PossRepImplementation[] { this, new DMY() })));

	/**
	 * 
	 */
	private Map<String, PossRepImplementation> possrepImplementationsPerComponent = initPossrepImplementationsPerComponent();

	/**
	 * 
	 */
	private DbmsDateImplementation ( ) {

	}

	/**
	 * @return
	 */
	private MyReadOnlyMap<String, String> initComponentNameMap ( ) {
		final HashMap<String, String> w = new HashMap<String, String>();
		// "conceptual" component : ISO(STRING)
		w.put(POSSREPCOMPONENTNAMES.ISO, TYPENAMES.STRING);
		return new MyReadOnlyMap<String, String>(w);
	}

	/**
	 * @return
	 */
	private MyReadOnlyMap<String, PossRepImplementation> initPossrepImplementationsPerComponent ( ) {
		final HashMap<String, PossRepImplementation> w = new HashMap<String, PossRepImplementation>();
		for (PossRepImplementation possrepImplementation : possrepImplementations) {
			for (String componentName : possrepImplementation.getComponentNameMap().keySet()) {
				w.put(componentName, possrepImplementation);
			}
		}
		return new MyReadOnlyMap<String, PossRepImplementation>(w);
	}

	/**
	 * Gets the day
	 * 
	 * @param valueBuffer
	 *            A value Buffer holding a date value
	 * @return the day
	 */
	int getDay (ScalarValueBuffer valueBuffer) {
		return valueBuffer.getByteBuffer().getInt() & 0x0000001f;
	}

	/**
	 * Gets the month
	 * 
	 * @param valueBuffer
	 *            A value Buffer holding a date value
	 * @return the month
	 */
	int getMonth (ScalarValueBuffer valueBuffer) {
		return ((valueBuffer.getByteBuffer().getInt() >> 5) % 12) + 1;
	}

	/**
	 * Gets the year
	 * 
	 * @param valueBuffer
	 *            A value Buffer holding a date value
	 * @return the year
	 */
	int getYear (ScalarValueBuffer valueBuffer) {
		return (valueBuffer.getByteBuffer().getInt() >> 5) / 12;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.TypeImplementations.PossRepImplementation#getComponentNameMap()
	 */
	public Map<String, String> getComponentNameMap ( ) {
		return componentNameMap;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.PossRepImplementation#extractComponentValue(java.lang.String, be.erwinsmout.SIRA_PRISE.typeimplementations.ScalarValueBuffer)
	 */
	public ValueBuffer getComponentValue (String componentName, ScalarValueBuffer valueBuffer) {
		try {
			return DbmsStringImplementation.getValueBuffer(valueToExternalRepresentation(valueBuffer));
		} catch (InvalidValueException e) {
			throw new RuntimeException(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.TypeImplementation#getDefaultPossrepImplementation()
	 */
	public PossRepImplementation getDefaultPossrepImplementation ( ) {
		return this;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.DBMS.TypeImplementations.TypeImplementation#getDefaultSize()
	 */
	public int getDfltLogicalSize ( ) {
		return 1;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.DBMS.TypeImplementations.TypeImplementation#getMaxSize()
	 */
	public int getMaxLogicalSize ( ) {
		return 1;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.DBMS.TypeImplementations.TypeImplementation#getMinSize()
	 */
	public int getMinLogicalSize ( ) {
		return 1;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.TypeImplementation#getPhysicalSizeFor(int)
	 */
	public int getPhysicalSizeFor (int logicalSize) {
		return 4;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.SIRAPRISE.typeimplementations.TypeImplementation#getPossrepImplementation(java.lang.String)
	 */
	@Override
	public PossRepImplementation getPossrepImplementation (String componentName) {
		PossRepImplementation possRepImplementation = possrepImplementationsPerComponent.get(componentName);
		if (possRepImplementation == null) {
			throw new IllegalArgumentException(MyMessageFormat.format(Messages.getString("IntervalTypeImplementation.UnknownCompoenent"), new String[] { componentName, this.getClass().getName() })); //$NON-NLS-1$
		}
		return possRepImplementation;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.TypeImplementation#getPossRepImplementation()
	 */
	public Set<PossRepImplementation> getPossrepImplementations ( ) {
		return possrepImplementations;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.PossRepImplementation#getName()
	 */
	public String getPossrepName ( ) {
		return POSSREPNAMES.ISO;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.PossRepImplementation#valueFromComponentValues(java.util.HashMap, int)
	 */
	public ValueBuffer valueFromComponentValues (HashMap<String, ValueBuffer> componentValueMap, int logicalSize) {
		return valueFromExternalRepresentation(DbmsStringImplementation.getJavaString((ScalarValueBuffer) componentValueMap.get(POSSREPCOMPONENTNAMES.ISO)));
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.DBMS.TypeImplementations.TypeImplementation#valueFromExternalRepresentation(java.lang.String)
	 */
	public ValueBuffer valueFromExternalRepresentation (String value) {
		if (value.length() != 10 || value.charAt(4) != '-' || value.charAt(7) != '-') {
			throw new IllegalArgumentException(MyMessageFormat.format(Messages.getString("DbmsDateImplementation.Value"), new String[] { value })); //$NON-NLS-1$
		}
		int yyyy, mm, dd;
		try {
			yyyy = Integer.parseInt(value.substring(0, 4));
			mm = Integer.parseInt(value.substring(5, 7));
			dd = Integer.parseInt(value.substring(8));
		} catch (NumberFormatException e) {
			throw new IllegalArgumentException(MyMessageFormat.format(Messages.getString("DbmsDateImplementation.Value"), new String[] { value })); //$NON-NLS-1$
		}
		return getValueBuffer(yyyy, mm, dd);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.DBMS.Server.TypeImplementation#fromExternal(java.lang.String, int)
	 */
	public ValueBuffer valueFromExternalRepresentation (String value, int maximumLogicalLengthAllowed) throws InvalidValueException {
		return valueFromExternalRepresentation(value);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.PossRepImplementation#valueToEscapedExternalRepresentation(be.erwinsmout.SIRA_PRISE.typeimplementations.ValueBuffer)
	 */
	public String valueToEscapedExternalRepresentation (ValueBuffer valueBuffer) {
		return valueToExternalRepresentation(valueBuffer);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.DBMS.Server.TypeImplementation#toExternal(java.nio.ByteBuffer, int, int, int)
	 */
	public String valueToExternalRepresentation (ValueBuffer valueBuffer) {
		int i = ((ScalarValueBuffer) valueBuffer).getByteBuffer().getInt();
		int dd = i & 0x0000001f;
		i = i >> 5;
		int mm = i % 12 + 1;
		int yyyy = i / 12;
		return FixedLengthFormatter.prependZeroes(4, Integer.toString(yyyy)) + '-' + FixedLengthFormatter.prependZeroes(2, Integer.toString(mm)) + '-' + FixedLengthFormatter.prependZeroes(2, Integer.toString(dd));
	}
}
