/*
 * Created on 26-apr-2004
 */
package be.SIRAPRISE.typeimplementations;

import java.nio.ByteBuffer;

import java.util.*;

import be.SIRAPRISE.client.NAMES.POSSREPNAMES;
import be.SIRAPRISE.client.NAMES.TYPENAMES;
import be.erwinsmout.MyMessageFormat;

/**
 * The implementation for the SIRA_PRISE decimal type. The physical encoding of the values is as follows :
 * <ul>
 * <li>The encoded value consists of 10 bytes, labeled b9-b0</li>
 * <li>Bytes b9-b2 hold a java long, whose value is equal to the number that is obtained by removing the decimal sign, all leading zeroes from the integer fraction, and all trailing zeroes from the decimal fraction of the number. E.g. for the decimal number 01.50, this is the value 15. Observe that this is where the limit of "no more than 18 significant digits" derives from.</li>
 * <li>Byte b1 holds the number of digits present in the integer fraction. E.g. for the number 0.5, this is 0, for the number 150.0, this is 3.</li>
 * <li>Byte b0 holds the number of digits in the decimals fraction. E.g. for the number 0.5, this is 1, for the number 150.0, this is 0.</li>
 * </ul>
 * 
 * @author Erwin Smout
 * @deprecated The use of this type is discouraged. The definition is rather awkward, the implementation rather poor, and the set of provided operators incomplete.
 */
public final class DbmsDecimalImplementation implements TypeImplementation, PossRepImplementation {

	/**
	 * 
	 */
	private static DbmsDecimalImplementation instance = new DbmsDecimalImplementation();

	/**
	 * The powers of 10, so that they don't have to be computed using math.pow()
	 */
	private static final double[] powers_d = new double[] { 1, 10, 100, 1000, 10000, 100000, 1000000, 10000000, 100000000, 1000000000, 10000000000L, 100000000000L, 1000000000000L, 10000000000000L, 100000000000000L, 1000000000000000L, 10000000000000000L, 100000000000000000L, 1000000000000000000L };

	/**
	 * The powers of 10, so that they don't have to be computed using math.pow()
	 */
	final static long[] powers_l = new long[] { 1, 10, 100, 1000, 10000, 100000, 1000000, 10000000, 100000000, 1000000000, 10000000000L, 100000000000L, 1000000000000L, 10000000000000L, 100000000000000L, 1000000000000000L, 10000000000000000L, 100000000000000000L, 1000000000000000000L };

	/**
	 * Inspects a ScalarValueBuffer holding a sira_prise decimal and returns that decimal as a java double
	 * 
	 * @param valueBuffer
	 *            The scalarvaluebuffer holding a sira_prise decimal value
	 * @return the contained sira_prise decimal
	 */
	static double getJavaDecimal (ScalarValueBuffer valueBuffer) {
		// Integer and decimal position
		ByteBuffer byteBuffer = valueBuffer.getByteBuffer();
		long value = byteBuffer.getLong();
		byteBuffer.get(); // digits
		byte decimals = byteBuffer.get(); // [0-18]
		return value / powers_d[decimals];
	}

	/**
	 * Gets the instance
	 * 
	 * @return the instance
	 */
	public static DbmsDecimalImplementation getInstance ( ) {
		return instance;
	}

	/**
	 * The components
	 */
	private HashMap<String, String> componentNameMap = new HashMap<String, String>();

	/**
	 * Map of the possreps
	 */
	private Set<PossRepImplementation> possrepImplementations = new HashSet<PossRepImplementation>();

	/**
	 * 
	 */
	private Map<String, PossRepImplementation> possrepImplementationsPerComponent = new HashMap<String, PossRepImplementation>();

	/**
	 * 
	 */
	private DbmsDecimalImplementation ( ) {
		possrepImplementations.add(this);
		// "conceptual" component : STRING(STRING)
		componentNameMap.put(TYPENAMES.STRING, TYPENAMES.STRING);

		for (PossRepImplementation possrepImplementation : possrepImplementations) {
			for (String componentName : possrepImplementation.getComponentNameMap().keySet()) {
				possrepImplementationsPerComponent.put(componentName, possrepImplementation);
			}
		}
	}

	/**
	 * Gets The ValueBuffer holding the value of which value is the external representation
	 * 
	 * @param value
	 *            The external representation of a value of this type
	 * @return The ValueBuffer holding the value of which value is the external representation
	 */
	private ValueBuffer getValueBuffer (String value) {
		String unsignedValue;
		int sign = 1;
		if (value.charAt(0) == '+' || value.charAt(0) == '-') {
			unsignedValue = value.substring(1);
			if (value.charAt(0) == '-') {
				sign = -1;
			}
		} else {
			int endpos = value.length() - 1;
			if (value.charAt(endpos) == '+' || value.charAt(endpos) == '-') {
				unsignedValue = value.substring(0, endpos);
				if (value.charAt(endpos) == '-') {
					sign = -1;
				}
			} else {
				unsignedValue = value;
			}
		}

		int cp = unsignedValue.indexOf('.');
		if (cp < 0) {
			cp = unsignedValue.indexOf(',');
		}
		String integerPortionText;
		int decimals;
		if (cp < 0) {
			integerPortionText = unsignedValue;
			decimals = 0;
		} else {
			integerPortionText = unsignedValue.substring(0, cp);

			String fractionalPortionText = unsignedValue.substring(cp + 1);
			decimals = fractionalPortionText.length();
			while (decimals > 0 && fractionalPortionText.charAt(decimals - 1) == '0') {
				fractionalPortionText = fractionalPortionText.substring(0, --decimals);
			}
			integerPortionText += fractionalPortionText;
		}

		int digits = integerPortionText.length() - decimals;
		while (digits > 0 && integerPortionText.charAt(0) == '0') {
			integerPortionText = integerPortionText.substring(1);
			digits--;
		}

		if (digits + decimals > 18) {
			throw new IllegalArgumentException(MyMessageFormat.format(Messages.getString("DbmsDecimalImplementation.TotalDigits"), new String[] { value, Integer.toString(digits + decimals) })); //$NON-NLS-1$
		}

		long integerValue;
		try {
			if (integerPortionText.length() > 0) {
				integerValue = Long.parseLong(integerPortionText) * sign;
			} else {
				integerValue = 0;
			}
		} catch (NumberFormatException e) {
			throw new IllegalArgumentException(MyMessageFormat.format(Messages.getString("DbmsDecimalImplementation.NumericDigits"), new String[] { value })); //$NON-NLS-1$
		}

		ByteBuffer byteBuffer = ByteBuffer.allocate(10);
		byteBuffer.putLong(integerValue);
		byteBuffer.put((byte) digits);
		byteBuffer.put((byte) decimals);
		return new ScalarValueBuffer(byteBuffer);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.TypeImplementations.PossRepImplementation#getComponentNameMap()
	 */
	public Map<String, String> getComponentNameMap ( ) {
		return componentNameMap;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.PossRepImplementation#extractComponentValue(java.lang.String, be.erwinsmout.SIRA_PRISE.typeimplementations.ScalarValueBuffer)
	 */
	public ValueBuffer getComponentValue (String componentName, ScalarValueBuffer valueBuffer) {
		try {
			return DbmsStringImplementation.getValueBuffer(valueToExternalRepresentation(valueBuffer));
		} catch (InvalidValueException e) {
			throw new RuntimeException(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.TypeImplementation#getDefaultPossrepImplementation()
	 */
	public PossRepImplementation getDefaultPossrepImplementation ( ) {
		return this;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.DBMS.TypeImplementations.TypeImplementation#getDefaultSize()
	 */
	public int getDfltLogicalSize ( ) {
		return 18;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.DBMS.TypeImplementations.TypeImplementation#getMaxSize()
	 */
	public int getMaxLogicalSize ( ) {
		return 18;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.DBMS.TypeImplementations.TypeImplementation#getMinSize()
	 */
	public int getMinLogicalSize ( ) {
		return 18;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.TypeImplementation#getPhysicalSizeFor(int)
	 */
	public int getPhysicalSizeFor (int logicalSize) {
		return 10;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.SIRAPRISE.typeimplementations.TypeImplementation#getPossrepImplementation(java.lang.String)
	 */
	@Override
	public PossRepImplementation getPossrepImplementation (String componentName) {
		PossRepImplementation possRepImplementation = possrepImplementationsPerComponent.get(componentName);
		if (possRepImplementation == null) {
			throw new IllegalArgumentException(MyMessageFormat.format(Messages.getString("IntervalTypeImplementation.UnknownCompoenent"), new String[] { componentName, this.getClass().getName() })); //$NON-NLS-1$
		}
		return possRepImplementation;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.TypeImplementation#getPossRepImplementation()
	 */
	public Set<PossRepImplementation> getPossrepImplementations ( ) {
		return possrepImplementations;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.PossRepImplementation#getName()
	 */
	public String getPossrepName ( ) {
		return POSSREPNAMES.DECIMAL;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.PossRepImplementation#valueFromComponentValues(java.util.HashMap, int)
	 */
	public ValueBuffer valueFromComponentValues (HashMap<String, ValueBuffer> componentValueMap, int logicalSize) {
		return getValueBuffer(DbmsStringImplementation.getJavaString((ScalarValueBuffer) componentValueMap.get(TYPENAMES.STRING)));
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.DBMS.TypeImplementations.TypeImplementation#valueFromExternalRepresentation(java.lang.String)
	 */
	public ValueBuffer valueFromExternalRepresentation (String value) throws InvalidValueException {
		return getValueBuffer(value);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.DBMS.Server.TypeImplementation#fromExternal(java.lang.String)
	 */
	public ValueBuffer valueFromExternalRepresentation (String value, int maximumLogicalLengthAllowed) throws InvalidValueException {
		return getValueBuffer(value);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.PossRepImplementation#valueToEscapedExternalRepresentation(be.erwinsmout.SIRA_PRISE.typeimplementations.ValueBuffer)
	 */
	public String valueToEscapedExternalRepresentation (ValueBuffer valueBuffer) {
		return valueToExternalRepresentation(valueBuffer);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.DBMS.Server.TypeImplementation#toExternal(java.nio.ByteBuffer, int, int, int)
	 */
	public String valueToExternalRepresentation (ValueBuffer valueBuffer) {
		ByteBuffer byteBuffer = ((ScalarValueBuffer) valueBuffer).getByteBuffer();
		long integerValue = byteBuffer.getLong();
		int signpresent = integerValue < 0 ? 1 : 0;
		int digits = byteBuffer.get();
		StringBuilder representation = new StringBuilder(Long.toString(integerValue));
		representation.insert(digits + signpresent, '.');
		return representation.toString();
	}
}
