/*
 * Created on 26-apr-2004
 */
package be.SIRAPRISE.typeimplementations;

import java.nio.ByteBuffer;
import java.util.*;

import be.SIRAPRISE.client.NAMES.POSSREPNAMES;
import be.SIRAPRISE.client.NAMES.TYPENAMES;
import be.SIRAPRISE.util.MyReadOnlyMap;
import be.SIRAPRISE.util.MyReadOnlySet;
import be.erwinsmout.MyMessageFormat;

/**
 * The implementation for the system-defined FLOAT type. The physical encoding of FLOAT values consists of 8 bytes holding a java double.
 * 
 * @author Erwin Smout
 */
public final class DbmsFloatImplementation implements TypeImplementation, PossRepImplementation {

	/**
	 * the instance.
	 */
	private static DbmsFloatImplementation instance = new DbmsFloatImplementation();

	/**
	 * Gets the instance
	 * 
	 * @return the instance
	 */
	public static DbmsFloatImplementation getInstance ( ) {
		return instance;
	}

	/**
	 * Gets a java double value out of a sira prise float scalarvaluebuffer
	 * 
	 * @param valueBuffer
	 *            A scalarvaluebuffer holding a siraprise float value
	 * @return a java double value out of a sira prise float scalarvaluebuffer
	 */
	public static double getJavaDouble (ValueBuffer valueBuffer) {
		return ((ScalarValueBuffer) valueBuffer).getByteBuffer().getDouble();
	}

	/**
	 * Transforms a java type into a SIRA_PRISE ValueBuffer
	 * 
	 * @param javaDouble
	 *            a java double
	 * @return A ScalarValueBuffer holding the given java double value as a float
	 */
	public static ScalarValueBuffer getValueBuffer (double javaDouble) {
		if (Double.isNaN(javaDouble)) {
			throw new IllegalArgumentException(Messages.getString("DbmsFloatImplementation.NotANumber")); //$NON-NLS-1$
		}
		ByteBuffer byteBuffer = ByteBuffer.allocate(8);
		if (javaDouble == 0.0) {
			byteBuffer.putDouble(0.0); // to overcome the unequal bitpatterns of IEEE +0.0 and -0.0.
		} else {
			byteBuffer.putDouble(javaDouble);
		}
		return new ScalarValueBuffer(byteBuffer);
	}

	/**
	 * The component name/type mapping
	 */
	private Map<String, String> componentNameMap = initComponentNameMap();

	/**
	 * Map of the possreps
	 */
	private Set<PossRepImplementation> possrepImplementations = new MyReadOnlySet<PossRepImplementation>(new HashSet<PossRepImplementation>(Arrays.asList(new PossRepImplementation[] { this })));

	/**
	 * 
	 */
	private Map<String, PossRepImplementation> possrepImplementationsPerComponent = initPossrepImplementationsPerComponent();

	/**
	 * 
	 */
	private DbmsFloatImplementation ( ) {

	}

	/**
	 * @return
	 */
	private MyReadOnlyMap<String, String> initComponentNameMap ( ) {
		// The default component has a "conceptual" component named STRING of type STRING
		HashMap<String, String> w = new HashMap<String, String>();
		w.put(TYPENAMES.STRING, TYPENAMES.STRING);
		return new MyReadOnlyMap<String, String>(w);
	}

	/**
	 * @return
	 */
	private MyReadOnlyMap<String, PossRepImplementation> initPossrepImplementationsPerComponent ( ) {
		final HashMap<String, PossRepImplementation> w = new HashMap<String, PossRepImplementation>();
		for (PossRepImplementation possrepImplementation : possrepImplementations) {
			for (String componentName : possrepImplementation.getComponentNameMap().keySet()) {
				w.put(componentName, possrepImplementation);
			}
		}
		return new MyReadOnlyMap<String, PossRepImplementation>(w);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.TypeImplementations.PossRepImplementation#getComponentNameMap()
	 */
	public Map<String, String> getComponentNameMap ( ) {
		return componentNameMap;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.PossRepImplementation#extractComponentValue(java.lang.String, be.erwinsmout.SIRA_PRISE.typeimplementations.ScalarValueBuffer)
	 */
	public ValueBuffer getComponentValue (String componentName, ScalarValueBuffer valueBuffer) {
		try {
			return DbmsStringImplementation.getValueBuffer(valueToExternalRepresentation(valueBuffer));
		} catch (InvalidValueException e) {
			throw new RuntimeException(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.TypeImplementation#getDefaultPossrepImplementation()
	 */
	public PossRepImplementation getDefaultPossrepImplementation ( ) {
		return this;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.DBMS.TypeImplementations.TypeImplementation#getDefaultSize()
	 */
	public int getDfltLogicalSize ( ) {
		return 1;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.DBMS.TypeImplementations.TypeImplementation#getMaxSize()
	 */
	public int getMaxLogicalSize ( ) {
		return 1;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.DBMS.TypeImplementations.TypeImplementation#getMinSize()
	 */
	public int getMinLogicalSize ( ) {
		return 1;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.TypeImplementation#getPhysicalSizeFor(int)
	 */
	public int getPhysicalSizeFor (int logicalSize) {
		return 8;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.SIRAPRISE.typeimplementations.TypeImplementation#getPossrepImplementation(java.lang.String)
	 */
	@Override
	public PossRepImplementation getPossrepImplementation (String componentName) {
		PossRepImplementation possRepImplementation = possrepImplementationsPerComponent.get(componentName);
		if (possRepImplementation == null) {
			throw new IllegalArgumentException(MyMessageFormat.format(Messages.getString("IntervalTypeImplementation.UnknownCompoenent"), new String[] { componentName, this.getClass().getName() })); //$NON-NLS-1$
		}
		return possRepImplementation;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.TypeImplementation#getPossRepImplementation()
	 */
	public Set<PossRepImplementation> getPossrepImplementations ( ) {
		return possrepImplementations;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.PossRepImplementation#getName()
	 */
	public String getPossrepName ( ) {
		return POSSREPNAMES.FLOAT;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.PossRepImplementation#valueFromComponentValues(java.util.HashMap, int)
	 */
	public ValueBuffer valueFromComponentValues (HashMap<String, ValueBuffer> componentValueMap, int logicalSize) {
		return valueFromExternalRepresentation(DbmsStringImplementation.getJavaString((ScalarValueBuffer) componentValueMap.get(TYPENAMES.STRING)));
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.DBMS.TypeImplementations.TypeImplementation#valueFromExternalRepresentation(java.lang.String)
	 */
	public ValueBuffer valueFromExternalRepresentation (String value) {
		double javaDouble;
		try {
			javaDouble = Double.parseDouble(value);
		} catch (NumberFormatException e) {
			throw new IllegalArgumentException(MyMessageFormat.format(Messages.getString("DbmsFloatImplementation.Value"), new String[] { value })); //$NON-NLS-1$
		}
		return getValueBuffer(javaDouble);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.DBMS.Server.TypeImplementation#fromExternal(java.lang.String, int)
	 */
	public ValueBuffer valueFromExternalRepresentation (String value, int maximumLogicalLengthAllowed) throws InvalidValueException {
		return valueFromExternalRepresentation(value);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.PossRepImplementation#valueToEscapedExternalRepresentation(be.erwinsmout.SIRA_PRISE.typeimplementations.ValueBuffer)
	 */
	public String valueToEscapedExternalRepresentation (ValueBuffer valueBuffer) {
		return valueToExternalRepresentation(valueBuffer);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.DBMS.Server.TypeImplementation#toExternal(java.nio.ByteBuffer, int, int, int)
	 */
	public String valueToExternalRepresentation (ValueBuffer valueBuffer) {
		return Double.toString(getJavaDouble(valueBuffer));
	}
}