/*
 * Created on 7-jun-04
 */
package be.SIRAPRISE.typeimplementations;

import java.nio.ByteBuffer;
import java.util.*;

import be.SIRAPRISE.client.NAMES.TYPENAMES;
import be.SIRAPRISE.util.BracketParser;
import be.SIRAPRISE.util.DuplicateNameException;
import be.SIRAPRISE.util.InvalidEscapedCharacterException;
import be.SIRAPRISE.util.MissingEscapedCharacterException;
import be.SIRAPRISE.util.MyReadOnlyMap;
import be.SIRAPRISE.util.MyReadOnlySet;
import be.SIRAPRISE.util.NoClosingBracketException;
import be.SIRAPRISE.util.NoOpeningBracketException;
import be.erwinsmout.MyMessageFormat;

/**
 * Type 'Foo' is provided to demonstrate the type-plugin feature. Its encoding is as follows :
 * <ul>
 * <li>Two 8-byte java doubles, representing the X and Y component values, respectively</li>
 * </ul>
 * 
 * @author Erwin Smout
 */
public final class DbmsFooImplementation implements TypeImplementation, PossRepImplementation {

	/**
	 * the instance.
	 */
	private static final DbmsFooImplementation instance = new DbmsFooImplementation();

	/**
	 * 
	 */
	private static final String X = "X"; //$NON-NLS-1$

	/**
	 * 
	 */
	private static final String Y = "Y"; //$NON-NLS-1$

	/**
	 * @return
	 */
	private static MyReadOnlyMap<String, String> initComponentNameMap ( ) {
		HashMap<String, String> w = new HashMap<String, String>();
		w.put(X, TYPENAMES.FLOAT);
		w.put(Y, TYPENAMES.FLOAT);
		return new MyReadOnlyMap<String, String>(w);
	}

	/**
	 * Gets A ValueBuffer holding the given value
	 * 
	 * @param x
	 * @param y
	 * @return A ValueBuffer holding the given value
	 */
	public static ValueBuffer getFooValuebuffer (double x, double y) {
		ByteBuffer byteBuffer = ByteBuffer.allocate(16);
		if (x == 0.0) {
			byteBuffer.putDouble(0.0); // to overcome the unequal bitpatterns of IEEE +0.0 and -0.0.
		} else {
			byteBuffer.putDouble(x);
		}
		if (y == 0.0) {
			byteBuffer.putDouble(0.0); // to overcome the unequal bitpatterns of IEEE +0.0 and -0.0.
		} else {
			byteBuffer.putDouble(y);
		}
		return new ScalarValueBuffer(byteBuffer);
	}

	/**
	 * Gets the instance
	 * 
	 * @return the instance
	 */
	public static DbmsFooImplementation getInstance ( ) {
		return instance;
	}

	/**
	 * The components of the default possrep (=degrees minutes seconds)
	 */
	private final MyReadOnlyMap<String, String> componentNameMap = initComponentNameMap();

	/**
	 * Map of the possreps
	 */
	private final Set<PossRepImplementation> possrepImplementations = new MyReadOnlySet<PossRepImplementation>(new HashSet<PossRepImplementation>(Arrays.asList(new PossRepImplementation[] { this })));

	/**
	 * 
	 */
	private final Map<String, PossRepImplementation> possrepImplementationsPerComponent = initPossrepImplementationsPerComponent();

	/**
	 * Builds the DbmsAngleImplementation object
	 */
	private DbmsFooImplementation ( ) {

	}

	/**
	 * @return
	 */
	private Map<String, PossRepImplementation> initPossrepImplementationsPerComponent ( ) {
		HashMap<String, PossRepImplementation> w = new HashMap<String, PossRepImplementation>();
		for (PossRepImplementation possrepImplementation : possrepImplementations) {
			for (String componentName : possrepImplementation.getComponentNameMap().keySet()) {
				w.put(componentName, possrepImplementation);
			}
		}
		return new MyReadOnlyMap<String, PossRepImplementation>(w);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.TypeImplementations.PossRepImplementation#getComponentNameMap()
	 */
	public Map<String, String> getComponentNameMap ( ) {
		return componentNameMap;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.PossRepImplementation#extractComponentValue(java.lang.String, be.erwinsmout.SIRA_PRISE.typeimplementations.ScalarValueBuffer)
	 */
	public ValueBuffer getComponentValue (String componentName, ScalarValueBuffer valueBuffer) {
		if (componentName.equalsIgnoreCase(X)) {
			return DbmsFloatImplementation.getValueBuffer(valueBuffer.getByteBuffer().getDouble());
		}
		if (componentName.equalsIgnoreCase(Y)) {
			return DbmsFloatImplementation.getValueBuffer(((ByteBuffer) valueBuffer.getByteBuffer().position(8)).getDouble());
		}
		throw new IllegalArgumentException(componentName);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.TypeImplementation#getDefaultPossrepImplementation()
	 */
	public PossRepImplementation getDefaultPossrepImplementation ( ) {
		return this;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.DBMS.TypeImplementations.TypeImplementation#getDefaultSize()
	 */
	public int getDfltLogicalSize ( ) {
		return 1;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.DBMS.TypeImplementations.TypeImplementation#getMaxSize()
	 */
	public int getMaxLogicalSize ( ) {
		return 1;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.DBMS.TypeImplementations.TypeImplementation#getMinSize()
	 */
	public int getMinLogicalSize ( ) {
		return 1;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.TypeImplementation#getPhysicalSizeFor(int)
	 */
	public int getPhysicalSizeFor (int logicalSize) {
		return 16;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.SIRAPRISE.typeimplementations.TypeImplementation#getPossrepImplementation(java.lang.String)
	 */
	@Override
	public PossRepImplementation getPossrepImplementation (String componentName) {
		PossRepImplementation possRepImplementation = possrepImplementationsPerComponent.get(componentName);
		if (possRepImplementation == null) {
			throw new IllegalArgumentException(MyMessageFormat.format(Messages.getString("IntervalTypeImplementation.UnknownCompoenent"), new String[] { componentName, this.getClass().getName() })); //$NON-NLS-1$
		}
		return possRepImplementation;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.TypeImplementation#getPossRepImplementation()
	 */
	public Set<PossRepImplementation> getPossrepImplementations ( ) {
		return possrepImplementations;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.PossRepImplementation#getName()
	 */
	public String getPossrepName ( ) {
		return "FOO"; //$NON-NLS-1$
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.PossRepImplementation#valueFromComponentValues(java.util.HashMap, int)
	 */
	public ValueBuffer valueFromComponentValues (HashMap<String, ValueBuffer> componentValueMap, int logicalSize) {
		return getFooValuebuffer(DbmsFloatImplementation.getJavaDouble(componentValueMap.get(X)), DbmsFloatImplementation.getJavaDouble(componentValueMap.get(Y)));
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.DBMS.TypeImplementations.TypeImplementation#valueFromExternalRepresentation(java.lang.String)
	 */
	public ValueBuffer valueFromExternalRepresentation (String value) throws InvalidValueException {
		Map<String, String> componentValuesMap;
		try {
			componentValuesMap = BracketParser.createMapFromEscapedString(value, false);
		} catch (NoClosingBracketException e1) {
			throw new InvalidValueException(Messages.getString("DbmsAngleImplementation.BracketSyntax"), e1); //$NON-NLS-1$
		} catch (InvalidEscapedCharacterException e1) {
			throw new InvalidValueException(Messages.getString("DbmsAngleImplementation.BracketSyntax"), e1); //$NON-NLS-1$
		} catch (MissingEscapedCharacterException e1) {
			throw new InvalidValueException(Messages.getString("DbmsAngleImplementation.BracketSyntax"), e1); //$NON-NLS-1$
		} catch (NoOpeningBracketException e1) {
			throw new InvalidValueException(Messages.getString("DbmsAngleImplementation.BracketSyntax"), e1); //$NON-NLS-1$
		} catch (DuplicateNameException e1) {
			throw new InvalidValueException(Messages.getString("DbmsAngleImplementation.BracketSyntax"), e1); //$NON-NLS-1$
		}

		String x = componentValuesMap.get(X);
		String y = componentValuesMap.get(Y);
		if (x == null || y == null) {
			throw new InvalidValueException(Messages.getString("DbmsAngleImplementation.ComponentMissing")); //$NON-NLS-1$
		}

		return getFooValuebuffer(Double.parseDouble(x), Double.parseDouble(y));
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.DBMS.TypeImplementations.TypeImplementation#valueFromExternalRepresentation(java.lang.String, int)
	 */
	public ValueBuffer valueFromExternalRepresentation (String value, int maximumLogicalLengthAllowed) throws InvalidValueException {
		return valueFromExternalRepresentation(value);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.PossRepImplementation#valueToEscapedExternalRepresentation(be.erwinsmout.SIRA_PRISE.typeimplementations.ValueBuffer)
	 */
	public String valueToEscapedExternalRepresentation (ValueBuffer valueBuffer) {
		return valueToExternalRepresentation(valueBuffer);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.DBMS.TypeImplementations.TypeImplementation#valueToExternalRepresentation(java.nio.ByteBuffer, int, int, int)
	 */
	public String valueToExternalRepresentation (ValueBuffer valueBuffer) {
		final ByteBuffer byteBuffer = ((ScalarValueBuffer) valueBuffer).getByteBuffer();
		return X + "(" + Double.toString(byteBuffer.getDouble()) + ")" + Y + "(" + Double.toString(byteBuffer.getDouble()) + ")"; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
	}
}