/*
 * Created on 16-jun-04
 */
package be.SIRAPRISE.typeimplementations;

import java.nio.ByteBuffer;
import java.util.*;

import be.SIRAPRISE.client.NAMES.POSSREPNAMES;
import be.SIRAPRISE.client.NAMES.TYPENAMES;
import be.SIRAPRISE.util.BracketParser;
import be.SIRAPRISE.util.MyReadOnlyMap;
import be.SIRAPRISE.util.MyReadOnlySet;
import be.erwinsmout.MyMessageFormat;

/**
 * The implementation for the system-defined NAME type. The physical encoding of NAME values is as follows :
 * <ul>
 * <li>Being a variable-length type, the first four bytes hold an integer that defines the logical length of the value, in this case the number of characters in the NAME value.</li>
 * <li>The next bytes are the characters of the NAME value, at a rate of one byte per character. This is possible because all valid characters in a NAME value are in the unicode range below 128.</li>
 * </ul>
 * 
 * @author Erwin Smout
 */
public final class DbmsNameImplementation implements TypeImplementation, PossRepImplementation {

	/**
	 * the instance.
	 */
	private static final DbmsNameImplementation instance = new DbmsNameImplementation();

	/**
	 * The maximum logical length of a sira_prise name.
	 */
	private static final int MAXLOGICALSIZE = 250;

	/**
	 * The minimum logical length of a sira_prise name
	 */
	private static final int MINLOGICALSIZE = 1;

	/**
	 * List of all valid characters
	 */
	private static final String validChars = "ABCDEFGHIJKLMNOPQRSTUVWXYZ1234567890@_. "; //$NON-NLS-1$

	/**
	 * Gets The size of the byte buffer needed to encode a name of the given logical size
	 * 
	 * @param logicalSize
	 *            The logical size of a name to be encoded
	 * @return The size of the byte buffer needed to encode a name of the given logical size
	 */
	private static int physicalSizeFor (int logicalSize) {
		return 4 + logicalSize;
	}

	/**
	 * Gets a ValueBuffer containing the given Name value
	 * 
	 * @param value
	 *            The Name value to be placed in a ScalarValueBuffer
	 * @return A ValueBuffer containing the given Name value
	 */
	public static ScalarValueBuffer getValueBuffer (String value) {
		return getValueBuffer(value, value.length());
	}

	/**
	 * Gets a ValueBuffer of the given size containing the given Name value
	 * 
	 * @param value
	 *            The Name value to be placed in a ScalarValueBuffer
	 * @param maximumLogicalLengthAllowed
	 *            The maximum logical length allowed
	 * @return A ValueBuffer containing the given Name value
	 */
	public static ScalarValueBuffer getValueBuffer (String value, int maximumLogicalLengthAllowed) {
		if (value.length() < MINLOGICALSIZE || value.length() > Math.min(MAXLOGICALSIZE, maximumLogicalLengthAllowed)) {
			throw new IllegalArgumentException(MyMessageFormat.format(Messages.getString("DbmsNameImplementation.InvalidSize"), new String[] { value, Integer.toString(MINLOGICALSIZE), Integer.toString(Math.min(MAXLOGICALSIZE, maximumLogicalLengthAllowed)), Integer.toString(value.length()) })); //$NON-NLS-1$
		}

		ByteBuffer byteBuffer = ByteBuffer.allocate(physicalSizeFor(value.length()));
		byteBuffer.putInt(value.length());

		int p = 0;
		while (p < value.length()) {
			char c = Character.toUpperCase(value.charAt(p++));
			if (validChars.indexOf(c) < 0) {
				throw new IllegalArgumentException(MyMessageFormat.format(Messages.getString("DbmsNameImplementation.InvalidToken"), new Object[] { Character.toString(c), TYPENAMES.NAME })); //$NON-NLS-1$
			} else {
				byteBuffer.put((byte) c);
			}
		}

		return new ScalarValueBuffer(byteBuffer);
	}

	/**
	 * Gets the instance
	 * 
	 * @return the instance
	 */
	public static DbmsNameImplementation getInstance ( ) {
		return instance;
	}

	/**
	 * Gets the Name value contained in the valuebuffer in java String format
	 * 
	 * @param buffer
	 *            a ValueBuffer containing a Name value
	 * @return The Name value contained in the valuebuffer in java String format
	 */
	public static String getJavaString (ScalarValueBuffer buffer) {
		ByteBuffer byteBuffer = buffer.getByteBuffer();
		byteBuffer.position(0);
		int byteLength = byteBuffer.getInt(), i = 0;
		byte[] bytes = new byte[byteLength];
		while (byteLength-- > 0) {
			bytes[i++] = byteBuffer.get();
		}
		return new String(bytes);
	}

	/**
	 * The components
	 */
	private Map<String,String> componentNameMap = initComponentNameMap();

	/**
	 * Map of the possreps
	 */
	private Set<PossRepImplementation> possrepImplementations = new MyReadOnlySet<PossRepImplementation>(new HashSet<PossRepImplementation>(Arrays.asList(new PossRepImplementation[] { this })));

	/**
	 * 
	 */
	private Map<String, PossRepImplementation> possrepImplementationsPerComponent = initPossrepImplementationsPerComponent();

	/**
	 * 
	 */
	private DbmsNameImplementation ( ) {

	}

	/**
	 * @return
	 */
	private MyReadOnlyMap<String, String> initComponentNameMap ( ) {
		// The default component has a "conceptual" component named STRING of type STRING
		HashMap<String, String> w = new HashMap<String, String>();
		w.put(TYPENAMES.STRING, TYPENAMES.STRING);
		return new MyReadOnlyMap<String, String>(w);
	}

	/**
	 * @return
	 */
	private Map<String, PossRepImplementation> initPossrepImplementationsPerComponent ( ) {
		final HashMap<String, PossRepImplementation> w = new HashMap<String, PossRepImplementation>();
		for (PossRepImplementation possrepImplementation : possrepImplementations) {
			for (String componentName : possrepImplementation.getComponentNameMap().keySet()) {
				w.put(componentName, possrepImplementation);
			}
		}
		return new MyReadOnlyMap<String, PossRepImplementation>(w);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.TypeImplementations.PossRepImplementation#getComponentNameMap()
	 */
	public Map<String, String> getComponentNameMap ( ) {
		return componentNameMap;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.PossRepImplementation#extractComponentValue(java.lang.String, be.erwinsmout.SIRA_PRISE.typeimplementations.ScalarValueBuffer)
	 */
	public ValueBuffer getComponentValue (String componentName, ScalarValueBuffer valueBuffer) {
		try {
			return DbmsStringImplementation.getValueBuffer(valueToExternalRepresentation(valueBuffer));
		} catch (InvalidValueException e) {
			throw new RuntimeException(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.TypeImplementation#getDefaultPossrepImplementation()
	 */
	public PossRepImplementation getDefaultPossrepImplementation ( ) {
		return this;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.DBMS.TypeImplementations.TypeImplementation#getDefaultSize()
	 */
	public int getDfltLogicalSize ( ) {
		return 32;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.DBMS.TypeImplementations.TypeImplementation#getMaxSize()
	 */
	public int getMaxLogicalSize ( ) {
		return MAXLOGICALSIZE;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.DBMS.TypeImplementations.TypeImplementation#getMinSize()
	 */
	public int getMinLogicalSize ( ) {
		return MINLOGICALSIZE;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.TypeImplementation#getPhysicalSizeFor(int)
	 */
	public int getPhysicalSizeFor (int logicalSize) {
		return physicalSizeFor(logicalSize);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.SIRAPRISE.typeimplementations.TypeImplementation#getPossrepImplementation(java.lang.String)
	 */
	@Override
	public PossRepImplementation getPossrepImplementation (String componentName) {
		PossRepImplementation possRepImplementation = possrepImplementationsPerComponent.get(componentName);
		if (possRepImplementation == null) {
			throw new IllegalArgumentException(MyMessageFormat.format(Messages.getString("IntervalTypeImplementation.UnknownCompoenent"), new String[] { componentName, this.getClass().getName() })); //$NON-NLS-1$
		}
		return possRepImplementation;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.TypeImplementation#getPossRepImplementation()
	 */
	public Set<PossRepImplementation> getPossrepImplementations ( ) {
		return possrepImplementations;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.PossRepImplementation#getName()
	 */
	public String getPossrepName ( ) {
		return POSSREPNAMES.NAME;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.PossRepImplementation#valueFromComponentValues(java.util.HashMap, int)
	 */
	public ValueBuffer valueFromComponentValues (HashMap<String, ValueBuffer> componentValueMap, int logicalSize) {
		String value = DbmsStringImplementation.getJavaString((ScalarValueBuffer) componentValueMap.get(TYPENAMES.STRING));
		return getValueBuffer(value, logicalSize);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.DBMS.TypeImplementations.TypeImplementation#valueFromExternalRepresentation(java.lang.String)
	 */
	public ValueBuffer valueFromExternalRepresentation (String value) throws InvalidValueException {
		return getValueBuffer(value);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.DBMS.Server.TypeImplementation#fromExternal(java.lang.String, int)
	 */
	public ValueBuffer valueFromExternalRepresentation (String value, int maximumLogicalLengthAllowed) throws InvalidValueException {
		return getValueBuffer(value, maximumLogicalLengthAllowed);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.PossRepImplementation#valueToEscapedExternalRepresentation(be.erwinsmout.SIRA_PRISE.typeimplementations.ValueBuffer)
	 */
	public String valueToEscapedExternalRepresentation (ValueBuffer valueBuffer) {
		String s = getJavaString((ScalarValueBuffer) valueBuffer);
		return BracketParser.meta(s);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.DBMS.Server.TypeImplementation#toExternal(java.nio.ByteBuffer, int, int, int)
	 */
	public String valueToExternalRepresentation (ValueBuffer valueBuffer) {
		return getJavaString((ScalarValueBuffer) valueBuffer);
	}
}