/*
 * Created on 19-apr-04
 */
package be.SIRAPRISE.typeimplementations;

import java.nio.ByteBuffer;
import java.util.*;

import be.SIRAPRISE.client.NAMES.TYPENAMES;
import be.SIRAPRISE.util.BracketParser;
import be.SIRAPRISE.util.InvalidEscapedCharacterException;
import be.SIRAPRISE.util.MissingEscapedCharacterException;
import be.SIRAPRISE.util.MyReadOnlyMap;
import be.SIRAPRISE.util.MyReadOnlySet;
import be.erwinsmout.MyMessageFormat;

/**
 * The implementation for the String type. The physical encoding of STRING values is as follows :
 * <ul>
 * <li>Being a variable-length type, the first four bytes hold an integer that defines the logical length of the value, in this case the number of characters in the STRING value.</li>
 * <li>The next bytes are the characters of the STRING value, at a rate of two bytes per character. Each two-byte integer value tells the offset of the concerned character in the Unicode BMP plane.</li>
 * </ul>
 * 
 * @author Erwin Smout
 */
public final class DbmsStringImplementation implements TypeImplementation, PossRepImplementation {

	/**
	 * 
	 */
	private static DbmsStringImplementation instance = new DbmsStringImplementation();

	/**
	 * The maximum logical size of a SIRA_PRISE string (1G)
	 */
	public static final int MAXLOGICALSIZE = (1 << 30) - 4; // Such that physicalLength can never get b31 set : 4 + 2 * ((1 << 30) - 4) = 4 + (1 << 31) - 8) = 1 << 31 - 4

	/**
	 * The minimum logical size of a SIRA_PRISE string
	 */
	public static final int MINLOGICALSIZE = 0;

	/**
	 * Computes the needed byte capacity for storing SIRA_PRISE strings of the given logical size
	 * 
	 * @param logicalSize
	 *            The length in number of characters of a SIRA_PRISE string
	 * @return The number of bytes needed to encode this string in SIRA_PRISE
	 */
	private static int physicalSizeFor (int logicalSize) {
		return 4 + 2 * logicalSize;
	}

	/**
	 * Gets a SIRA_PRISE scalarvaluebuffer from a java String
	 * 
	 * @param value
	 *            the java String
	 * @param maximumLogicalLengthAllowed
	 *            The maximum length that the context of this invocation allows the SIRA_PRISE string to have
	 * @return A ValueBuffer holding the given string
	 * @throws InvalidValueException
	 *             If the length of <code>value</code> exceeds <code>maximumLogicalLengthAllowed</code>
	 */
	static ScalarValueBuffer getValueBuffer (String value, int maximumLogicalLengthAllowed) throws InvalidValueException {
		int logicalLength = value.length();
		if (logicalLength > maximumLogicalLengthAllowed) {
			throw new InvalidValueException(MyMessageFormat.format(Messages.getString("DbmsStringImplementation.InvalidSize"), new String[] { Integer.toString(logicalLength), Integer.toString(MINLOGICALSIZE), Integer.toString(maximumLogicalLengthAllowed), value })); //$NON-NLS-1$
		}

		ByteBuffer buffer = ByteBuffer.allocate(physicalSizeFor(value.length()));
		buffer.putInt(logicalLength);
		buffer.asCharBuffer().put(value);
		return new ScalarValueBuffer(buffer);
	}

	/**
	 * Gets the instance
	 * 
	 * @return the instance
	 */
	public static DbmsStringImplementation getInstance ( ) {
		return instance;
	}

	/**
	 * Gets a java String from a SIRA_PRISE scalarvaluebuffer
	 * 
	 * @param valueBuffer
	 *            The ScalarValueBuffer containing a SIRA_PRISE String value
	 * @return The String contained in the given ScalarValueBuffer
	 */
	public static String getJavaString (ScalarValueBuffer valueBuffer) {
		ByteBuffer buffer = valueBuffer.getByteBuffer();
		char[] ca = new char[buffer.getInt()];
		buffer.asCharBuffer().get(ca);
		return new String(ca);
	}

	/**
	 * Gets a SIRA_PRISE scalarvaluebuffer from a java String
	 * 
	 * @param value
	 *            the java String
	 * @return A ValueBuffer holding the given string
	 * @throws InvalidValueException
	 */
	public static ScalarValueBuffer getValueBuffer (String value) throws InvalidValueException {
		return getValueBuffer(value, physicalSizeFor(Math.min(MAXLOGICALSIZE, value.length())));
	}

	/**
	 * The possrep component map
	 */
	private MyReadOnlyMap<String, String> componentNameMap = initComponentNameMap();

	/**
	 * The possreps
	 */
	private Set<PossRepImplementation> possrepImplementations = new MyReadOnlySet<PossRepImplementation>(new HashSet<PossRepImplementation>(Arrays.asList(new PossRepImplementation[] { this })));

	/**
	 * 
	 */
	private Map<String, PossRepImplementation> possrepImplementationsPerComponent = new MyReadOnlyMap<String, PossRepImplementation>(new HashMap<String, PossRepImplementation>());

	/**
	 * 
	 */
	private DbmsStringImplementation ( ) {

	}

	/**
	 * @return
	 */
	private MyReadOnlyMap<String, String> initComponentNameMap ( ) {
		// "conceptual" components in this possrep : STRING(STRING)
		// STRING is NOT allowed to have a component of its own type
		// componentNameMap.put(TYPENAMES.STRING, TYPENAMES.STRING); //$NON-NLS-1$//$NON-NLS-2$

		return new MyReadOnlyMap<String, String>(new HashMap<String, String>());
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.TypeImplementations.PossRepImplementation#getComponentNameMap()
	 */
	public Map<String, String> getComponentNameMap ( ) {
		return componentNameMap;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.PossRepImplementation#extractComponentValue(java.lang.String, be.erwinsmout.SIRA_PRISE.typeimplementations.ScalarValueBuffer)
	 */
	public ValueBuffer getComponentValue (String componentName, ScalarValueBuffer valueBuffer) {
		return valueBuffer;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.TypeImplementation#getDefaultPossrepImplementation()
	 */
	public PossRepImplementation getDefaultPossrepImplementation ( ) {
		return this;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.DBMS.TypeImplementations.TypeImplementation#getDefaultSize()
	 */
	public int getDfltLogicalSize ( ) {
		return 124;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.DBMS.TypeImplementations.TypeImplementation#getMaxSize()
	 */
	public int getMaxLogicalSize ( ) {
		return MAXLOGICALSIZE;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.DBMS.TypeImplementations.TypeImplementation#getMinSize()
	 */
	public int getMinLogicalSize ( ) {
		return MINLOGICALSIZE;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.TypeImplementation#getPhysicalSizeFor(int)
	 */
	public int getPhysicalSizeFor (int logicalSize) {
		return physicalSizeFor(logicalSize);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.SIRAPRISE.typeimplementations.TypeImplementation#getPossrepImplementation(java.lang.String)
	 */
	@Override
	public PossRepImplementation getPossrepImplementation (String componentName) {
		PossRepImplementation possRepImplementation = possrepImplementationsPerComponent.get(componentName);
		if (possRepImplementation == null) {
			throw new IllegalArgumentException(MyMessageFormat.format(Messages.getString("IntervalTypeImplementation.UnknownCompoenent"), new String[] { componentName, this.getClass().getName() })); //$NON-NLS-1$
		}
		return possRepImplementation;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.TypeImplementation#getPossRepImplementation()
	 */
	public Set<PossRepImplementation> getPossrepImplementations ( ) {
		return possrepImplementations;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.PossRepImplementation#getName()
	 */
	public String getPossrepName ( ) {
		return TYPENAMES.STRING;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.PossRepImplementation#valueFromComponentValues(java.util.HashMap, int)
	 */
	public ValueBuffer valueFromComponentValues (HashMap<String, ValueBuffer> componentValueMap, int logicalSize) {
		throw new UnsupportedOperationException(); // Type string simply HAS NO possrep components
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.DBMS.TypeImplementations.TypeImplementation#valueFromExternalRepresentation(java.lang.String)
	 */
	public ValueBuffer valueFromExternalRepresentation (String value) throws InvalidValueException {
		return valueFromExternalRepresentation(value, Math.min(MAXLOGICALSIZE, value.length()));
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.DBMS.Server.TypeImplementation#fromExternal(java.lang.String, int)
	 */
	public ValueBuffer valueFromExternalRepresentation (String value, int maximumLogicalLengthAllowed) throws InvalidValueException {
		try {
			return getValueBuffer(BracketParser.unMeta(value), maximumLogicalLengthAllowed);
		} catch (InvalidEscapedCharacterException e) {
			throw new InvalidValueException(MyMessageFormat.format(Messages.getString("DbmsStringImplementation.InvalidEscape"), new String[] { value }), e); //$NON-NLS-1$
		} catch (MissingEscapedCharacterException e) {
			throw new InvalidValueException(MyMessageFormat.format(Messages.getString("DbmsStringImplementation.InvalidEscape"), new String[] { value }), e); //$NON-NLS-1$
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.SIRA_PRISE.typeimplementations.PossRepImplementation#valueToEscapedExternalRepresentation(be.erwinsmout.SIRA_PRISE.typeimplementations.ValueBuffer)
	 */
	public String valueToEscapedExternalRepresentation (ValueBuffer valueBuffer) {
		return BracketParser.meta(getJavaString((ScalarValueBuffer) valueBuffer));
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see be.erwinsmout.DBMS.Server.TypeImplementation#toExternal(java.nio.ByteBuffer, int, int, int)
	 */
	public String valueToExternalRepresentation (ValueBuffer valueBuffer) {
		return getJavaString((ScalarValueBuffer) valueBuffer);
	}
}