/*
 * Created on 29-sep-2005
 */
package be.SIRAPRISE.typeimplementations;

import java.util.HashMap;
import java.util.Map;

/**
 * Defines the methods that the implementation class for a SIRA_PRISE PossRep must provide
 * 
 * @author Erwin Smout
 */
public interface PossRepImplementation {

	/**
	 * "Extracts" a possrep component value out of a value of the type
	 * 
	 * @param componentName
	 *            The possrep component Name
	 * @param valueBuffer
	 *            The scalarValueBuffer holding the value of which the possrep's component's value is to be extracted
	 * @return The value of the component in this possrep's representation of the given value
	 */
	public ValueBuffer getComponentValue (String componentName, ScalarValueBuffer valueBuffer);

	/**
	 * Gets a map of component/type names. The keys in the map are the names of the components of this possrep. The values in the map are the typenames corresponding to (the values for) the named component. Since both the keys and the values in the map are all names, they must both be all uppercase. The map can be empty, if this is a possrep that does not consist of components.
	 * 
	 * @return a map of component/type names.
	 */
	public Map<String, String> getComponentNameMap ( );

	/**
	 * Gets the possrep name
	 * 
	 * @return The possrep name
	 */
	public String getPossrepName ( );

	/**
	 * Convert a value of this possrep's type to internal format. The implementation is responsible for ensuring that the result is within bounds with the possrep's type's minimum or maximum logical size (and thus with the corresponding minimum and maximum physical encoding size). If the type is variable-length, then any length may be chosen for the ByteBuffer returned, as long as the encoding of the value fits within that byteBuffer length. If the encoding of a value takes less bytes than the chosen length, then the remaining bytes must be filled with null bytes (0x00). Failing to obey this rule will subvert certain checksum
	 * computations that are done internally by SIRA_PRISE, and thus cause it to possibly produce wrong results. Note that if a type is variable-length, then the first four bytes in the returned ScalarValueBuffer's ByteBuffer must hold the logical length of the value.
	 * 
	 * @param value
	 *            the value in external format.
	 * @return the value in internal format.
	 * @throws InvalidValueException
	 *             If the textual representation does not represent a value of the possrep's type.
	 */
	public ValueBuffer valueFromExternalRepresentation (String value) throws InvalidValueException;

	/**
	 * Converts a value of this type from external to internal format, given a specified maximum allowable logical size for the value to be returned. All SIRA_PRISE's invoking methods guarantee that this size is within bounds with the possrep's type's minimal and maximal logical sizes. So, invocations from other places notwithstanding, this parameter doesn't need checking. If the type is variable-length, then any length may be chosen for the ByteBuffer returned, as long as the encoding of the value fits within that byteBuffer length. If the encoding of a value takes less bytes than the chosen length, then the remaining bytes must be
	 * filled with null bytes (0x00). Failing to obey this rule will subvert certain checksum computations that are done internally by SIRA_PRISE, and thus cause it to possibly produce wrong results. Note that if a type is variable-length, then the first four bytes in the returned ScalarValueBuffer's ByteBuffer must hold the logical length of the value.
	 * 
	 * @param value
	 *            the textual representation of the value.
	 * @param maximumLogicalLengthAllowed
	 *            the maximum logical length that the returned value can be allowed to have.
	 * @return The parsed value, encoded in internal format.
	 * @throws InvalidValueException
	 *             If the textual representation does not represent a value of the possrep's type.
	 */
	public ValueBuffer valueFromExternalRepresentation (String value, int maximumLogicalLengthAllowed) throws InvalidValueException;

	/**
	 * Converts a value in internal format to the appropriate textual representation of that value for this possrep.
	 * 
	 * @param valueBuffer
	 *            The value in internal format.
	 * @return The string representation of the given value, for this possrep.
	 */
	public String valueToExternalRepresentation (ValueBuffer valueBuffer);

	/**
	 * Converts a value in internal format to the appropriate textual representation of that value for this possrep. In this method, all tokens of the SIRA_PRISE metalanguage must be output in 'escaped' format, i.e. each '(' character in the representation of the value must be represented as the string "\(", each ')' as a "\)" and each '\' as a "\\". Observe that this does NOT apply to the opening and closing parentheses that delimit the value of a component.
	 * 
	 * @param valueBuffer
	 *            The value in internal format.
	 * @return The string representation of the given value, for this possrep.
	 */
	public String valueToEscapedExternalRepresentation (ValueBuffer valueBuffer);

	/**
	 * Returns a ValueBuffer holding the value of the type that this PossRepImplementation is for, and that corresponds to the component values given in the componentValueMap argument.
	 * 
	 * @param componentValueMap
	 *            A Map holding component names and their corresponding values, held in a ValueBuffer.
	 * @param logicalSize
	 *            The maximum logical size that the returned value is allowed to have. This only applies to certain variable-length types. E.g. imagine that type STRING has a possrep with components LENGTH and NONBLANKS, such that the string represented by LENGTH(12)NONBLANKS(A) represents the string "A ". An InvalidValueException should be thrown if these component values are passed to this method, along with a logicalSize of, say, 8.
	 * @return a ValueBuffer holding the value of the type that this PossRepImplementation is for, and that corresponds to the component values given in the componentValueMap argument.
	 */
	public ValueBuffer valueFromComponentValues (HashMap<String, ValueBuffer> componentValueMap, int logicalSize);
}