/*
 * Created on 14-mei-2007
 */
package be.SIRAPRISE.typeimplementations;

import java.nio.ByteBuffer;

import be.SIRAPRISE.util.MyByteBuffer;

/**
 * A ScalarValueBuffer is the class that can be used as a container for any sort of scalar value. It is a wrapper class around a java ByteBuffer.
 * 
 * @author Erwin Smout
 */
public class ScalarValueBuffer extends ValueBuffer {

	/**
	 * The bytebuffer holding the encoded value
	 */
	private ByteBuffer byteBuffer;

	/**
	 * Creates the value buffer and makes it store the scalar value contained in the ByteBuffer given. The given ByteBuffer is rewound before the value held is copied into a local ByteBuffer. This copies all bytes from the given byteBuffer, starting at the first byte, and up to the given byteBuffer's current limit. Changes made later to byteBuffer will therefore not be reflected in this ScalarValueBuffer object.
	 * 
	 * @param byteBuffer
	 *            The ByteBuffer containing the (internal encoding of the) value.
	 */
	public ScalarValueBuffer (ByteBuffer byteBuffer) {
		// the allocation of a new, local, bytebuffer ensures maximum isolation from whatever manipulations the caller later performs
		// on the bytebuffer given, which is still under his control. This prevents inadvertent changes to the value stored here.
		// the asReadOnlyBuffer() in the getByteBuffer() method guarantees that the value will always stay intact, and also that the
		// buffer will always remain in this "initial" state.
		// All of this should make it possible to cache the hashcode
		this.byteBuffer = ByteBuffer.allocate(((ByteBuffer) byteBuffer.rewind()).remaining());
		this.byteBuffer.put(byteBuffer);
		this.byteBuffer.rewind();
	}

	/**
	 * Tells whether or not this ScalarValueBuffer is equal to another object. A ScalarValueBuffer can only be equal to another ScalarValueBuffer, and never to any other type of object. Two ScalarValueBuffer are equal if, and only if,
	 * <p>
	 * <ol>
	 * <li>
	 * <p>
	 * Their ByteBuffers are of equal length, and
	 * </p>
	 * </li>
	 * <li>
	 * <p>
	 * All bytes in the two ByteBuffers' corresponding positions, are equal.
	 * </p>
	 * </li>
	 * </ol>
	 * </p>
	 * 
	 * @param obj
	 *            The object to which this buffer is to be compared
	 * @return <tt>true</tt> if, and only if, this ScalarValueBuffer is equal to the given object
	 */
	public final boolean equals (Object obj) {
		if (obj instanceof ScalarValueBuffer) {
			ByteBuffer otherByteBuffer = ((ScalarValueBuffer) obj).getByteBuffer();
			return getByteBuffer().equals(otherByteBuffer);
		} else {
			return false;
		}
	}

	/**
	 * Gets a read-only bytebuffer holding the encoded value
	 * 
	 * @return A read-only bytebuffer holding the encoded value
	 */
	public final synchronized ByteBuffer getByteBuffer ( ) {
		return ((ByteBuffer) byteBuffer.rewind()).asReadOnlyBuffer();
	}

	public final int hashCode ( ) {
		return getByteBuffer().hashCode();
	}

	public String toString ( ) {
		return MyByteBuffer.toHex(getByteBuffer());
	}
}
