/*
 * Created on 19-apr-04
 */
package be.SIRAPRISE.typeimplementations;

import java.util.Set;

/**
 * Defines the methods that the implementation class for a sira_prise scalar type must provide.
 * 
 * @author Erwin Smout
 */
public interface TypeImplementation {

	/**
	 * Gets the default (logical) size of an attribute of this type. The implementation is responsible for ensuring that the default size is >= the minimum size, and <= the maximum size.
	 * 
	 * @return the default (logical) size of an attribute of this type.
	 */
	public int getDfltLogicalSize ( );

	/**
	 * Gets the maximum logical size of an attribute of this type. If this type has an ordering operator, then the corresponding interval type will have a fixed (i.e. min=max=default) logical size that is equal to twice this maximum size.
	 * 
	 * @return The maximum logical size of an attribute of this type.
	 */
	public int getMaxLogicalSize ( );

	/**
	 * Gets the minimum logical size of an attribute of this type.
	 * 
	 * @return The minimum logical size of an attribute of this type
	 */
	public int getMinLogicalSize ( );

	/**
	 * Gets the set of all PossRepImplementations for this type.
	 * 
	 * @return The set of all PossRepImplementations for this type.
	 */
	public Set<PossRepImplementation> getPossrepImplementations ( );

	/**
	 * Determines the number of bytes needed to encode a value of the given logical size of this type. If the type is variable-length, then the number returned here must be >= 4. These four mandatory bytes are used to hold the actual logical length of the encoded value.
	 * 
	 * @param logicalSize
	 *            The logical size of a declared attribute of this type
	 * @return The total number of bytes needed to encode a value of the given logical size of this type, i.e. including the four length bytes that are used in variable-length types to hold the logical length of the value itself.
	 */
	public int getPhysicalSizeFor (int logicalSize);

	/**
	 * Gets the possrep implementation that is to be considered the default for the type. If the type does not provide any possrep(implementation)s at all, then this method should return null, and this type will be treated as a singleton type.
	 * 
	 * @return the possrep implementation that is to be considered the default for the type, or null if the type is a singleton.
	 */
	public PossRepImplementation getDefaultPossrepImplementation ( );

	/**
	 * Gets the PossrepImplementation object implementing the possrep of which the given component name names a component. This method is introduced in version 1.3. The method is optional, but implementers are strongly encouraged to add this method to any implementing class they already have developed.
	 * 
	 * @param componentName
	 *            The name of a component of some possrep of this type (and for which the corresponding PossrepImplementation object is to be returned)
	 * @return The PossrepImplementation object implementing the possrep of which the given component name names a component
	 */
	public PossRepImplementation getPossrepImplementation (String componentName);
}
