/*
 * Created on 10-dec-03
 */
package be.SIRAPRISE.util;

import java.util.HashMap;
import java.util.LinkedList;
import java.util.Map;

/**
 * Provides methods for working with bracketed expressions.
 * 
 * @author Erwin
 */
public class BracketParser {

	/**
	 * Test whether an expression is a "simple" expression. That is, whether the first and last positions are a matching pair of opening and closing bracket. That is, whether they can be removed from the string without loss of meaning. e.g. "(a and b and c)" has such brackets, but "(a and b) and (c and d)" has not. Note that the input string must be trimmed.
	 * 
	 * @param s
	 *            A String holding an expression to be tested for unneeded outermost brackets (e.g. '((a=b))')
	 * @return true if the string holds an expression with unneeded outermost brackets, false if not.
	 */
	private static boolean hasUnneededOutermostBrackets (String s) {
		if (s.startsWith("(") && s.endsWith(")")) { //$NON-NLS-1$ //$NON-NLS-2$
			try {
				if (findCorrespondingClosingBracket(s) == s.length() - 1) {
					return true;
				} else {
					return false;
				}
			} catch (NoClosingBracketException e) {
				return false;
			}
		} else {
			return false;
		}
	}

	/**
	 * Creates a HashMap for a String in the general format Name1(value1)Name2(value2)... An entry is added to the map for each name in the String, and the corresponding value is the value that appears within parentheses with that name
	 * 
	 * @param s
	 *            A string in the general format name1(value1)name2(value2)...
	 * @return A hashmap holding an entry for each name(value) pair in the input text, where key=name and value=value
	 * @throws MissingEscapedCharacterException
	 * @throws NoOpeningBracketException
	 * @throws InvalidEscapedCharacterException
	 * @throws NoClosingBracketException
	 */
	public static LinkedList<String> createListFromEscaped (String s) throws NoClosingBracketException, InvalidEscapedCharacterException, MissingEscapedCharacterException, NoOpeningBracketException {
		LinkedList<String> list = new LinkedList<String>();

		int p = 0;
		while (p < s.length()) {
			NameValueResult result = getNameAndMandatoryValueFromEscaped(s, p);
			NameValuePair nameValuePair = result.getNameValuePair();
			String valuePart = nameValuePair.getValue();
			list.add(valuePart);
			p = result.getNextParsePos();
		}

		return list;
	}

	/**
	 * Creates a HashMap for a String in the general format Name1(value1)Name2(value2)... An entry is added to the map for each name in the String, and the corresponding value is the value that appears within parentheses with that name
	 * 
	 * @param s
	 *            A string in the general format name1(value1)name2(value2)...
	 * @return A hashmap holding an entry for each name(value) pair in the input text, where key=name and value=value
	 * @throws NoOpeningBracketException
	 * @throws NoClosingBracketException
	 */
	public static LinkedList<String> createListFromNonEscaped (String s) throws NoClosingBracketException, NoOpeningBracketException {
		LinkedList<String> list = new LinkedList<String>();

		int p = 0;
		while (p < s.length()) {
			NameValueResult result = getNameAndMandatoryValueFromNonEscaped(s, p);
			NameValuePair nameValuePair = result.getNameValuePair();
			String valuePart = nameValuePair.getValue();
			list.add(valuePart);
			p = result.getNextParsePos();
		}

		return list;
	}

	/**
	 * Creates a HashMap for a String in the general format Name1(value1)Name2(value2)... An entry is added to the map for each name in the String, and the corresponding value is the value that appears within parentheses with that name
	 * 
	 * @param s
	 *            A string in the general format name1(value1)name2(value2)...
	 * @return A hashmap holding an entry for each name(value) pair in the input text, where key=name and value=value
	 * @throws InvalidEscapedCharacterException
	 * @throws MissingEscapedCharacterException
	 * @throws NoClosingBracketException
	 * @throws DuplicateNameException
	 */
	public static HashMap<String, String> createMapFromEscaped (String s) throws NoClosingBracketException, InvalidEscapedCharacterException, MissingEscapedCharacterException, DuplicateNameException {
		HashMap<String, String> attributeValueMap = new HashMap<String, String>();

		int p = 0;
		while (p < s.length()) {
			NameValueResult result = getNameValueFromEscaped(s, p);
			NameValuePair nameValuePair = result.getNameValuePair();
			String namePart = nameValuePair.getName();
			String valuePart = nameValuePair.getValue();
			if (attributeValueMap.containsKey(namePart)) {
				throw new DuplicateNameException(namePart);
			}
			attributeValueMap.put(namePart, valuePart);
			p = result.getNextParsePos();
		}

		return attributeValueMap;
	}

	/**
	 * Creates a HashMap for a String in the general format Name1(value1)Name2(value2)... An entry is added to the map for each name in the String, and the corresponding value is the value that appears within parentheses with that name
	 * 
	 * @param s
	 *            A string in the general format name1(value1)name2(value2)...
	 * @return A hashmap holding an entry for each name(value) pair in the input text, where key=name and value=value
	 * @throws NoClosingBracketException
	 * @throws DuplicateNameException
	 */
	public static HashMap<String, String> createMapFromNonEscaped (String s) throws NoClosingBracketException, DuplicateNameException {
		HashMap<String, String> attributeValueMap = new HashMap<String, String>();

		int p = 0;
		while (p < s.length()) {
			NameValueResult result = getNameValueFromNonEscaped(s, p);
			NameValuePair nameValuePair = result.getNameValuePair();
			String namePart = nameValuePair.getName();
			String valuePart = nameValuePair.getValue();
			if (attributeValueMap.containsKey(namePart)) {
				throw new DuplicateNameException(namePart);
			}
			attributeValueMap.put(namePart, valuePart);
			p = result.getNextParsePos();
		}

		return attributeValueMap;
	}

	/**
	 * Creates a HashMap for a String in the general format Name1(value1)Name2(value2)... An entry is added to the map for each name in the String, and the corresponding value is the value that appears within parentheses with that name
	 * 
	 * @param s
	 *            A string in the general format name1(value1)name2(value2)...
	 * @return A hashmap holding an entry for each name(value) pair in the input text, where key=name and value=value
	 * @throws DuplicateNameException
	 * @throws NoClosingBracketException
	 * @throws InvalidEscapedCharacterException
	 * @throws MissingEscapedCharacterException
	 * @throws NoOpeningBracketException
	 */
	public static HashMap<String, String> createMapWithMandatoryValuesFromEscaped (String s) throws DuplicateNameException, NoClosingBracketException, InvalidEscapedCharacterException, MissingEscapedCharacterException, NoOpeningBracketException {
		HashMap<String, String> attributeValueMap = new HashMap<String, String>();
		fillMapWithMandatoryValuesFromEscaped(s, attributeValueMap);
		return attributeValueMap;
	}

	/**
	 * Creates a HashMap for a String in the general format Name1(value1)Name2(value2)... An entry is added to the map for each name in the String, and the corresponding value is the value that appears within parentheses with that name
	 * 
	 * @param s
	 *            A string in the general format name1(value1)name2(value2)...
	 * @return A hashmap holding an entry for each name(value) pair in the input text, where key=name and value=value
	 * @throws DuplicateNameException
	 * @throws NoClosingBracketException
	 * @throws NoOpeningBracketException
	 */
	public static HashMap<String, String> createMapWithMandatoryValuesFromNonEscaped (String s) throws DuplicateNameException, NoClosingBracketException, NoOpeningBracketException {
		HashMap<String, String> attributeValueMap = new HashMap<String, String>();
		fillMapWithMandatoryValuesFromNonEscaped(s, attributeValueMap);
		return attributeValueMap;
	}

	/**
	 * @param s
	 * @param attributeValueMap
	 * @throws NoClosingBracketException
	 * @throws InvalidEscapedCharacterException
	 * @throws MissingEscapedCharacterException
	 * @throws NoOpeningBracketException
	 * @throws DuplicateNameException
	 */
	public static void fillMapWithMandatoryValuesFromEscaped (String s, Map<String, String> attributeValueMap) throws NoClosingBracketException, InvalidEscapedCharacterException, MissingEscapedCharacterException, NoOpeningBracketException, DuplicateNameException {
		int p = 0;
		while (p < s.length()) {
			NameValueResult result = getNameAndMandatoryValueFromEscaped(s, p);
			NameValuePair nameValuePair = result.getNameValuePair();
			String namePart = nameValuePair.getName();
			String valuePart = nameValuePair.getValue();
			if (attributeValueMap.containsKey(namePart)) {
				throw new DuplicateNameException(namePart);
			}
			attributeValueMap.put(namePart, valuePart);
			p = result.getNextParsePos();
		}
	}

	/**
	 * @param s
	 * @param attributeValueMap
	 * @throws NoClosingBracketException
	 * @throws NoOpeningBracketException
	 * @throws DuplicateNameException
	 */
	public static void fillMapWithMandatoryValuesFromNonEscaped (String s, Map<String, String> attributeValueMap) throws NoClosingBracketException, NoOpeningBracketException, DuplicateNameException {
		int p = 0;
		while (p < s.length()) {
			NameValueResult result = getNameAndMandatoryValueFromNonEscaped(s, p);
			NameValuePair nameValuePair = result.getNameValuePair();
			String namePart = nameValuePair.getName();
			String valuePart = nameValuePair.getValue();
			if (attributeValueMap.containsKey(namePart)) {
				throw new DuplicateNameException(namePart);
			}
			attributeValueMap.put(namePart, valuePart);
			p = result.getNextParsePos();
		}
	}

	/**
	 * @param s
	 * @param attributeValueMap
	 * @throws DuplicateNameException
	 * @throws NoOpeningBracketException
	 * @throws MissingEscapedCharacterException
	 * @throws InvalidEscapedCharacterException
	 * @throws NoClosingBracketException
	 */
	public static void fillNonEmptyMapWithMandatoryValuesFromEscaped (String s, Map<String, String> attributeValueMap) throws NoClosingBracketException, InvalidEscapedCharacterException, MissingEscapedCharacterException, NoOpeningBracketException, DuplicateNameException {
		fillMapWithMandatoryValuesFromEscaped(s, attributeValueMap);
		if (attributeValueMap.size() < 1) {
			throw new IllegalArgumentException("Empty map not allowed from " + s); //$NON-NLS-1$
		}

		// int obp, cbp;
		// String snm, svl;
		// if (slength > 0) {
		// StringBuffer contents = new StringBuffer(s);
		// while (contents.length() > 0) {
		// obp = contents.toString().indexOf('(');
		// if (obp != -1) {
		// snm = contents.substring(0,obp);
		// contents.delete(0,snm.length());
		// try {
		// cbp = findCorrespondingClosingBracket(contents.toString());
		// } catch (NoClosingBracketException e) {
		// throw new SyntaxException(Messages.getString("NameValueMap.ClosingBracket"),new String[]{contents.toString()}); //$NON-NLS-1$
		// }
		// svl = removeSuperfluousOuterBrackets(contents.substring(0,cbp+1));
		// contents.delete(0,cbp+1);
		// if (snm.length() > 0 && svl.length() > 0) {
		// attributeValueMap.put(snm.toUpperCase(),svl);
		// }
		// } else {
		// throw new SyntaxException(Messages.getString("NameValueMap.OpeningBracket"),new String[]{contents.toString()}); //$NON-NLS-1$
		// }
		// }
		// } else {
		// if (!emptyMapAllowed) {
		//    			
		// }
		// }
	}

	/**
	 * @param s
	 * @param attributeValueMap
	 * @throws DuplicateNameException
	 * @throws NoOpeningBracketException
	 * @throws NoClosingBracketException
	 */
	public static void fillNonEmptyMapWithMandatoryValuesFromNonEscaped (String s, Map<String, String> attributeValueMap) throws NoClosingBracketException, NoOpeningBracketException, DuplicateNameException {
		fillMapWithMandatoryValuesFromNonEscaped(s, attributeValueMap);
		if (attributeValueMap.size() < 1) {
			throw new IllegalArgumentException("Empty map not allowed from " + s); //$NON-NLS-1$
		}
	}

	/**
	 * Find the position of the closing bracket that matches the opening bracket in position 0. It is assumed that the input indeed starts with an opening bracket, otherwise there obviously is no need for calling this method in the first place.
	 * 
	 * @param s
	 *            A String holding a bracketed expression that begins with an opening bracket
	 * @return the position in the String that holds the closing bracket that corresponds to the opening bracket in position 0. -1 if there is no such closing bracket, or the String does not start with an opening bracket.
	 * @throws NoClosingBracketException
	 */
	public static int findCorrespondingClosingBracket (String s) throws NoClosingBracketException {
		int p1 = 0, l = 0;
		// boolean inquotes=false;
		boolean escapeFromMeta = false;
		char c;

		do {
			c = s.charAt(p1);
			if (escapeFromMeta) {
				escapeFromMeta = false;
			} else {
				if (c == '\\') {
					escapeFromMeta = true;
				} else {
					if (c == '(') {
						l++;
					} else {
						if (c == ')') {
							l--;
						}
					}
				}
			}
		} while (l != 0 && ++p1 < s.length());

		// if the first position is not an opening bracket, we come here with p1 == 0
		// if there is no corresponding closing bracket, l will still be >0
		if ((p1 == 0) || (l != 0)) {
			throw new NoClosingBracketException(s);
		} else {
			return p1;
		}
	}

	/**
	 * Returns the 'name' part of an expression of the form "name(content)"
	 * 
	 * @param s
	 *            a string of the general form name(content)
	 * @param startPos
	 *            The first position in s to inspect for the name, in the range [0...length-1]
	 * @return the name of the object that the command refers to
	 * @throws NoClosingBracketException
	 * @throws InvalidEscapedCharacterException
	 * @throws MissingEscapedCharacterException
	 * @throws NoOpeningBracketException
	 */
	public static NameValueResult getNameAndMandatoryValueFromEscaped (String s, int startPos) throws NoClosingBracketException, InvalidEscapedCharacterException, MissingEscapedCharacterException, NoOpeningBracketException {
		int checkPos = Math.max(0, startPos);
		int slength = s.length();

		boolean escapeFromMeta = false;
		boolean inspect = true;
		while (checkPos < slength && inspect) {
			if (escapeFromMeta) {
				switch (s.charAt(checkPos)) {
					case '(':
					case ')':
					case '\\':
						escapeFromMeta = false;
						checkPos++;
						break;
					default:
						throw new InvalidEscapedCharacterException(checkPos, s.charAt(checkPos));
				}
			} else {
				switch (s.charAt(checkPos)) {
					case '(':
						inspect = false;
						break;
					case '\\':
						escapeFromMeta = true;
						checkPos++;
						break;
					default:
						checkPos++;
				}
			}
		}

		if (escapeFromMeta) {
			// Backslash as the last character
			throw new MissingEscapedCharacterException();
		}
		if (inspect) {
			// Then no opening bracket was found, and we can already return a result with only a NAME and an empty VALUE part
			throw new NoOpeningBracketException(s.substring(startPos));
		}
		String namePart = s.substring(startPos, checkPos).trim();

		// checkPos still points to the '('
		inspect = true;
		int level = 1;
		checkPos++;
		int openingBracketPos = checkPos;

		while (level > 0 && checkPos < slength) {
			if (escapeFromMeta) {
				switch (s.charAt(checkPos)) {
					case '(':
					case ')':
					case '\\':
						escapeFromMeta = false;
						break;
					default:
						throw new InvalidEscapedCharacterException(checkPos, s.charAt(checkPos));
				}
			} else {
				switch (s.charAt(checkPos)) {
					case ')':
						level--;
						break;
					case '(':
						level++;
						break;
					case '\\':
						escapeFromMeta = true;
				}
			}
			checkPos++;
		}

		if (escapeFromMeta) {
			// Backslash as the last character
			throw new MissingEscapedCharacterException();
		}

		// if the first position is not an opening bracket, we come here with p1 == 0
		// if there is no corresponding closing bracket, l will still be >0
		if (level > 0) {
			throw new NoClosingBracketException(s.substring(startPos));
		} else {
			checkPos--;
			return new NameValueResult(startPos, checkPos, new NameValuePair(namePart, s.substring(openingBracketPos, checkPos)));
		}
	}

	/**
	 * Returns the 'name' part of an expression of the form "name(content)"
	 * 
	 * @param s
	 *            a string of the general form name(content)
	 * @param startPos
	 *            The first position in s to inspect for the name, in the range [0...length-1]
	 * @return the name of the object that the command refers to
	 * @throws NoClosingBracketException
	 * @throws NoOpeningBracketException
	 */
	public static NameValueResult getNameAndMandatoryValueFromNonEscaped (String s, int startPos) throws NoClosingBracketException, NoOpeningBracketException {
		int checkPos = Math.max(0, startPos);
		int slength = s.length();

		boolean inspect = true;
		while (checkPos < slength && inspect) {
			switch (s.charAt(checkPos)) {
				case '(':
					inspect = false;
					break;
				default:
					checkPos++;
			}
		}

		if (inspect) {
			// Then no opening bracket was found, and we can already return a result with only a NAME and an empty VALUE part
			throw new NoOpeningBracketException(s.substring(startPos));
		}
		String namePart = s.substring(startPos, checkPos).trim();

		// checkPos still points to the '('
		inspect = true;
		int level = 1;
		checkPos++;
		int openingBracketPos = checkPos;

		while (level > 0 && checkPos < slength) {
			switch (s.charAt(checkPos)) {
				case ')':
					level--;
					break;
				case '(':
					level++;
					break;
			}
			checkPos++;
		}

		// if the first position is not an opening bracket, we come here with p1 == 0
		// if there is no corresponding closing bracket, l will still be >0
		if (level > 0) {
			throw new NoClosingBracketException(s.substring(startPos));
		} else {
			checkPos--;
			return new NameValueResult(startPos, checkPos, new NameValuePair(namePart, s.substring(openingBracketPos, checkPos)));
		}
	}

	/**
	 * Returns the 'name' part of an expression of the form "name(content)"
	 * 
	 * @param s
	 *            a string of the general form name(content)
	 * @param startPos
	 *            The first position in s to inspect for the name, in the range [0...length-1]
	 * @return the name of the object that the command refers to
	 * @throws NoClosingBracketException
	 * @throws InvalidEscapedCharacterException
	 * @throws MissingEscapedCharacterException
	 * @throws NoOpeningBracketException
	 */
	public static NameValueResult getNameAndMandatoryValueUntrimmedFromEscaped (String s, int startPos) throws NoClosingBracketException, InvalidEscapedCharacterException, MissingEscapedCharacterException, NoOpeningBracketException {
		int bracketPos = 0;
		int checkPos = Math.max(0, startPos);
		int slength = s.length();

		boolean escapeFromMeta = false;
		boolean inspect = true;
		while (checkPos < slength && inspect) {
			if (escapeFromMeta) {
				switch (s.charAt(checkPos)) {
					case '(':
					case ')':
					case '\\':
						escapeFromMeta = false;
						checkPos++;
						bracketPos++;
						break;
					default:
						throw new InvalidEscapedCharacterException(checkPos, s.charAt(checkPos));
				}
			} else {
				switch (s.charAt(checkPos)) {
					case '(':
						inspect = false;
						break;
					case '\\':
						escapeFromMeta = true;
						checkPos++;
						bracketPos++;
						break;
					default:
						checkPos++;
						bracketPos++;
				}
			}
		}

		if (escapeFromMeta) {
			// Backslash as the last character
			throw new MissingEscapedCharacterException();
		}
		if (inspect) {
			// Then no opening bracket was found, and we can already return a result with only a NAME and an empty VALUE part
			throw new NoOpeningBracketException(s.substring(startPos));
		}
		String namePart = s.substring(startPos, checkPos).trim();

		// checkPos still points to the '('
		inspect = true;
		int level = 1;
		checkPos++;
		bracketPos++;
		int openingBracketPos = bracketPos;

		while (level > 0 && checkPos < slength) {
			if (escapeFromMeta) {
				switch (s.charAt(checkPos)) {
					case '(':
					case ')':
					case '\\':
						escapeFromMeta = false;
						break;
					default:
						throw new InvalidEscapedCharacterException(checkPos, s.charAt(checkPos));
				}
			} else {
				switch (s.charAt(checkPos)) {
					case ')':
						level--;
						break;
					case '(':
						level++;
						break;
					case '\\':
						escapeFromMeta = true;
				}
			}
			checkPos++;
			bracketPos++;
		}

		if (escapeFromMeta) {
			// Backslash as the last character
			throw new MissingEscapedCharacterException();
		}

		// if the first position is not an opening bracket, we come here with p1 == 0
		// if there is no corresponding closing bracket, l will still be >0
		if (level > 0) {
			throw new NoClosingBracketException(s.substring(startPos));
		} else {
			bracketPos--;
			checkPos--;
			return new NameValueResult(startPos, checkPos, new NameValuePair(namePart, s.substring(openingBracketPos, checkPos)));
		}
	}

	/**
	 * Returns the 'name' part of an expression of the form "name(content)"
	 * 
	 * @param s
	 *            a string of the general form name(content)
	 * @return the name of the object that the command refers to
	 * @throws NoOpeningBracketException
	 * @deprecated
	 */
	public static BracketParseResult getNameBeforeBracket (String s) throws NoOpeningBracketException {
		int bracketpos = s.indexOf('(');
		if (bracketpos < 0) {
			throw new NoOpeningBracketException(s);
		} else {
			return new BracketParseResult(s.substring(0, bracketpos).trim().toUpperCase(), bracketpos);
		}
	}

	/**
	 * Returns the 'name' part of an expression of the form "name(content)"
	 * 
	 * @param s
	 *            a string of the general form name(content)
	 * @param startPos
	 *            The first position in s to inspect for the name, in the range [0...length-1]
	 * @return the name of the object that the command refers to
	 * @throws NoClosingBracketException
	 * @throws InvalidEscapedCharacterException
	 * @throws MissingEscapedCharacterException
	 */
	public static NameValueResult getNameValueFromEscaped (String s, int startPos) throws NoClosingBracketException, InvalidEscapedCharacterException, MissingEscapedCharacterException {
		int checkPos = Math.max(0, startPos);
		int slength = s.length();

		boolean escapeFromMeta = false;
		boolean inspect = true;
		while (checkPos < slength && inspect) {
			if (escapeFromMeta) {
				switch (s.charAt(checkPos)) {
					case '(':
					case ')':
					case '\\':
						escapeFromMeta = false;
						checkPos++;
						break;
					default:
						throw new InvalidEscapedCharacterException(checkPos, s.charAt(checkPos));
				}
			} else {
				switch (s.charAt(checkPos)) {
					case '(':
						inspect = false;
						break;
					case '\\':
						escapeFromMeta = true;
						checkPos++;
						break;
					default:
						checkPos++;
				}
			}
		}

		if (escapeFromMeta) {
			// Backslash as the last character
			throw new MissingEscapedCharacterException();
		}
		String namePart;
		if (inspect) {
			// Then no opening bracket was found, and we can already return a result with only a NAME and an empty VALUE part
			namePart = s.substring(startPos).trim();
			return new NameValueResult(startPos, checkPos, new NameValuePair(namePart, "")); //$NON-NLS-1$
		}
		namePart = s.substring(startPos, checkPos).trim();

		// checkPos still points to the '('
		inspect = true;
		int level = 1;
		checkPos++;
		int openingBracketPos = checkPos;

		while (level > 0 && checkPos < slength) {
			if (escapeFromMeta) {
				switch (s.charAt(checkPos)) {
					case '(':
					case ')':
					case '\\':
						escapeFromMeta = false;
						break;
					default:
						throw new InvalidEscapedCharacterException(checkPos, s.charAt(checkPos));
				}
			} else {
				switch (s.charAt(checkPos)) {
					case ')':
						level--;
						break;
					case '(':
						level++;
						break;
					case '\\':
						escapeFromMeta = true;
				}
			}
			checkPos++;
		}

		if (escapeFromMeta) {
			// Backslash as the last character
			throw new MissingEscapedCharacterException();
		}

		// if the first position is not an opening bracket, we come here with p1 == 0
		// if there is no corresponding closing bracket, l will still be >0
		if (level > 0) {
			throw new NoClosingBracketException(s.substring(startPos));
		} else {
			checkPos--;
			return new NameValueResult(startPos, checkPos, new NameValuePair(new String(namePart), s.substring(openingBracketPos, checkPos).trim()));
		}
	}

	/**
	 * Returns the 'name' part of an expression of the form "name(content)"
	 * 
	 * @param s
	 *            a string of the general form name(content)
	 * @param startPos
	 *            The first position in s to inspect for the name, in the range [0...length-1]
	 * @return the name of the object that the command refers to
	 * @throws NoClosingBracketException
	 */
	public static NameValueResult getNameValueFromNonEscaped (String s, int startPos) throws NoClosingBracketException {
		int checkPos = Math.max(0, startPos);
		int slength = s.length();

		boolean inspect = true;
		while (checkPos < slength && inspect) {
			switch (s.charAt(checkPos)) {
				case '(':
					inspect = false;
					break;
				default:
					checkPos++;
			}
		}

		String namePart;
		if (inspect) {
			// Then no opening bracket was found, and we can already return a result with only a NAME and an empty VALUE part
			namePart = s.substring(startPos).trim();
			return new NameValueResult(startPos, checkPos, new NameValuePair(namePart, "")); //$NON-NLS-1$
		}
		namePart = s.substring(startPos, checkPos).trim();

		// checkPos still points to the '('
		inspect = true;
		int level = 1;
		checkPos++;
		int openingBracketPos = checkPos;

		while (level > 0 && checkPos < slength) {
			switch (s.charAt(checkPos)) {
				case ')':
					level--;
					break;
				case '(':
					level++;
					break;
			}
			checkPos++;
		}

		// if the first position is not an opening bracket, we come here with p1 == 0
		// if there is no corresponding closing bracket, l will still be >0
		if (level > 0) {
			throw new NoClosingBracketException(s.substring(startPos));
		} else {
			checkPos--;
			return new NameValueResult(startPos, checkPos, new NameValuePair(namePart, s.substring(openingBracketPos, checkPos).trim()));
		}
	}

	/**
	 * Returns the 'name' part of an expression of the form "name(content)"
	 * 
	 * @param s
	 *            a string of the general form name(content)
	 * @param startPos
	 *            The first position in s to inspect for the name, in the range [0...length-1]
	 * @return the name of the object that the command refers to
	 * @throws NoClosingBracketException
	 * @throws InvalidEscapedCharacterException
	 * @throws MissingEscapedCharacterException
	 */
	public static NameValueResult getNameValueUntrimmedFromEscaped (String s, int startPos) throws NoClosingBracketException, InvalidEscapedCharacterException, MissingEscapedCharacterException {
		int checkPos = Math.max(0, startPos);
		int slength = s.length();

		boolean escapeFromMeta = false;
		boolean inspect = true;
		while (checkPos < slength && inspect) {
			if (escapeFromMeta) {
				switch (s.charAt(checkPos)) {
					case '(':
					case ')':
					case '\\':
						escapeFromMeta = false;
						checkPos++;
						break;
					default:
						throw new InvalidEscapedCharacterException(checkPos, s.charAt(checkPos));
				}
			} else {
				switch (s.charAt(checkPos)) {
					case '(':
						inspect = false;
						break;
					case '\\':
						escapeFromMeta = true;
						checkPos++;
						break;
					default:
						checkPos++;
				}
			}
		}

		if (escapeFromMeta) {
			// Backslash as the last character
			throw new MissingEscapedCharacterException();
		}
		String namePart;
		if (inspect) {
			// Then no opening bracket was found, and we can already return a result with only a NAME and an empty VALUE part
			namePart = s.substring(startPos).trim();
			return new NameValueResult(startPos, checkPos, new NameValuePair(namePart, "")); //$NON-NLS-1$
		}
		namePart = s.substring(startPos, checkPos).trim();

		// checkPos still points to the '('
		inspect = true;
		int level = 1;
		checkPos++;
		int openingBracketPos = checkPos;

		while (level > 0 && checkPos < slength) {
			if (escapeFromMeta) {
				switch (s.charAt(checkPos)) {
					case '(':
					case ')':
					case '\\':
						escapeFromMeta = false;
						break;
					default:
						throw new InvalidEscapedCharacterException(checkPos, s.charAt(checkPos));
				}
			} else {
				switch (s.charAt(checkPos)) {
					case ')':
						level--;
						break;
					case '(':
						level++;
						break;
					case '\\':
						escapeFromMeta = true;
				}
			}
			checkPos++;
		}

		if (escapeFromMeta) {
			// Backslash as the last character
			throw new MissingEscapedCharacterException();
		}

		// if the first position is not an opening bracket, we come here with p1 == 0
		// if there is no corresponding closing bracket, l will still be >0
		if (level > 0) {
			throw new NoClosingBracketException(s.substring(startPos));
		} else {
			checkPos--;
			return new NameValueResult(startPos, checkPos, new NameValuePair(namePart, s.substring(openingBracketPos, checkPos)));
		}
	}

	/**
	 * An arbitrary string, presumed to be of the form 'somename(somecontent)somethingelse'. The 'somecontent' part is returned in a bracketparseresult object, along with the number of parsed characters + 1. The number thus represents the index in the input string of the first unparsed character.
	 * 
	 * @param s
	 *            the input string
	 * @return A BracketParseResult object holding the number of parsed characeters, and the string value that appears between the first opening bracket found in the input string and its corresponding closing bracket. The number of characters parsed includes the closing bracket.
	 * @throws NoOpeningBracketException
	 * @throws NoClosingBracketException
	 * @throws MissingEscapedCharacterException
	 * @throws InvalidEscapedCharacterException
	 * @deprecated
	 */
	public static BracketParseResult getUnMetaContentsWithinBrackets (String s) throws NoOpeningBracketException, NoClosingBracketException, InvalidEscapedCharacterException, MissingEscapedCharacterException {
		int openBracketPos = s.indexOf('(');
		if (openBracketPos == -1) {
			throw new NoOpeningBracketException(s);
		} else {
			String s2 = s.substring(openBracketPos);
			int closingbracketpos = findCorrespondingClosingBracket(s2);
			String s3 = s2.substring(1, closingbracketpos).trim();
			return new BracketParseResult(unMeta(s3), openBracketPos + closingbracketpos + 1);
		}
	}

	/**
	 * An arbitrary string, presumed to be of the form 'somename(somecontent)somethingelse'. The 'somecontent' part is trimmed and then returned in a bracketparseresult object, along with the number of parsed characters + 1. The number thus represents the index in the input string of the first unparsed character.
	 * 
	 * @param s
	 *            the input string
	 * @return A BracketParseResult object holding the number of parsed characeters, and the string value that appears between the first opening bracket found in the input string and its corresponding closing bracket. The number of characters parsed includes the closing bracket.
	 * @throws NoOpeningBracketException
	 * @throws NoClosingBracketException
	 * @throws MissingEscapedCharacterException
	 * @throws InvalidEscapedCharacterException
	 * @deprecated
	 */
	public static BracketParseResult getUnMetaContentsWithinBracketsUntrimmed (String s) throws NoOpeningBracketException, NoClosingBracketException, InvalidEscapedCharacterException, MissingEscapedCharacterException {
		int openBracketPos = s.indexOf('(');
		if (openBracketPos == -1) {
			throw new NoOpeningBracketException(s);
		} else {
			String s2 = s.substring(openBracketPos);
			int closingbracketpos = findCorrespondingClosingBracket(s2);
			String s3 = s2.substring(1, closingbracketpos);
			return new BracketParseResult(unMeta(s3), openBracketPos + closingbracketpos + 1);
		}
	}


	/**
	 * Adapts a given string for representation of the siraprise metalanguage characters ()\
	 * 
	 * @param s
	 *            The string to be adapted for representation of the siraprise metalanguage characters ()\
	 * @return The string s, adapted for representation of the siraprise metalanguage characters ()\
	 */
	public static String meta (String s) {
		int slength = s.length();
		StringBuffer w = new StringBuffer(slength + (slength >>> 3));
		int i = 0;
		while (i < slength) {
			char c = s.charAt(i++);
			switch (c) {
				case '(':
					w.append("\\("); //$NON-NLS-1$
					break;
				case ')':
					w.append("\\)"); //$NON-NLS-1$
					break;
				case '\\':
					w.append("\\\\"); //$NON-NLS-1$
					break;
				default:
					w.append(c);
			}
		}
		return w.toString();
	}

	/**
	 * Removes superfluous outer brackets from arg. e.g. (A) will return A, whereas (A)(B) will return its input.
	 * 
	 * @param arg
	 *            The argument from which superfluous outer brackets are to be removed
	 * @return The argument, with all its superfluous outer brackets removed.
	 */
	public static String removeSuperfluousOuterBrackets (String arg) {
		String wrk = arg;
		while (hasUnneededOutermostBrackets(wrk)) {
			wrk = wrk.substring(1, wrk.length() - 1).trim();
		}
		return wrk;
	}

	/**
	 * Translates strings in sira_prise meta language to their actual value. This means any occurence of '\\' is replaced by '\', '\(' by '(' and '\)' by ')'
	 * 
	 * @param s
	 *            The string to be un-meta-ified
	 * @return The un-meta-ified string
	 * @throws InvalidEscapedCharacterException
	 * @throws MissingEscapedCharacterException
	 */
	public static String unMeta (String s) throws InvalidEscapedCharacterException, MissingEscapedCharacterException {
		int slength = s.length();
		StringBuffer w = new StringBuffer(slength);
		int i = 0;
		boolean escapeFromMeta = false;
		while (i < slength) {
			char c = s.charAt(i++);
			if (escapeFromMeta) {
				switch (c) {
					case '(':
					case ')':
					case '\\':
						w.append(c);
						escapeFromMeta = false;
						break;
					default:
						throw new InvalidEscapedCharacterException(i, s.charAt(i));
				}
			} else {
				switch (c) {
					case '\\':
						escapeFromMeta = true;
						break;
					default:
						w.append(c);
				}
			}
		}

		if (escapeFromMeta) {
			// Backslash as the last character
			throw new MissingEscapedCharacterException();
		}

		return w.toString();
	}

	/**
	 * Translates strings in sira_prise meta language to their actual value. This means any occurence of '\\' is replaced by '\', '\(' by '(' and '\)' by ')'. Invalid escape sequences in the string argument result in an IllegalArgumentException.
	 * 
	 * @param s
	 *            The string to be un-meta-ified
	 * @return The un-meta-ified string
	 */
	public static String unMetaWithoutException (String s) {
		try {
			return unMeta(s);
		} catch (InvalidEscapedCharacterException e) {
			throw new IllegalArgumentException(s + ":\n" + e.getMessage()); //$NON-NLS-1$
		} catch (MissingEscapedCharacterException e) {
			throw new IllegalArgumentException(s);
		}
	}

	/**
	 * Parses a bracketed String and builds a hashmap with the information contained within the (map) value is the (attribute) value, also in string format.
	 * 
	 * @param s
	 *            The textual representation of a bracketed expression
	 * @param emptyMapAllowed
	 *            true if the result is not expected to contain at least one entry
	 * @return An attributevaluemap containing attribute name/value pairs. The key is the name in string format,
	 * @throws DuplicateNameException 
	 * @throws NoOpeningBracketException 
	 * @throws MissingEscapedCharacterException 
	 * @throws InvalidEscapedCharacterException 
	 * @throws NoClosingBracketException 
	 */
	public static Map<String, String> createMapFromEscapedString (String s, boolean emptyMapAllowed) throws NoClosingBracketException, InvalidEscapedCharacterException, MissingEscapedCharacterException, NoOpeningBracketException, DuplicateNameException {
		Map<String, String> attributeValueMap = new IntersectableHashMap<String, String>();
		if (emptyMapAllowed) {
			fillMapWithMandatoryValuesFromEscaped(s, attributeValueMap);
		} else {
			fillNonEmptyMapWithMandatoryValuesFromEscaped(s, attributeValueMap);
		}
		return attributeValueMap;
	}

}